package com.instabug.apm.handler.session;

import static com.instabug.apm.constants.Constants.SessionEndStatusCode;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.SessionCacheModel;
import com.instabug.apm.constants.Constants;
import com.instabug.library.model.common.Session;

import java.util.List;

public interface SessionHandler {

    /**
     * Clones all core session properties and starts a new APM session
     *
     * @param coreSession the core session
     * @see com.instabug.library.SessionManager
     */
    void startSession(@NonNull Session coreSession);

    /**
     * Ends the currently active session with the default end reason
     *
     * @see SessionEndStatusCode
     */
    void endSession();

    /**
     * Ends the currently active session with the passed reason
     *
     * @param reason indicates the session termination reason
     * @see SessionEndStatusCode
     */
    void endSession(@SessionEndStatusCode int reason);

    /**
     * @return the currently active session or null if non was found
     */
    @Nullable
    Session getCurrentSession();

    /**
     * @return core session id related to the current session
     */
    @Nullable
    String getCurrentCoreSessionId();

    /**
     * Get all sessions which are marked as ready to be sent
     * see {@link com.instabug.apm.constants.Constants.SessionSyncStatus}
     *
     * @return list of ready to be sent sessions
     */
    @NonNull
    List<SessionCacheModel> getReadyToBeSentSessions();

    /**
     * Gets the session following to the current session id
     *
     * @param currentSessionID id of the current session
     * @return next session model if current session is not the last session, null otherwise
     */
    @Nullable
    SessionCacheModel getNextSession(String currentSessionID);

    /**
     * Changes sync status for list of sessions
     *
     * @param sessionsIDs list of sessions ids to change sync tatus for
     * @param syncStatus  new sync status. see {@link com.instabug.apm.constants.Constants.SessionSyncStatus}
     */
    void changeSessionSyncStatus(@NonNull List<String> sessionsIDs,
                                 @Constants.SessionSyncStatus int syncStatus);

    /**
     * Delete cached sessions according to its sync status.
     * see {@link com.instabug.apm.constants.Constants.SessionSyncStatus}
     *
     * @param syncStatus session sync status
     */
    void deleteSessionsBySyncStatus(@Constants.SessionSyncStatus int syncStatus);

    /**
     * Updates the end reason of a session
     *
     * @param coreSessionId   id of the core session to be updated
     * @param sessionDuration duration of the session in microseconds
     * @param reason          ending reason
     * @see SessionEndStatusCode
     */
    void updateSessionEndReason(@NonNull String coreSessionId, long sessionDuration, @SessionEndStatusCode int reason);

    /**
     * gets sessions by core session ids
     *
     * @param coreIds list of core sessions ids
     * @return a collection of records represented by {@link SessionCacheModel} if found
     * or empty collection otherwise.
     */
    @NonNull
    List<SessionCacheModel> getSessionsByCoreIds(@NonNull List<String> coreIds);

    /**
     * Deletes the session/s associated with this ID from this database if it is in the given list.
     * Otherwise, the records remain unchanged.
     *
     * @param coreIds list of core session ID
     */
    void deleteSessionsByCoreIds(@NonNull List<String> coreIds);

    /**
     * Queries the previous session of the passed session id
     *
     * @return previous session id if found or null otherwise
     */
    @Nullable
    String getPreviousSessionId(@NonNull String currentSession);

}
