package com.instabug.apm.handler.networklog

import com.instabug.apm.constants.ErrorMessages.SANITIZATION_FAILED_WITH_EXCEPTION
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.model.APMNetworkLog
import com.instabug.apm.sanitization.Sanitizer

class NetworkLogHandlerSanitizationDecorator(
    private val handler: NetworkLogHandler,
    private val sanitizer: Sanitizer<APMNetworkLog>,
    private val logger: Logger
) : NetworkLogHandler {
    override fun insertNetworkLog(networkLog: APMNetworkLog): Long =
        runCatching {
            sanitizer.sanitize(networkLog)
        }.getOrElse { error ->
            logger.logSDKErrorWithStackTrace(SANITIZATION_FAILED_WITH_EXCEPTION, error)
            null
        }?.let(handler::insertNetworkLog) ?: -1L

    override fun updateNetworkLog(networkLog: APMNetworkLog?) =
        handler.updateNetworkLog(networkLog)

    override fun cleanUp() = handler.cleanUp()

    override fun removeAll() = handler.removeAll()

    override fun removeGraphQlData() = handler.removeGraphQlData()

    override fun removeGrpcData() = handler.removeGrpcData()

    override fun getEndedNetworkLogsForSession(sessionID: String?): MutableList<APMNetworkLog>? =
        handler.getEndedNetworkLogsForSession(sessionID)

    override fun forceStop() = handler.forceStop()

    override fun isValidAttribute(traceName: String?, key: String?, value: String?): Boolean =
        handler.isValidAttribute(traceName, key, value)

    override fun addAttribute(
        traceId: Long,
        traceName: String?,
        executedInBackground: Boolean,
        key: String?,
        value: String?
    ) = handler.addAttribute(traceId, traceName, executedInBackground, key, value)

    override fun getTraceAttributes(traceId: Long): MutableMap<String, String> =
        handler.getTraceAttributes(traceId)

    override fun clearW3CExternalTraceIdCache() = handler.clearW3CExternalTraceIdCache()

    override fun clearGeneratedW3CExternalTraceIdCache() =
        handler.clearGeneratedW3CExternalTraceIdCache()

    override fun clearCapturedW3CExternalTraceIdCache() =
        handler.clearCapturedW3CExternalTraceIdCache()
}