package com.instabug.apm.common.concurrent

import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.logger.internal.Logger
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.util.threading.OrderedExecutorService
import java.util.concurrent.Callable
import java.util.concurrent.ExecutionException
import java.util.concurrent.ExecutorService
import java.util.concurrent.Future

private val sharedOrderedExecutorErrorMessage
    get() = "ordered executor(${Thread.currentThread().name})"

class OrderedExecutor(
    private val key: String,
    private val delegate: OrderedExecutorService
) : ExecutorService by delegate {

    override fun execute(command: Runnable) {
        delegate.execute(key, command)
    }

    override fun <V> submit(task: Callable<V>): Future<V> =
        delegate.submit(key, task)
}

internal inline fun <T> ExecutorService.executeAndGet(crossinline runnable: () -> T): T =
    submit<T> { runnable() }.get()

internal inline fun <T> ExecutorService.catchingExecuteAndGet(crossinline runnable: () -> T): T? =
    runCatching { executeAndGet(runnable) }
        .onFailure(::checkAndReportError)
        .getOrNull()

private fun checkAndReportError(throwable: Throwable) {
    val logger: Logger = ServiceLocator.getApmLogger()
    when (throwable) {
        is InterruptedException -> logger.logSDKProtected("$sharedOrderedExecutorErrorMessage has been interrupted")
        is ExecutionException -> logger.reportAndLogError(
            "$sharedOrderedExecutorErrorMessage encountered execution error: ${throwable.message}",
            throwable
        )

        is OutOfMemoryError -> logger.reportAndLogError(
            "$sharedOrderedExecutorErrorMessage encountered an OOM error",
            throwable
        )

        else -> logger.reportAndLogError(
            "$sharedOrderedExecutorErrorMessage encountered an error ${throwable.message}",
            throwable
        )
    }
}

private fun Logger.reportAndLogError(message: String, throwable: Throwable) {
    logSDKErrorWithStackTrace(message, throwable)
    IBGDiagnostics.reportNonFatal(throwable, message)
}