package com.instabug.apm.cache.handler.executiontraces;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler;
import com.instabug.apm.cache.model.ExecutionTraceCacheModel;
import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.library.model.common.Session;

import java.util.List;
import java.util.concurrent.Executor;

public class ExecutionTracesMigrationHandlerImpl implements ExecutionTracesMigrationHandler {

    @NonNull
    private final ExecutionTracesCacheHandler cacheHandler;
    @NonNull
    private final DanglingExecutionTracesCacheHandler danglingCacheHandler;
    @NonNull
    private final APMConfigurationProvider configurationProvider;
    @Nullable
    private final SessionMetaDataCacheHandler sessionMetaDataCacheHandler;

    public ExecutionTracesMigrationHandlerImpl(@NonNull ExecutionTracesCacheHandler cacheHandler,
                                               @NonNull DanglingExecutionTracesCacheHandler danglingCacheHandler,
                                               @NonNull APMConfigurationProvider configurationProvider,
                                               @NonNull Executor executor,
                                               @Nullable SessionMetaDataCacheHandler sessionMetaDataCacheHandler) {
        this.cacheHandler = cacheHandler;
        this.danglingCacheHandler = danglingCacheHandler;
        this.configurationProvider = configurationProvider;
        this.sessionMetaDataCacheHandler = sessionMetaDataCacheHandler;
    }

    @Override
    public void migrate(@NonNull final Session runningSession, @NonNull final Session lastSession) {
        long limit = configurationProvider.getExecutionTraceLimitPerRequest();
        List<ExecutionTraceCacheModel> batch;
        do {
            batch = getBatch(limit);
            if (batch != null) {
                for (ExecutionTraceCacheModel trace : batch) {
                    if (shouldLinkToLastSession(trace)) migrate(trace, lastSession);
                    else migrate(trace, runningSession);
                }
                deleteBatch(batch);
            }
        } while (batch != null && batch.size() > 0);
    }

    @VisibleForTesting
    @Nullable
    List<ExecutionTraceCacheModel> getBatch(long limit) {
        return danglingCacheHandler.getEndedTraces(limit);
    }

    @VisibleForTesting
    void migrate(@NonNull ExecutionTraceCacheModel trace, @NonNull Session session) {
        if (sessionMetaDataCacheHandler != null) {
            cacheHandler.insertTrace(session.getId(), trace);
            sessionMetaDataCacheHandler.addToTracesTotalCount(session.getId(), 1);
        }
    }

    @VisibleForTesting
    void deleteBatch(@NonNull List<ExecutionTraceCacheModel> batch) {
        danglingCacheHandler.removeEndedTraces(batch.size());
    }

    @VisibleForTesting
    boolean shouldLinkToLastSession(@NonNull ExecutionTraceCacheModel trace) {
        //started foreground && ended background => prev session
        //started background && ended background => next session
        //started background && ended foreground => next session
        return !trace.startedInBackground() && trace.endedInBackground()
                || !trace.startedInBackground() && !trace.endedInBackground();
    }
}
