package com.instabug.apm.cache.handler.executiontraces;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.ExecutionTracesEntry.COLUMN_DURATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.ExecutionTracesEntry.COLUMN_ENDED_ON_BACKGROUND;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.ExecutionTracesEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.ExecutionTracesEntry.COLUMN_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.ExecutionTracesEntry.COLUMN_SESSION_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.ExecutionTracesEntry.COLUMN_STARTED_ON_BACKGROUND;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.ExecutionTracesEntry.COLUMN_START_TIME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.ExecutionTracesEntry.DURATION_DEFAULT_VALUE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.ExecutionTracesEntry.TABLE_NAME;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.collection.ArrayMap;

import com.instabug.apm.cache.model.ExecutionTraceCacheModel;
import com.instabug.apm.constants.ErrorMessages;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.internal.utils.stability.execution.ReturnableExecutable;
import com.instabug.library.internal.utils.stability.handler.exception.ExceptionHandler;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class ExecutionTracesCacheHandlerImpl implements ExecutionTracesCacheHandler {

    private ExceptionHandler exceptionHandler = ServiceLocator.getExceptionHandler();
    @Nullable
    private DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
    private Logger apmLogger = ServiceLocator.getApmLogger();

    @Override
    public boolean insertTrace(@NonNull String sessionID, @NonNull ExecutionTraceCacheModel trace) {
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            ContentValues values = new ContentValues();
            values.put(COLUMN_ID, trace.getId());
            values.put(COLUMN_SESSION_ID, sessionID);
            if (trace.getName() != null) {
                values.put(COLUMN_NAME, trace.getName());
            }
            values.put(COLUMN_START_TIME, trace.getStartTime());
            values.put(COLUMN_DURATION, trace.getDuration());
            values.put(COLUMN_STARTED_ON_BACKGROUND, trace.startedInBackground() ? 1 : 0);
            values.put(COLUMN_ENDED_ON_BACKGROUND, trace.endedInBackground() ? 1 : 0);
            long rowId = sqLiteDatabaseWrapper.insert(InstabugDbContract.ExecutionTracesEntry.TABLE_NAME, null, values);
            if (rowId != -1 && trace.getName() != null && trace.getAttrs() != null) {
                setAttributes(trace.getId(), trace.getName(), trace.getAttrs());
            }
            sqLiteDatabaseWrapper.close();
            return rowId != -1;
        }
        return false;
    }

    @Nullable
    private List<ExecutionTraceCacheModel> getTracesByQueryString(String tracesQuery) {
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            Cursor cursor = null;
            try {
                cursor = sqLiteDatabaseWrapper.rawQuery(tracesQuery, null);
                ArrayList<ExecutionTraceCacheModel> cacheModels = getTracesFromCursor(sqLiteDatabaseWrapper, cursor);
                sqLiteDatabaseWrapper.close();
                return cacheModels;
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return null;
    }

    @SuppressLint("Range")
    private ArrayList<ExecutionTraceCacheModel> getTracesFromCursor(SQLiteDatabaseWrapper sqLiteDatabaseWrapper,
                                                                    @Nullable Cursor cursor) {
        ArrayList<ExecutionTraceCacheModel> executionTraceCacheModels = new ArrayList<>();
        if (cursor != null) {
            while (cursor.moveToNext()) {
                ExecutionTraceCacheModel executionTraceCacheModel = new ExecutionTraceCacheModel();
                executionTraceCacheModel.setId(cursor.getLong(cursor.getColumnIndex(COLUMN_ID)));
                executionTraceCacheModel.setName(cursor.getString(cursor.getColumnIndex(COLUMN_NAME)));
                executionTraceCacheModel.setStartTime(cursor.getLong(cursor.getColumnIndex(COLUMN_START_TIME)));
                executionTraceCacheModel.setDuration(cursor.getLong(cursor.getColumnIndex(COLUMN_DURATION)));
                executionTraceCacheModel.setStartedInBackground(cursor.getInt(cursor.getColumnIndex(COLUMN_STARTED_ON_BACKGROUND)) == 1);
                executionTraceCacheModel.setEndedInBackground(cursor.getInt(cursor.getColumnIndex(COLUMN_ENDED_ON_BACKGROUND)) == 1);
                String attributesQuery = "select* from " + InstabugDbContract.ExecutionTracesAttributesEntry.TABLE_NAME
                        + " where " + InstabugDbContract.ExecutionTracesAttributesEntry.COLUMN_EXECUTION_TRACE_ID + " = " + executionTraceCacheModel.getId();
                Cursor attributesCursor = null;
                try {
                    attributesCursor = sqLiteDatabaseWrapper.rawQuery(attributesQuery, null);
                    if (attributesCursor != null) {
                        Map<String, String> attrsMap = new ArrayMap<>();
                        while (attributesCursor.moveToNext()) {
                            attrsMap.put(attributesCursor.getString(attributesCursor.getColumnIndex(InstabugDbContract.ExecutionTracesAttributesEntry.COLUMN_KEY)),
                                    attributesCursor.getString(attributesCursor.getColumnIndex(InstabugDbContract.ExecutionTracesAttributesEntry.COLUMN_VALUE)));
                        }
                        executionTraceCacheModel.setAttrs(attrsMap);
                    }
                } catch (Exception ex) {
                    apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                    IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
                } finally {
                    if (attributesCursor != null) {
                        attributesCursor.close();
                    }
                }
                executionTraceCacheModels.add(executionTraceCacheModel);
            }
        }
        return executionTraceCacheModels;
    }

    @Override
    public void setAttribute(long traceId, @NonNull String traceName, @NonNull String key, @Nullable String value) {
        addAttribute(traceId, key, value);
    }

    @Override
    public void setAttributes(long traceId, @NonNull String traceName, @NonNull Map<String, String> attrs) {
        for (Map.Entry<String, String> attr : attrs.entrySet()) {
            setAttribute(traceId, traceName, attr.getKey(), attr.getValue());
        }
    }

    @SuppressLint("Range")
    @Nullable
    public Map<String, String> getTraceAttributes(long traceId) {
        if (databaseManager != null) {
            Map<String, String> attributes = new ArrayMap<>();
            String whereClause = InstabugDbContract.ExecutionTracesAttributesEntry.COLUMN_EXECUTION_TRACE_ID + " = ?";
            String[] whereValues = {traceId + ""};
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            Cursor cursor = null;
            try {
                cursor = sqLiteDatabaseWrapper.query(InstabugDbContract.ExecutionTracesAttributesEntry.TABLE_NAME, null
                        , whereClause, whereValues, null, null, null);
                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        attributes.put(cursor.getString(cursor.getColumnIndex(InstabugDbContract.ExecutionTracesAttributesEntry.COLUMN_KEY)),
                                cursor.getString(cursor.getColumnIndex(InstabugDbContract.ExecutionTracesAttributesEntry.COLUMN_VALUE)));
                    }
                }
                sqLiteDatabaseWrapper.close();
                return attributes;
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return null;
    }

    @Override
    public void removeAll() {
        if (databaseManager != null) {
            String removeTracesQuery = "delete from " + TABLE_NAME;
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(removeTracesQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public void removeUnEndedTraces() {
        if (databaseManager != null) {
            String selectTracesQuery = "select * from " + TABLE_NAME + " where " + COLUMN_DURATION + " = " + DURATION_DEFAULT_VALUE;
            List<ExecutionTraceCacheModel> unEndedTraces = getTracesByQueryString(selectTracesQuery);
            if (unEndedTraces != null) {
                for (ExecutionTraceCacheModel trace : unEndedTraces) {
                    if (trace.getName() != null) {
                        apmLogger.logSDKError(ErrorMessages.UN_ENDED_TRACE_NOT_SAVED.replace("$s", trace.getName()));
                    }
                }
            }
            String deleteTracesQuery = "delete from " + TABLE_NAME + " where " + COLUMN_DURATION + " = " + DURATION_DEFAULT_VALUE;
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(deleteTracesQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Nullable
    @Override
    public List<ExecutionTraceCacheModel> getExecutionTracesForSession(String sessionID) {
        if (databaseManager != null) {
            SQLiteDatabaseWrapper databaseWrapper = databaseManager.openDatabase();
            String selectionClause = COLUMN_SESSION_ID + " = ? AND "
                    + COLUMN_DURATION + " > ?";
            String[] selectionArgs = {sessionID, "0"};
            Cursor cursor = null;
            try {
                cursor = databaseWrapper.query(TABLE_NAME,
                        null, selectionClause, selectionArgs,
                        null, null, null);
                List<ExecutionTraceCacheModel> cacheList = getTracesFromCursor(databaseWrapper, cursor);
                databaseWrapper.close();
                return cacheList;
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return null;
    }

    @Override
    public void trimTracesToLimit(long limit) {
        if (databaseManager != null) {
            String selectByLimitDescendingQuery = "SELECT " + COLUMN_ID
                    + " FROM " + TABLE_NAME
                    + " ORDER BY " + COLUMN_START_TIME + " DESC"
                    + " LIMIT ? OFFSET ?";

            String whereClause = COLUMN_ID + " IN (" + selectByLimitDescendingQuery + ")";
            String[] whereArgs = {"-1", String.valueOf(limit)};

            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            try {
                sqLiteDatabaseWrapper.delete(TABLE_NAME, whereClause, whereArgs);
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (sqLiteDatabaseWrapper != null) {
                    sqLiteDatabaseWrapper.close();
                }
            }
        }
    }

    @Override
    public int trimTracesToLimit(String sessionID, long limit) {
        if (databaseManager != null) {
            String selectByLimitDescendingQuery = "SELECT " + COLUMN_ID
                    + " FROM " + TABLE_NAME
                    + " where " + COLUMN_SESSION_ID + " = ?"
                    + " ORDER BY " + COLUMN_START_TIME + " DESC"
                    + " LIMIT ?";

            String whereClause = COLUMN_SESSION_ID + " = ? AND "
                    + COLUMN_ID + " NOT IN (" + selectByLimitDescendingQuery + ")";
            String[] whereArgs = {sessionID, sessionID, String.valueOf(limit)};
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            int deletedTracesCount = sqLiteDatabaseWrapper.delete(TABLE_NAME, whereClause, whereArgs);
            sqLiteDatabaseWrapper.close();

            return deletedTracesCount;
        }
        return -1;
    }

    @Override
    public boolean exists(final long traceId) {
        return exceptionHandler.executeAndGet(new ReturnableExecutable<Boolean>() {
            @Override
            public Boolean execute() {
                if (databaseManager != null) {
                    String whereClause = COLUMN_ID + " = ? ";
                    String[] whereValues = {String.valueOf(traceId)};
                    SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                    Cursor cursor = null;
                    try {
                        cursor = database.query(TABLE_NAME, null, whereClause, whereValues, null, null, null);
                        if (cursor != null && cursor.getCount() > 0) {
                            return true;
                        }
                    } catch (Exception ex) {
                        apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                        IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
                    } finally {
                        if (cursor != null) {
                            cursor.close();
                        }
                        database.close();
                    }
                    return false;
                }
                return false;
            }
        }, false);
    }

    private void addAttribute(long traceId, @NonNull String key, @Nullable String value) {
        if (databaseManager != null) {
            ContentValues contentValues = new ContentValues();
            contentValues.put(InstabugDbContract.ExecutionTracesAttributesEntry.COLUMN_EXECUTION_TRACE_ID, traceId);
            contentValues.put(InstabugDbContract.ExecutionTracesAttributesEntry.COLUMN_KEY, key);
            if (value != null) {
                contentValues.put(InstabugDbContract.ExecutionTracesAttributesEntry.COLUMN_VALUE, value);
            }
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.insert(InstabugDbContract.ExecutionTracesAttributesEntry.TABLE_NAME, null, contentValues);
            sqLiteDatabaseWrapper.close();
        }
    }

    private void removeAttribute(long traceId, String key) {
        if (databaseManager != null) {
            String query = "delete from " + InstabugDbContract.ExecutionTracesAttributesEntry.TABLE_NAME + " where "
                    + InstabugDbContract.ExecutionTracesAttributesEntry.COLUMN_EXECUTION_TRACE_ID + " = " + traceId
                    + " and " + InstabugDbContract.ExecutionTracesAttributesEntry.COLUMN_KEY + " = \"" + key + "\"";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(query);
            sqLiteDatabaseWrapper.close();
        }
    }

}
