package com.instabug.apm.appflow.handler

import com.instabug.apm.appflow.configuration.AppFlowConfigurationProvider
import com.instabug.apm.appflow.log.logDroppedFlows
import com.instabug.apm.appflow.model.AppFlowCacheModel
import com.instabug.apm.appflow.model.AppFlowEndReason
import com.instabug.apm.appflow.model.AppFlowInsertionCacheModel
import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler
import com.instabug.apm.handler.session.SessionHandler
import com.instabug.apm.logger.internal.Logger
import com.instabug.library.SpanIDProvider

class AppFlowHandlerImpl(
    private val cacheHandler: AppFlowCacheHandler,
    private val sessionMetaDataCacheHandler: SessionMetaDataCacheHandler,
    private val sessionHandler: SessionHandler,
    private val configurations: AppFlowConfigurationProvider,
    private val logger: Logger,
    appLaunchIdProvider: SpanIDProvider
) : AppFlowHandler {

    private val appLaunchId: String by lazy {
        appLaunchIdProvider.spanId
    }

    private inline val appFlowsEnabled: Boolean get() = configurations.enabled

    private inline val coreSessionId: String?
        get() = sessionHandler.currentCoreSessionId

    override fun start(
        name: String,
        timeStampMicro: Long,
        timeMicro: Long,
        isBackground: Boolean
    ): Boolean? =
        takeIf { appFlowsEnabled }
            ?.let { startAppFlow(name, timeStampMicro, timeMicro, isBackground) }

    private fun startAppFlow(
        name: String,
        timeStampMicro: Long,
        timeMicro: Long,
        isBackground: Boolean
    ): Boolean = cacheHandler.insert(
        AppFlowInsertionCacheModel(
            name = name,
            timeStampMicro = timeStampMicro,
            timeMicro = timeMicro,
            appLaunchId = appLaunchId,
            apmSessionId = sessionHandler.currentSession?.id,
            coreSessionId = coreSessionId,
            isBackground = isBackground
        )
    ) != -1L

    override fun end(name: String, timeMicro: Long): Boolean? =
        takeIf { appFlowsEnabled }
            ?.let { cacheHandler.end(name, timeMicro, appLaunchId) > 0 }

    override fun endActiveFlowsWithReason(
        fromTimeMillis: Long,
        toTimeMillis: Long,
        @AppFlowEndReason endReason: Int
    ): Boolean? = takeIf { appFlowsEnabled }?.let {
        val durationMillis = toTimeMillis - fromTimeMillis
        durationMillis > configurations.idlingTimeThresholdMs
    }?.also { if (it) cacheHandler.endActiveFlowsWithEndReason(appLaunchId, endReason) }

    override fun endWithReason(name: String, @AppFlowEndReason reason: Int): Boolean? =
        takeIf { appFlowsEnabled }
            ?.let { cacheHandler.endWithReason(name, reason, appLaunchId) > 0 }

    override fun addAttribute(name: String, key: String, value: String): Boolean? =
        takeIf { appFlowsEnabled }
            ?.let { cacheHandler.addAttribute(name, key, value, appLaunchId) != -1L }

    override fun updateAttributeValue(name: String, key: String, newValue: String): Boolean? =
        takeIf { appFlowsEnabled }
            ?.let { cacheHandler.updateAttributeValue(name, key, newValue, appLaunchId) > 0 }

    override fun removeAttribute(name: String, key: String): Boolean? =
        takeIf { appFlowsEnabled }
            ?.let { cacheHandler.removeAttribute(name, key, appLaunchId) > 0 }

    override fun getAttributeCount(name: String): Int? =
        takeIf { appFlowsEnabled }
            ?.let { cacheHandler.getAttributeCount(name, appLaunchId) }


    override fun clear() = cacheHandler.clear()

    override fun updateCurrentSession(newSession: String): Int? =
        takeIf { appFlowsEnabled }
            ?.let {
                updateCoreSessionIdForActiveFlowsIfPossible()
                cacheHandler.migrateActiveFlows(newSession, appLaunchId)
            }

    private fun updateCoreSessionIdForActiveFlowsIfPossible() {
        coreSessionId?.let { cacheHandler.setActiveFlowsCoreSessionId(it, appLaunchId) }
    }

    override fun get(sessionId: String): List<AppFlowCacheModel> =
        cacheHandler.retrieve(sessionId)

    override fun trimAndUpdateCounts(newSessionId: String, previousSessionId: String?) =
        takeIf { appFlowsEnabled }
            ?.let {
                getPreviousSessionId(previousSessionId, newSessionId)?.let {
                    trimAndUpdateFlowCounts(newSessionId, it)
                    true
                } ?: false
            }

    override fun dropDanglingFlows() {
        takeIf { appFlowsEnabled }
            ?.let { cacheHandler.dropDanglingFLowsExcludingAppLaunch(appLaunchId) }
    }

    private fun trimAndUpdateFlowCounts(newSessionId: String, previousSessionId: String) {
        updateSessionTotalAppFlowCount(previousSessionId)
        trimSessionByRequestLimitAndUpdateDroppedCount(previousSessionId)
        trimByStoreLimitExcludingSession(newSessionId)
    }

    private fun updateSessionTotalAppFlowCount(sessionId: String) =
        cacheHandler.getCount(sessionId)
            .let { sessionMetaDataCacheHandler.setAppFlowTotalCount(sessionId, it) }

    private fun trimSessionByRequestLimitAndUpdateDroppedCount(sessionId: String) =
        cacheHandler.trimByRequestLimitForSession(sessionId, configurations.requestLimit)
            .takeIf { it > 0 }
            ?.also { sessionMetaDataCacheHandler.setAppFlowDroppedCount(sessionId, it) }
            ?.also { logger.logDroppedFlows(it) }

    private fun trimByStoreLimitExcludingSession(sessionId: String) =
        cacheHandler.trimByStoreLimitExcludingSession(sessionId, configurations.storeLimit)

    private fun getPreviousSessionId(previousSessionId: String?, newSessionId: String): String? =
        (previousSessionId ?: sessionHandler.getPreviousSessionId(newSessionId))

    override fun setActiveFlowsEndReason(
        reason: Int,
        isBackgroundFlagOverride: Boolean?
    ): Boolean? = reason
        .takeIf { appFlowsEnabled }
        ?.let { cacheHandler.setActiveFlowsEndReason(reason, isBackgroundFlagOverride, appLaunchId) > 0 }

    override fun setActiveFlowsBackgroundFlag(isBackground: Boolean): Boolean? =
        cacheHandler.takeIf { appFlowsEnabled }
            ?.setActiveFlowsBackgroundState(isBackground, appLaunchId)
            ?.let { it > 0 }

    override fun filterUnReadyCoreSessionIds(sessionIds: List<String>): List<String>? =
        cacheHandler.takeIf { appFlowsEnabled }
            ?.filterUnReadyCoreSessionIds(sessionIds, appLaunchId)
}