package com.instabug.apm.appflow.handler

import com.instabug.apm.appflow.model.AppFlowCacheModel
import com.instabug.apm.appflow.model.AppFlowEndReason

interface AppFlowHandler {

    /**
     * Start a new AppFlow with the provided parameters
     * it will not end active app flows with the same name, it has to be done manually through the
     * appropriate {AppFlowHandler} APIs
     * it's the responsibility of the caller to end the active app flows with the same name before
     * calling this API
     *
     * @param timeStampMicro: timestamp in micro seconds
     * @param timeMicro: time in micro marking the start of the app flow which is going to be used in deriving the flow duration
     * @param isBackground: true if flow started in background, false otherwise
     *
     * @return a flag indicating whether a new flow was created or not, null if feature is disabled
     */
    fun start(
        name: String,
        timeStampMicro: Long,
        timeMicro: Long,
        isBackground: Boolean = false
    ): Boolean?

    /**
     * End all active app flows with the provided name within the current app launch
     * Should always be one app flow affected
     *
     * @param timeMicro: time in micro marking the end of the app flow which is going to be used in deriving the flow duration
     *
     * @return a flag indicating whether whether flow was ended or not, null if feature is disabled
     */
    fun end(name: String, timeMicro: Long): Boolean?

    /**
     * End all abandoned active app flows for current app launch
     *
     * @param endReason: active app flows with this end reason will be marked as ended
     *
     * @return a flag indicating whether whether flow was ended or not, null if feature is disabled
     */
    fun endActiveFlowsWithReason(
        fromTimeMillis: Long,
        toTimeMillis: Long,
        @AppFlowEndReason endReason: Int
    ): Boolean?

    /**
     * End all active app flows with the provided name within the current app launch
     * Should always be one app flow affected
     *
     * @return a flag indicating whether whether flow was ended or not, null if feature is disabled
     */
    fun endWithReason(name: String, @AppFlowEndReason reason: Int): Boolean?

    /**
     * Add attribute to app flow, will only add it to the most recent active app flow with the
     * matching name in the current app launch
     *
     * @return a flag indicating whether whether attribute was added or not, null if feature is disabled
     */
    fun addAttribute(name: String, key: String, value: String): Boolean?

    /**
     * Update attribute value for a given key, will only affect the most recent active app flow
     * with the matching name and current app launch
     */
    fun updateAttributeValue(name: String, key: String, newValue: String): Boolean?

    /**
     * remove attribute from app flow, will only remove it from the most recent active app flow
     * with the matching name in the current app launch
     *
     * @return a flag indicating whether whether attribute was removed or not, null if feature is disabled
     */
    fun removeAttribute(name: String, key: String): Boolean?

    /**
     * Get the total attribute count of the most recent active app flow
     * with the matching name and current app launch id
     */
    fun getAttributeCount(name: String): Int?

    /**
     * Remove all app flow records and attributes from DB
     */
    fun clear()

    /**
     * Migrate all active app flows in the current app launch to apm session with the passed ID
     *
     * @return the number of affected app flows, null if feature was disabled
     */
    fun updateCurrentSession(newSession: String): Int?

    /**
     * Retrieve all app flows in apm session with the passed ID
     */
    fun get(sessionId: String): List<AppFlowCacheModel>

    /**
     * Update total AppFlow count in session metadata for the previousSession
     * Trim previousSession AppFlows by request limit and update it's dropped count in session metadata
     * Trim All app flows using store limit, excluding the newSession
     *
     * @return true if trimming was performed successfully, false otherwise and null if feature is disabled
     */
    fun trimAndUpdateCounts(newSessionId: String, previousSessionId: String?): Boolean?

    /**
     * Drop dangling app flows that are not assigned APM session from database
     * excluding the ones related to the current app launch
     */
    fun dropDanglingFlows()

    /**
     * Sets end reason and background state for all active app flows for current app launch
     *
     * @param isBackgroundFlagOverride the new background state applied to the active flows, null will not change the state
     * @return a flag indicating whether whether endReason is set or not, null if feature is disabled
     */
    fun setActiveFlowsEndReason(
        @AppFlowEndReason reason: Int,
        isBackgroundFlagOverride: Boolean? = null
    ): Boolean?

    /**
     * Update isBackground flag for active app flows in the current app launch
     *
     * @return a flag indicating whether there were active flows that got updated or not, null if features is disabled
     */
    fun setActiveFlowsBackgroundFlag(isBackground: Boolean): Boolean?

    /**
     * Filters unready core session ids due to unfinished APM session flow migration
     *
     * @return String list containing core session ids for APM sessions that still have active flows that are not migrated
     * the list will be empty in case no unready session ids or error or returns null if the feature is disabled
     *
     * @param sessionIds list of session ids to search in
     */
    fun filterUnReadyCoreSessionIds(sessionIds: List<String>): List<String>?
}