package com.instabug.apm.appStateDispacher

import com.instabug.library.core.eventbus.AppStateEvent
import com.instabug.library.core.eventbus.AppStateEventBus
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable
import com.instabug.library.core.eventbus.instabugeventbus.operators.distinctBy
import java.util.Collections
import java.util.WeakHashMap
import java.util.concurrent.Executor

class CombinedAppStateEvents(
    val currentAppStateEvent: AppStateEvent?,
    val previousAppStateEvent: AppStateEvent?
)

interface AppStateEventListener {
    operator fun invoke(events: CombinedAppStateEvents)
}

class AppStateEventDispatcher(private val executor: Executor) {

    private val appStateEventBus =
        AppStateEventBus.distinctBy { event -> event?.let { it::class } }
    private var appStateEventDisposable: IBGDisposable? = null
    private val listeners: MutableSet<AppStateEventListener> =
        Collections.synchronizedSet(Collections.newSetFromMap(WeakHashMap()))

    var currentAppStateEvent: AppStateEvent? = null
        get() = synchronized(this) { field }
        private set(value) = synchronized(this) { field = value }

    fun start() {
        if (appStateEventDisposable == null)
            appStateEventDisposable = appStateEventBus.subscribe(::dispatchAndUpdateCurrentState)
    }

    private fun dispatchAndUpdateCurrentState(
        newAppStateEvent: AppStateEvent?
    ) = executor.execute {
        synchronized(listeners) {
            listeners.forEach { it(CombinedAppStateEvents(newAppStateEvent, currentAppStateEvent)) }
        }
        currentAppStateEvent = newAppStateEvent
    }

    fun addListener(listener: AppStateEventListener) {
        listeners += listener
    }

    fun removeListener(listener: AppStateEventListener) {
        listeners -= listener
    }

    operator fun plusAssign(listener: AppStateEventListener) = addListener(listener)

    operator fun minusAssign(listener: AppStateEventListener) = removeListener(listener)
}