package com.instabug.apm.fragment

import androidx.annotation.VisibleForTesting
import androidx.fragment.app.Fragment
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.library.transform.TransformationClass
import com.instabug.library.transform.TransformationMethod
import java.util.*
import java.util.concurrent.Executor

@TransformationClass
class FragmentEventDispatcher {
    companion object {

        private const val executorId = "FragmentEventDispatcher"

        @JvmStatic
        @VisibleForTesting
        val listeners =
            Collections.synchronizedSet(HashSet<FragmentLifecycleEventListener>())

        @JvmStatic
        private inline val executor: Executor
            get() = ServiceLocator.getSingleThreadExecutor(executorId)

        @JvmStatic
        fun addListener(listener: FragmentLifecycleEventListener) {
            listeners.add(listener)
        }

        @JvmStatic
        fun removeListener(listener: FragmentLifecycleEventListener) {
            listeners.remove(listener)
        }

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreAttach(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPreAttach)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostAttach(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPostAttach)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreCreate(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPreCreate)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostCreate(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPostCreate)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreCreateView(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPreCreateView)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostCreateView(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPostCreateView)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreViewCreated(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPreViewCreated)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostViewCreated(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPostViewCreated)
        
        @TransformationMethod
        @JvmStatic
        fun onFragmentPreActivityCreated(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPreActivityCreated)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostActivityCreated(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPostActivityCreated)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreViewStateRestore(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPreViewStateRestore)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostViewStateRestore(fragment: Fragment) =
            dispatchIOEvent(
                fragment,
                FragmentLifecycleEventListener::onFragmentPostViewStateRestore
            )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreStart(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPreStart)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostStart(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPostStart)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreResume(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPreResume)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostResume(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPostResume)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreDeAttach(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPreDeAttach)

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostDeAttach(fragment: Fragment) =
            dispatchIOEvent(fragment, FragmentLifecycleEventListener::onFragmentPostDeAttach)

        private inline fun dispatchIOEvent(
            fragment: Fragment,
            crossinline event: ((FragmentLifecycleEventListener, Fragment, EventTimeMetricCapture) -> Unit)
        ) {
            val timeMetric = EventTimeMetricCapture()
            executor.execute {
                synchronized(listeners) {
                    listeners.forEach { listener ->
                        event.invoke(listener, fragment, timeMetric)
                    }
                }
            }
        }
    }
}