package com.instabug.apm.configuration;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.apm.APMImplementation;
import com.instabug.apm.constants.AppLaunchType;
import com.instabug.apm.constants.DefaultValues;
import com.instabug.apm.di.Provider;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.handler.experiment.ExperimentHandler;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.networkinterception.configuration.NetworkInterceptionConfigurationProvider;
import com.instabug.apm.networkinterception.configuration.NetworkInterceptionDefaultValuesKt;
import com.instabug.library.diagnostics.IBGDiagnostics;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashSet;

public class APMConfigurationHandlerImpl implements APMConfigurationHandler {

    private final APMConfigurationProvider apmConfigurationProvider;
    @Nullable
    private final NetworkInterceptionConfigurationProvider networkConfigurationProvider;
    private Logger apmLogger = ServiceLocator.getApmLogger();
    private final Provider<ConfigurationHandler[]> sdkConfigurationHandlersProvider;
    private final Provider<ConfigurationHandler[]> apmConfigurationHandlersProvider;

    public APMConfigurationHandlerImpl(
            APMConfigurationProvider apmConfigurationProvider,
            @Nullable NetworkInterceptionConfigurationProvider networkConfigurationProvider,
            Provider<ConfigurationHandler[]> sdkConfigurationHandlersProvider,
            Provider<ConfigurationHandler[]> apmConfigurationHandlersProvider
    ) {
        this.apmConfigurationProvider = apmConfigurationProvider;
        this.networkConfigurationProvider = networkConfigurationProvider;
        this.sdkConfigurationHandlersProvider = sdkConfigurationHandlersProvider;
        this.apmConfigurationHandlersProvider = apmConfigurationHandlersProvider;
    }

    @Override
    public boolean handleConfiguration(@Nullable String configuration) {
        boolean isConfigurationParsed = false;

        ConfigurationHandler[] sdkConfigurationHandlers = sdkConfigurationHandlersProvider.invoke();
        ConfigurationHandler[] apmConfigurationHandlers = apmConfigurationHandlersProvider.invoke();

        try {
            if (configuration != null) {
                JSONObject configurationJsonObject = new JSONObject(configuration);
                JSONObject apmJsonObject = configurationJsonObject.optJSONObject("apm");

                boolean shouldSendLegacyApmSessions = configurationJsonObject.optBoolean("send_legacy_apm_session");
                apmConfigurationProvider.setSendLegacyAPMSessions(shouldSendLegacyApmSessions);

                if (apmJsonObject != null) {
                    boolean isApmEnabled = apmJsonObject.optBoolean("enabled", DefaultValues.IS_APM_FEATURE_AVAILABLE);
                    apmConfigurationProvider.setAPMFeatureAvailability(isApmEnabled);

                    boolean isCrashDetectionEnabled = apmJsonObject.optBoolean("crash_detection_enabled",
                            DefaultValues.CRASH_DETECTION_ENABLED);
                    apmConfigurationProvider.setCrashDetectionEnabled(isCrashDetectionEnabled);

                    boolean debugModeEnabled = apmJsonObject.optBoolean("debug_mode_enabled",
                            DefaultValues.DEBUG_MODE_ENABLED);
                    apmConfigurationProvider.setDebugModeEnabled(debugModeEnabled);

                    long syncInterval = apmJsonObject.optLong("sync_interval", DefaultValues.SYNC_INTERVAL);
                    apmConfigurationProvider.setSyncInterval(syncInterval);

                    boolean sessionStoreLimitEnabled = apmJsonObject.optBoolean(
                            "session_store_limit_enabled",
                            DefaultValues.SESSION_STORE_LIMIT_ENABLED
                    );
                    apmConfigurationProvider.setSessionStoreLimitEnabled(sessionStoreLimitEnabled);

                    int sessionStoreLimit = apmJsonObject.optInt(
                            "session_store_limit",
                            DefaultValues.SESSION_STORE_LIMIT
                    );
                    apmConfigurationProvider.setSessionStoreLimit(sessionStoreLimit);

                    invokeConfigurationHandlers(apmJsonObject, apmConfigurationHandlers);
                    handleAppLaunchConfigurations(apmJsonObject);
                    handleExecutionTracesConfiguration(apmJsonObject);
                    handleNetworkConfiguration(apmJsonObject);
                    handleUiTraceConfiguration(apmJsonObject);
                    handleExperimentsConfigurations(apmJsonObject);
                    handleFragmentSpansConfiguration(apmJsonObject);
                } else {
                    apmLogger.logSDKProtected("Can't parse APM configurations, object is null.");
                }
                invokeConfigurationHandlers(configurationJsonObject, sdkConfigurationHandlers);
                isConfigurationParsed = true;
            }
        } catch (JSONException e) {
            apmLogger.logSDKError(e.getMessage() != null ? e.getMessage() : "", e);
        }

        if (apmConfigurationProvider.isAPMFeatureAvailable()) {
            apmLogger.logSDKProtected("APM feature configs: " +
                    "\nEnabled: " + apmConfigurationProvider.isAPMFeatureAvailable() +
                    "\nTraces Enabled: " + apmConfigurationProvider.isExecutionTraceFeatureEnabled() +
                    "\nCold App Launches Enabled: " + apmConfigurationProvider.isColdAppLaunchesFeatureEnabled() +
                    "\nHot App Launches Enabled: " + apmConfigurationProvider.isHotAppLaunchesFeatureEnabled() +
                    "\nNetwork Logs Enabled: " + apmConfigurationProvider.isNetworkFeatureEnabled() +
                    "\nUI Traces Enabled: " + apmConfigurationProvider.isUiHangsFeatureEnabled() +
                    "\nFragment spans Enabled: " + apmConfigurationProvider.isFragmentSpansFeatureEnabled()
            );
        } else {
            apmLogger.logSDKProtected("APM feature configs: " +
                    "\nEnabled: false");
            resetConfiguration(apmConfigurationHandlers);
        }
        return isConfigurationParsed;
    }

    private static void resetConfigurationHandlers(ConfigurationHandler[] configurationHandlers) {
        for (ConfigurationHandler configurationHandler: configurationHandlers) {
            if (configurationHandler!=null) configurationHandler.reset();
        }
    }

    private void invokeConfigurationHandlers(JSONObject apmJsonObject, ConfigurationHandler[] handlers) {
        for (ConfigurationHandler handler : handlers) {
            if (handler != null) handler.handleConfigurations(apmJsonObject);
        }
    }

    private void handleAppLaunchConfigurations(JSONObject apmJsonObject) {
        JSONObject appLaunchesObject = apmJsonObject.optJSONObject("launches");
        if (appLaunchesObject != null) {
            boolean coldAppLaunchesEnabled = appLaunchesObject.optBoolean("enabled",
                    DefaultValues.Launches.ENABLED);
            apmConfigurationProvider.setColdAppLaunchesFeatureEnabled(coldAppLaunchesEnabled);

            handleHotAppLaunchConfigurations(appLaunchesObject);
            handleWarmAppLaunchConfiguration(appLaunchesObject);

            if (!coldAppLaunchesEnabled) {
                resetColdAppLaunchesConfiguration();
                clearAppLaunchesCache(AppLaunchType.COLD);
            } else {
                long limitPerRequest = appLaunchesObject.optLong("limit_per_request",
                        DefaultValues.Launches.LIMIT_PER_REQUEST_COLD);
                apmConfigurationProvider.setColdAppLaunchesLimitPerRequest(limitPerRequest);

                long storeLimit = appLaunchesObject.optLong("store_limit",
                        DefaultValues.Launches.STORE_LIMIT_COLD);
                apmConfigurationProvider.setColdAppLaunchesStoreLimit(storeLimit);
            }

            // EndLaunches API
            boolean isEndColdLaunchEnabled = appLaunchesObject.optBoolean("end_api_enabled",
                    DefaultValues.Launches.ENABLED);
            apmConfigurationProvider.setEndColdAppLaunchFeatureEnabled(isEndColdLaunchEnabled);
            boolean isHotAppLaunchEnabled =
                    apmConfigurationProvider.isHotAppLaunchesFeatureEnabled();
            boolean isWarmAppLaunchEnabled =
                    apmConfigurationProvider.isWarmAppLaunchFeatureEnabled();
            boolean isAppLaunchFeatureDisabled =
                    !(coldAppLaunchesEnabled || isHotAppLaunchEnabled || isWarmAppLaunchEnabled);
            if (isAppLaunchFeatureDisabled) {
                clearAppLaunchesCache();
            }
        } else {
            apmLogger.logSDKProtected("Can't parse app launches configurations, object is null.");
            resetColdAppLaunchesConfiguration();
            resetHotAppLaunchesConfiguration();
            resetWarmAppLaunchesConfiguration();
            clearAppLaunchesCache();
        }
    }

    private void handleHotAppLaunchConfigurations(JSONObject appLaunchesObject) {
        JSONObject hotAppLaunchesObject = appLaunchesObject.optJSONObject("hot");
        if (hotAppLaunchesObject != null) {
            boolean hotAppLaunchesEnabled = hotAppLaunchesObject.optBoolean("enabled",
                    DefaultValues.Launches.ENABLED);
            apmConfigurationProvider.setHotAppLaunchesFeatureEnabled(hotAppLaunchesEnabled);

            if (!hotAppLaunchesEnabled) {
                resetHotAppLaunchesConfiguration();
                clearAppLaunchesCache(AppLaunchType.HOT);
            } else {
                long limitPerRequest = hotAppLaunchesObject.optLong("limit_per_request",
                        DefaultValues.Launches.LIMIT_PER_REQUEST_HOT);
                apmConfigurationProvider.setHotAppLaunchesLimitPerRequest(limitPerRequest);

                long storeLimit = hotAppLaunchesObject.optLong("store_limit",
                        DefaultValues.Launches.STORE_LIMIT_HOT);
                apmConfigurationProvider.setHotAppLaunchesStoreLimit(storeLimit);
            }

            // EndLaunches API
            boolean isEndHotLaunchEnabled = hotAppLaunchesObject.optBoolean("end_api_enabled",
                    DefaultValues.Launches.ENABLED);
            apmConfigurationProvider.setEndHotAppLaunchFeatureEnabled(isEndHotLaunchEnabled);
        } else {
            clearAppLaunchesCache(AppLaunchType.HOT);
        }
    }

    private void handleWarmAppLaunchConfiguration(@NonNull JSONObject appLaunchesObject) {
        final JSONObject warmAppLaunchObject = appLaunchesObject.optJSONObject("warm");
        if (warmAppLaunchObject != null) {
            boolean enabled = warmAppLaunchObject.optBoolean(
                    "enabled",
                    DefaultValues.Launches.WARM_ENABLED
            );
            long limitPerRequest = warmAppLaunchObject.optLong(
                    "limit_per_request",
                    DefaultValues.Launches.LIMIT_PER_REQUEST_WARM
            );
            long storeLimit = warmAppLaunchObject.optLong(
                    "store_limit",
                    DefaultValues.Launches.STORE_LIMIT_WARM
            );
            boolean endAppLaunchEnabled = warmAppLaunchObject.optBoolean(
                    "end_api_enabled",
                    DefaultValues.Launches.END_WARM_ENABLED
            );
            apmConfigurationProvider.setWarmAppLaunchFeatureEnabled(enabled);
            apmConfigurationProvider.setWarmAppLaunchRequestLimit(limitPerRequest);
            apmConfigurationProvider.setWarmAppLaunchStoreLimit(storeLimit);
            apmConfigurationProvider.setEndWarmAppLaunchFeatureEnabled(endAppLaunchEnabled);
            if (!enabled) {
                clearAppLaunchesCache(AppLaunchType.WARM);
            }
        } else {
            resetWarmAppLaunchesConfiguration();
            if (!apmConfigurationProvider.isWarmAppLaunchFeatureEnabled()) {
                clearAppLaunchesCache(AppLaunchType.WARM);
            }
        }
    }

    private void resetColdAppLaunchesConfiguration() {
        apmConfigurationProvider.setColdAppLaunchesFeatureEnabled(DefaultValues.Launches.ENABLED);
        apmConfigurationProvider.setColdAppLaunchesLimitPerRequest(DefaultValues.Launches.LIMIT_PER_REQUEST_COLD);
        apmConfigurationProvider.setColdAppLaunchesStoreLimit(DefaultValues.Launches.STORE_LIMIT_COLD);
    }

    private void resetHotAppLaunchesConfiguration() {
        apmConfigurationProvider.setHotAppLaunchesFeatureEnabled(DefaultValues.Launches.ENABLED);
        apmConfigurationProvider.setHotAppLaunchesLimitPerRequest(DefaultValues.Launches.LIMIT_PER_REQUEST_COLD);
        apmConfigurationProvider.setHotAppLaunchesStoreLimit(DefaultValues.Launches.STORE_LIMIT_COLD);
    }

    private void resetWarmAppLaunchesConfiguration() {
        apmConfigurationProvider.resetWarmAppLaunchConfigurations();
    }

    public void clearAppLaunchesCache(@NonNull @AppLaunchType String type) {
        ServiceLocator.getApmImplementation().clearAppLaunchesCache(type);
    }

    public void clearAppLaunchesCache() {
        ServiceLocator.getApmImplementation().clearAppLaunchesCache();
    }

    private void handleExecutionTracesConfiguration(JSONObject apmJsonObject) {
        JSONObject tracesObject = apmJsonObject.optJSONObject("traces");
        if (tracesObject != null) {
            boolean enabled = tracesObject.optBoolean("enabled",
                    DefaultValues.Traces.ENABLED);
            apmConfigurationProvider.setExecutionTraceFeatureEnabled(enabled);
            if (!enabled) {
                resetExecutionTraceConfiguration();
                clearExecutionTraceCache();
            } else {
                long limitPerRequest = tracesObject.optLong("limit_per_request",
                        DefaultValues.Traces.LIMIT_PER_REQUEST);
                apmConfigurationProvider.setExecutionTraceLimitPerRequest(limitPerRequest);

                long storeLimit = tracesObject.optLong("store_limit",
                        DefaultValues.Traces.STORE_LIMIT);
                apmConfigurationProvider.setExecutionTraceStoreLimit(storeLimit);

                int storeAttributesLimit = tracesObject.optInt("store_attributes_limit",
                        DefaultValues.Traces.ATTRIBUTES_STORE_LIMIT);
                apmConfigurationProvider.setExecutionTraceStoreAttributesLimit(storeAttributesLimit);
            }
        } else {
            apmLogger.logSDKProtected("Can't parse execution traces configurations, object is null.");
            resetExecutionTraceConfiguration();
            clearExecutionTraceCache();
        }
    }

    public void clearExecutionTraceCache() {
        ServiceLocator.getApmImplementation().clearExecutionTracesCache();
    }

    public void resetExecutionTraceConfiguration() {
        apmConfigurationProvider.setExecutionTraceFeatureEnabled(DefaultValues.Traces.ENABLED);
        apmConfigurationProvider.setExecutionTraceLimitPerRequest(DefaultValues.Traces.LIMIT_PER_REQUEST);
        apmConfigurationProvider.setExecutionTraceStoreLimit(DefaultValues.Traces.STORE_LIMIT);
        apmConfigurationProvider.setExecutionTraceStoreAttributesLimit(DefaultValues.Traces.ATTRIBUTES_STORE_LIMIT);
    }

    @VisibleForTesting
    protected void handleUiTraceConfiguration(JSONObject apmJsonObject) {
        JSONObject uiObject = null;
        if (apmJsonObject != null) {
            uiObject = apmJsonObject.optJSONObject("ui");
        }
        if (uiObject != null) {
            boolean uiHangsEnabled = uiObject.optBoolean("enabled",
                    DefaultValues.UiHangs.ENABLED);
            boolean uiLoadingEnabled = handleUiLoadingConfiguration(uiObject.optJSONObject("screen_loading"));
            apmConfigurationProvider.setUiHangsFeatureEnabled(uiHangsEnabled);
            boolean uiTracesEnabled = uiHangsEnabled || uiLoadingEnabled;
            if (!uiTracesEnabled) {
                resetUiTraceConfiguration();
            } else {

                float smallDropThreshold = (float) uiObject.optDouble("small_drop_duration_mus",
                        (double) DefaultValues.UiTraces.SMALL_DROP_THRESHOLD);
                apmConfigurationProvider.setUiTraceSmallDropThreshold(smallDropThreshold);

                float largeDropThreshold = (float) uiObject.optDouble("large_drop_duration_mus",
                        (double) DefaultValues.UiTraces.LARGE_DROP_THRESHOLD);
                apmConfigurationProvider.setUiTraceLargeDropThreshold(largeDropThreshold);

                long syncSize = uiObject.optLong("limit_per_request",
                        DefaultValues.UiTraces.LIMIT_PER_REQUEST);
                apmConfigurationProvider.setUiTraceLimitPerRequest(syncSize);

                long cacheSize = uiObject.optLong("store_limit",
                        DefaultValues.UiTraces.STORE_LIMIT);
                apmConfigurationProvider.setUiTraceStoreLimit(cacheSize);
            }
        } else {
            apmLogger.logSDKProtected("Can't parse ui traces configurations, object is null.");
            resetUiTraceConfiguration();
        }
    }

    private boolean handleUiLoadingConfiguration(@Nullable JSONObject jsonObject) {
        if (jsonObject != null) {
            boolean isEnabled = jsonObject.optBoolean(
                    "enabled",
                    DefaultValues.UiLoadingMetrics.ENABLED
            );
            boolean isEndScreenLoadingEnabled = jsonObject.optBoolean(
                    "end_api_enabled",
                    DefaultValues.UiLoadingMetrics.END_SCREEN_LOADING_ENABLED
            );
            if (apmConfigurationProvider != null) {
                apmConfigurationProvider.setUiLoadingMetricsFeatureEnabled(isEnabled);
                apmConfigurationProvider.setEndScreenLoadingFeatureEnabled(isEndScreenLoadingEnabled);
                return isEnabled;
            }
        }
        return DefaultValues.UiLoadingMetrics.ENABLED;
    }

    private void resetUiTraceConfiguration() {
        resetUiHangsConfiguration();
        resetUiLoadingConfiguration();
        apmConfigurationProvider.setUiTraceLimitPerRequest(DefaultValues.UiTraces.LIMIT_PER_REQUEST);
        apmConfigurationProvider.setUiTraceStoreLimit(DefaultValues.UiTraces.STORE_LIMIT);
        apmConfigurationProvider.setUiTraceLargeDropThreshold(DefaultValues.UiTraces.LARGE_DROP_THRESHOLD);
        apmConfigurationProvider.setUiTraceSmallDropThreshold(DefaultValues.UiTraces.SMALL_DROP_THRESHOLD);
    }

    private void resetUiHangsConfiguration() {
        if (apmConfigurationProvider != null) {
            apmConfigurationProvider.setUiHangsFeatureEnabled(DefaultValues.UiHangs.ENABLED);
        }
    }

    private void resetUiLoadingConfiguration() {
        if (apmConfigurationProvider != null) {
            apmConfigurationProvider.setUiLoadingMetricsFeatureEnabled(
                    DefaultValues.UiLoadingMetrics.ENABLED
            );
            apmConfigurationProvider.setEndScreenLoadingFeatureEnabled(
                    DefaultValues.UiLoadingMetrics.END_SCREEN_LOADING_ENABLED
            );
        }
    }

    /**
     * ToDo: move each feature handler and their keys to a new handler and keys class ¯\_(ツ)_/¯
     */
    private void handleNetworkConfiguration(JSONObject apmJsonObject) {
        JSONObject networkObject = apmJsonObject.optJSONObject("network");
        if (networkObject != null) {
            boolean enabled = networkObject.optBoolean("enabled",
                    DefaultValues.Network.ENABLED);
            apmConfigurationProvider.setNetworkEnabled(enabled);
            if (!enabled) {
                resetNetworkConfiguration();
                clearNetworkLogCache();
            } else {
                long networkLogsRequestLimit = networkObject.optLong("limit_per_request",
                        DefaultValues.Network.LIMIT_PER_REQUEST);
                apmConfigurationProvider.setNetworkLogsLimitPerRequest(networkLogsRequestLimit);

                long networkLogsCacheLimit = networkObject.optLong("store_limit",
                        DefaultValues.Network.STORE_LIMIT);
                apmConfigurationProvider.setNetworkLogsStoreLimit(networkLogsCacheLimit);

                int storeAttributesLimit = networkObject.optInt("store_attributes_limit",
                        DefaultValues.Network.ATTRIBUTES_STORE_LIMIT);
                apmConfigurationProvider.setNetworkLogsStoreAttributesLimit(storeAttributesLimit);

                boolean isGraphQlEnabled = networkObject.optBoolean(
                        "graphql_enabled",
                        DefaultValues.Network.GRAPH_QL_ENABLED
                );
                apmConfigurationProvider.setNetworkGraphQlFeatureEnabled(isGraphQlEnabled);
                if (!isGraphQlEnabled) {
                    clearGraphQlCache();
                }
                boolean isGrpcEnabled = networkObject.optBoolean(
                        "grpc_enabled",
                        DefaultValues.Network.GRPC_ENABLED
                );
                apmConfigurationProvider.setNetworkGrpcInterceptionFeatureEnabled(isGrpcEnabled);
                if (!isGrpcEnabled) {
                    clearGrpcCache();
                }
                handleNetworkSanitizationConfigurations(networkObject);
            }
        } else {
            apmLogger.logSDKProtected("Can't parse network logs configurations, object is null.");
            resetNetworkConfiguration();
            clearNetworkLogCache();
        }
        handleW3CExternalTraceIdConfigurations();
    }

    private void handleNetworkSanitizationConfigurations(JSONObject networkObject) {
        if (networkConfigurationProvider != null) {
            boolean isKeywordSanitizationEnabled =
                    networkObject.optBoolean(
                            "sanitization_enabled",
                            NetworkInterceptionDefaultValuesKt.DEFAULT_IBG_KEYWORD_SANITIZATION_FEATURE_ENABLED
                    );
            networkConfigurationProvider.setKeywordSanitizationFeatureEnabled(
                    isKeywordSanitizationEnabled
            );
            JSONArray keywordsArray = networkObject.optJSONArray("sanitize_keys");
            if (keywordsArray != null) {
                try {
                    HashSet<String> keywordSet = new HashSet<>();
                    for (int index = 0; index < keywordsArray.length(); index++) {
                        keywordSet.add(keywordsArray.getString(index));
                    }
                    networkConfigurationProvider.setIBGSanitizationKeywords(keywordSet);
                } catch (JSONException e) {
                    networkConfigurationProvider.resetIBGSanitizationKeywords();
                    IBGDiagnostics.reportNonFatal(e, "failed to parse sanitization keywords");
                }
            } else {
                networkConfigurationProvider.resetIBGSanitizationKeywords();
            }
        }
    }

    private void handleW3CExternalTraceIdConfigurations(){
        boolean isW3ExternalTraceIdAvailable =
                apmConfigurationProvider.isW3CNetworkExternalTraceIdEnabled();
        if (!isW3ExternalTraceIdAvailable)
            ServiceLocator.getApmImplementation().clearW3CNetworkExternalTraceIdCache();
        else {
            boolean isAttachingGeneratedW3Available = apmConfigurationProvider.
                    isAttachingGeneratedW3CExternalTraceIdFeatureAvailable();
            boolean isAttachingCapturedW3Available = apmConfigurationProvider.
                    isAttachingCapturedW3CExternalTraceIdFeatureAvailable();

            if (!isAttachingGeneratedW3Available) {
                ServiceLocator.getApmImplementation().clearGeneratedW3CExternalTraceIdCache();
            }
            if (!isAttachingCapturedW3Available)
                ServiceLocator.getApmImplementation().clearCapturedW3CExternalTraceIdCache();
        }
    }

    private void clearNetworkLogCache() {
        ServiceLocator.getApmImplementation().clearNetworkCache();
    }

    private void clearGraphQlCache() {
        APMImplementation apmImplementation = ServiceLocator.getApmImplementation();
        if (apmImplementation != null) {
            apmImplementation.clearGraphQlCache();
        }
    }

    private void clearGrpcCache() {
        APMImplementation apmImplementation = ServiceLocator.getApmImplementation();
        if (apmImplementation != null) {
            apmImplementation.clearGrpcCache();
        }
    }


    private void resetNetworkConfiguration() {
        apmConfigurationProvider.setNetworkEnabled(DefaultValues.Network.ENABLED);
        apmConfigurationProvider.setNetworkLogsLimitPerRequest(DefaultValues.Network.LIMIT_PER_REQUEST);
        apmConfigurationProvider.setNetworkLogsStoreLimit(DefaultValues.Network.STORE_LIMIT);
        apmConfigurationProvider.setNetworkLogsStoreAttributesLimit(DefaultValues.Network.ATTRIBUTES_STORE_LIMIT);
        apmConfigurationProvider.setNetworkGraphQlFeatureEnabled(DefaultValues.Network.GRAPH_QL_ENABLED);
        apmConfigurationProvider.setNetworkGrpcInterceptionFeatureEnabled(DefaultValues.Network.GRPC_ENABLED);
        if (networkConfigurationProvider != null) {
            networkConfigurationProvider.resetKeywordSanitizationConfigurations();
        }
    }

    private void handleExperimentsConfigurations(JSONObject apmJsonObject) {
        boolean enabled = false;
        JSONObject experimentsJsonObject = apmJsonObject.optJSONObject("experiments");
        if (experimentsJsonObject != null) {
            enabled = experimentsJsonObject.optBoolean(
                    "enabled",
                    DefaultValues.Experiments.ENABLED
            );
            apmConfigurationProvider.setExperimentsFeatureEnabled(enabled);
            if (enabled) {
                int limitPerRequest =
                        experimentsJsonObject.optInt(
                                "limit_per_request",
                                DefaultValues.Experiments.LIMIT_PER_REQUEST
                        );
                apmConfigurationProvider.setExperimentsLimitPerRequest(limitPerRequest);
            } else {
                apmConfigurationProvider.resetExperimentsLimitPerRequest();
            }
        } else {
            resetExperimentsConfigurations();
        }
        if (!enabled) {
            clearExperiments();
        }
    }

    private void handleFragmentSpansConfiguration(JSONObject apmJsonObject) {
        boolean enabled = DefaultValues.FragmentSpans.ENABLED;
        JSONObject fragmentsJson = apmJsonObject.optJSONObject("fragments");
        if (fragmentsJson != null) {
            enabled = fragmentsJson.optBoolean("enabled", DefaultValues.FragmentSpans.ENABLED);
            apmConfigurationProvider.setFragmentSpansFeatureEnabledFlag(enabled);

            int storeLimit = fragmentsJson.optInt("store_limit", DefaultValues.FragmentSpans.STORE_LIMIT);
            apmConfigurationProvider.setFragmentSpansStoreLimit(storeLimit);

            int limitPerRequest = fragmentsJson.optInt("limit_per_request", DefaultValues.FragmentSpans.LIMIT_PER_REQUEST);
            apmConfigurationProvider.setFragmentSpansLimitPerRequest(limitPerRequest);
        } else {
            resetFragmentSpansConfiguration();
        }

        if (!enabled) {
            ServiceLocator.getFragmentSpansHelper().onFeatureDisabled();
        }
    }

    private void resetFragmentSpansConfiguration() {
        apmConfigurationProvider.resetFragmentSpansFeatureEnabledFlag();
        apmConfigurationProvider.resetFragmentSpansStoreLimit();
        apmConfigurationProvider.resetFragmentSpansLimitPerRequest();
    }


    private void resetExperimentsConfigurations() {
        apmConfigurationProvider.resetExperimentsFeatureEnabledFlag();
        apmConfigurationProvider.resetExperimentsLimitPerRequest();
    }

    private void clearExperiments() {
        ExperimentHandler experimentHandler = ServiceLocator.getExperimentHandler();
        if (experimentHandler != null) {
            experimentHandler.clearAllExperimentsAsync();
        }
    }

    private void clearApmState() {
        APMStateProvider stateProvider = ServiceLocator.getApmStateProvider();
        if (stateProvider != null) {
            stateProvider.clearState();
        }
    }


    private void resetConfiguration(ConfigurationHandler[] apmConfigurationHandlers) {
        apmConfigurationProvider.setAPMFeatureAvailability(DefaultValues.IS_APM_FEATURE_AVAILABLE);
        apmConfigurationProvider.setCrashDetectionEnabled(DefaultValues.CRASH_DETECTION_ENABLED);
        apmConfigurationProvider.setDebugModeEnabled(DefaultValues.DEBUG_MODE_ENABLED);
        apmConfigurationProvider.setSyncInterval(DefaultValues.SYNC_INTERVAL);
        apmConfigurationProvider.resetSessionStoreLimitEnabledFlag();
        apmConfigurationProvider.resetSessionStoreLimit();
        resetNetworkConfiguration();
        clearNetworkLogCache();
        resetUiTraceConfiguration();
        resetExecutionTraceConfiguration();
        clearExecutionTraceCache();
        resetColdAppLaunchesConfiguration();
        resetHotAppLaunchesConfiguration();
        resetWarmAppLaunchesConfiguration();
        clearAppLaunchesCache();
        resetExperimentsConfigurations();
        resetFragmentSpansConfiguration();
        clearExperiments();
        clearApmState();
        ServiceLocator.getFragmentSpansHelper().onFeatureDisabled();
        resetConfigurationHandlers(apmConfigurationHandlers);
    }
}