package com.instabug.apm.cache.handler.uitrace;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_BATTERY_LEVEL;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_CONTAINER_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_DURATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_LARGE_DROPS_DURATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_MODULE_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_ORIENTATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_POWER_SAVE_MODE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_REFRESH_RATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_SCREEN_TITLE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_SESSION_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_SMALL_DROPS_DURATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_START_TIME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.COLUMN_USER_DEFINED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry.TABLE_NAME;

import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.UiLoadingModel;
import com.instabug.apm.cache.model.UiTraceCacheModel;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.webview.webview_trace.handler.APMWebViewTraceCacheHandler;
import com.instabug.apm.webview.webview_trace.model.WebViewCacheModel;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiLoadingMetricEntry;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;

import java.util.ArrayList;
import java.util.List;

public class UiTraceCacheHandlerImpl implements UiTraceCacheHandler {
    @Nullable
    private DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
    private Logger apmLogger = ServiceLocator.getApmLogger();
    private UiLoadingMetricCacheHandler uiLoadingMetricCacheHandler =
            ServiceLocator.getUiLoadingMetricCacheHandler();

    @Override
    public long insert(UiTraceCacheModel uiTrace) {
        if (uiTrace == null) {
            return -1;
        }
        DatabaseManager localDbManager = getDatabaseManager();
        if (localDbManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = localDbManager.openDatabase();
            ContentValues values = getContentValues(uiTrace);
            putIdToContentValuesIfPossible(uiTrace, values);
            return sqLiteDatabaseWrapper.insertWithOnConflict(TABLE_NAME, null, values);
        }
        return -1;
    }

    private static void putIdToContentValuesIfPossible(UiTraceCacheModel uiTrace, ContentValues values) {
        if (uiTrace.getId() > 0) values.put(COLUMN_ID, uiTrace.getId());
    }

    @Override
    public int update(@Nullable UiTraceCacheModel uiTrace) {
        DatabaseManager localDbManager = getDatabaseManager();
        if (uiTrace == null || localDbManager == null) return 0;
        SQLiteDatabaseWrapper sqLiteDatabaseWrapper = localDbManager.openDatabase();
        ContentValues contentValues = getContentValues(uiTrace);
        String whereClause = COLUMN_ID + " = ?";
        String[] whereArgs = {"" + uiTrace.getId()};
        return sqLiteDatabaseWrapper.update(TABLE_NAME, contentValues, whereClause, whereArgs);
    }

    @Override
    public long insertUiLoadingModel(@Nullable UiTraceCacheModel uiTrace) {
        if (uiTrace == null) return -1;
        UiLoadingModel uiLoadingModel = uiTrace.getUiLoadingModel();
        if (uiLoadingModel != null && uiTrace.getId() != -1 && uiLoadingMetricCacheHandler != null) {
            return uiLoadingMetricCacheHandler.insert(uiLoadingModel, uiTrace.getId());
        }
        return -1;
    }

    private ContentValues getContentValues(UiTraceCacheModel uiTrace) {
        ContentValues values = new ContentValues();
        if (uiTrace.getName() != null) {
            values.put(COLUMN_NAME, uiTrace.getName());
        }
        if (uiTrace.getSessionId() != null) {
            values.put(COLUMN_SESSION_ID, uiTrace.getSessionId());
        }
        values.put(COLUMN_DURATION, uiTrace.getDuration());
        values.put(COLUMN_SMALL_DROPS_DURATION, uiTrace.getSmallDropsDuration());
        values.put(COLUMN_LARGE_DROPS_DURATION, uiTrace.getLargeDropsDuration());
        values.put(COLUMN_BATTERY_LEVEL, uiTrace.getBatteryLevel());
        values.put(COLUMN_USER_DEFINED, uiTrace.isUserDefined());
        values.put(COLUMN_REFRESH_RATE, uiTrace.getRefreshRate());
        values.put(COLUMN_START_TIME, uiTrace.getStartTimestamp());
        if (uiTrace.getScreenTitle() != null) {
            values.put(COLUMN_SCREEN_TITLE, uiTrace.getScreenTitle());
        }
        if (uiTrace.getPowerSaveMode() != null) {
            values.put(COLUMN_POWER_SAVE_MODE, uiTrace.getPowerSaveMode());
        }
        if (uiTrace.getContainerName() != null) {
            values.put(COLUMN_CONTAINER_NAME, uiTrace.getContainerName());
        }
        if (uiTrace.getModuleName() != null) {
            values.put(COLUMN_MODULE_NAME, uiTrace.getModuleName());
        }
        if (uiTrace.getOrientation() != null) {
            values.put(COLUMN_ORIENTATION, uiTrace.getOrientation());
        }
        return values;
    }

    @Override
    public void cleanUp() {
        DatabaseManager localDbManager = getDatabaseManager();
        if (localDbManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = localDbManager.openDatabase();
            String query = "DELETE FROM " + TABLE_NAME
                    + " WHERE " + COLUMN_NAME + " IS NULL"
                    + " OR " + COLUMN_DURATION + " = 0";
            try {
                sqLiteDatabaseWrapper.execSQL(query);
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (sqLiteDatabaseWrapper != null) {
                    sqLiteDatabaseWrapper.close();
                }
            }
        }
    }

    @Override
    public void trimToLimit(long limit) {
        DatabaseManager localDbManager = getDatabaseManager();
        if (localDbManager != null) {
            String selectByLimitDescendingQuery = " SELECT " + COLUMN_ID
                    + " FROM " + TABLE_NAME
                    + " ORDER BY " + COLUMN_ID + " DESC"
                    + " LIMIT ?";
            String whereClause = COLUMN_ID + " NOT IN (" + selectByLimitDescendingQuery + ")";
            String[] whereArgs = {String.valueOf(limit)};

            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = localDbManager.openDatabase();
            try {
                sqLiteDatabaseWrapper.delete(TABLE_NAME, whereClause, whereArgs);
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (sqLiteDatabaseWrapper != null) {
                    sqLiteDatabaseWrapper.close();
                }
            }
        }
    }

    @Override
    public int trimToLimit(String sessionID, long limit) {
        DatabaseManager localDbManager = getDatabaseManager();
        if (localDbManager != null) {
            String selectByLimitDescendingQuery = "SELECT " + COLUMN_ID
                    + " FROM " + TABLE_NAME
                    + " where " + COLUMN_SESSION_ID + " = ? AND " + COLUMN_DURATION + " > 0 "
                    + " ORDER BY " + COLUMN_ID + " DESC"
                    + " LIMIT ?";

            String whereClause = COLUMN_SESSION_ID + " = ? AND " + COLUMN_DURATION + " > 0 AND "
                    + COLUMN_ID + " NOT IN (" + selectByLimitDescendingQuery + " ) ";
            String[] whereArgs = {sessionID, sessionID, String.valueOf(limit)};
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = localDbManager.openDatabase();
            int deletedTracesCount = sqLiteDatabaseWrapper.delete(TABLE_NAME, whereClause, whereArgs);
            sqLiteDatabaseWrapper.close();

            return deletedTracesCount;
        }
        return -1;
    }

    @Override
    public int clearPreviousUnEndedTraces(String sessionID) {
        DatabaseManager localDbManager = getDatabaseManager();
        if (localDbManager == null) return 0;
        String whereClause = COLUMN_SESSION_ID + " != ? AND " + COLUMN_DURATION + " = 0 ";
        String[] whereArgs = {sessionID};
        SQLiteDatabaseWrapper sqLiteDatabaseWrapper = localDbManager.openDatabase();
        return sqLiteDatabaseWrapper.delete(TABLE_NAME, whereClause, whereArgs);
    }

    @Nullable
    @Override
    public List<UiTraceCacheModel> getByLimit(int limit) {
        DatabaseManager localDbManager = getDatabaseManager();
        if (localDbManager != null) {
            List<UiTraceCacheModel> cacheList = new ArrayList<>();
            String query = "SELECT * FROM " + TABLE_NAME + " LIMIT ?";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = localDbManager.openDatabase();
            String[] selectionArgs = {String.valueOf(limit)};
            Cursor cursor = null;
            try {
                cursor = sqLiteDatabaseWrapper.rawQuery(query, selectionArgs);
                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        UiTraceCacheModel cacheModel = readFromCursor(cursor);
                        cacheList.add(cacheModel);
                    }
                }
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
                sqLiteDatabaseWrapper.close();
            }
            fillUiTraceMetrics(cacheList);
            return cacheList;
        }
        return null;
    }

    private UiTraceCacheModel readFromCursor(Cursor cursor) {
        UiTraceCacheModel cacheModel = new UiTraceCacheModel();
        cacheModel.setId(cursor.getLong(cursor.getColumnIndexOrThrow(COLUMN_ID)));
        cacheModel.setName(cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_NAME)));
        cacheModel.setScreenTitle(cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_SCREEN_TITLE)));
        cacheModel.setDuration(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_DURATION)));
        cacheModel.setSmallDropsDuration(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_SMALL_DROPS_DURATION)));
        cacheModel.setLargeDropsDuration(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_LARGE_DROPS_DURATION)));
        cacheModel.setBatteryLevel(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_BATTERY_LEVEL)));
        cacheModel.setPowerSaveMode(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_POWER_SAVE_MODE)) == 1);
        cacheModel.setRefreshRate(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_REFRESH_RATE)));
        cacheModel.setStartTimestamp(cursor.getLong(cursor.getColumnIndexOrThrow(COLUMN_START_TIME)));
        cacheModel.setContainerName(cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_CONTAINER_NAME)));
        cacheModel.setModuleName(cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_MODULE_NAME)));
        cacheModel.setOrientation(cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_ORIENTATION)));
        cacheModel.setUserDefined(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_USER_DEFINED)) == 1);
        return cacheModel;
    }

    private void fillUiTraceMetrics(@Nullable List<UiTraceCacheModel> cacheList) {
        if (cacheList == null) {
            return;
        }
        for (UiTraceCacheModel uiTraceCacheModel : cacheList) {
            if (uiTraceCacheModel != null) {
                addUiLoadingModelToUiTraces(uiTraceCacheModel);
                addWebViewCacheModelToUiTraces(uiTraceCacheModel);
            }
        }
    }

    private void addUiLoadingModelToUiTraces(@NonNull UiTraceCacheModel uiTraceCacheModel) {
        UiLoadingModel uiLoadingModel =
                uiLoadingMetricCacheHandler.getUiLoadingMetricForUiTrace(uiTraceCacheModel.getId());
        if (uiLoadingModel != null) {
            uiTraceCacheModel.setUiLoadingModel(uiLoadingModel);
        }
    }

    private void addWebViewCacheModelToUiTraces(@NonNull UiTraceCacheModel uiTraceCacheModel) {
        APMWebViewTraceCacheHandler webViewTraceHandler = ServiceLocator.getWebViewTraceCacheHandler();
        if (webViewTraceHandler == null) return;
        List<WebViewCacheModel> webViewCacheModels = webViewTraceHandler.retrieve(uiTraceCacheModel.getId());
        if (webViewCacheModels != null) {
            uiTraceCacheModel.setWebViewTraces(webViewCacheModels);
        }
    }

    @Override
    public void deleteByLimit(int limit) {
        DatabaseManager localDbManager = getDatabaseManager();
        if (localDbManager != null) {
            String selectByLimitQuery = "SELECT "
                    + COLUMN_ID + " FROM " + TABLE_NAME
                    + " LIMIT ?";
            String whereClause = COLUMN_ID + " IN (" + selectByLimitQuery + ")";
            String[] whereArgs = {String.valueOf(limit)};

            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = localDbManager.openDatabase();
            try {
                sqLiteDatabaseWrapper.delete(TABLE_NAME, whereClause, whereArgs);
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (sqLiteDatabaseWrapper != null) {
                    sqLiteDatabaseWrapper.close();
                }
            }
        }
    }

    @Override
    public void removeAll() {
        DatabaseManager localDbManager = getDatabaseManager();
        if (localDbManager != null) {
            String removeTracesQuery = "DELETE FROM " + InstabugDbContract.APMUiTraceEntry.TABLE_NAME;
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = localDbManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(removeTracesQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public void removeUiHangs() {
        DatabaseManager localDbManager = getDatabaseManager();
        if (localDbManager != null) {
            String updateQuery = "UPDATE " + TABLE_NAME
                    + " SET " + COLUMN_SMALL_DROPS_DURATION + " = -1" + InstabugDbContract.COMMA_SEP
                    + " " + COLUMN_LARGE_DROPS_DURATION + " = -1";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = localDbManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(updateQuery);
            sqLiteDatabaseWrapper.close();
            removeInvalidUiTraces();
        }
    }

    private void removeInvalidUiTraces() {
        DatabaseManager localDbManager = getDatabaseManager();
        if (localDbManager != null) {
            String updateQuery = "DELETE FROM " + TABLE_NAME
                    + " WHERE " +
                    "(" + COLUMN_SMALL_DROPS_DURATION + " = -1 OR " + COLUMN_LARGE_DROPS_DURATION + " = -1) " +
                    "AND " + COLUMN_ID + " NOT IN (SELECT " + APMUiLoadingMetricEntry.COLUMN_UI_TRACE_ID + " FROM " + APMUiLoadingMetricEntry.TABLE_NAME + " ) ";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = localDbManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(updateQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public void removeUiLoadingMetrics() {
        if (uiLoadingMetricCacheHandler != null) {
            uiLoadingMetricCacheHandler.removeAll();
            removeInvalidUiTraces();
        }
    }

    @Nullable
    @Override
    public List<UiTraceCacheModel> getUiTracesForSession(@NonNull String sessionID) {
        if (sessionID == null) {
            return null;
        }
        DatabaseManager localDbManager = getDatabaseManager();
        List<UiTraceCacheModel> cacheModels = null;
        if (localDbManager != null) {
            cacheModels = new ArrayList<>();
            SQLiteDatabaseWrapper databaseWrapper = localDbManager.openDatabase();
            String selectionClause = COLUMN_SESSION_ID + " = ? AND " + COLUMN_DURATION + " > ? ";
            String[] selectionArgs = {sessionID, "0"};
            Cursor cursor = null;
            try {
                cursor = databaseWrapper.query(TABLE_NAME,
                        null, selectionClause, selectionArgs,
                        null, null, null);
                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        cacheModels.add(readFromCursor(cursor));
                    }
                }
                databaseWrapper.close();
            } catch (Exception ex) {
                cacheModels = null;
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        if (cacheModels != null) {
            fillUiTraceMetrics(cacheModels);
        }
        return cacheModels;
    }

    @Nullable
    private DatabaseManager getDatabaseManager() {
        if (databaseManager == null) {
            databaseManager = ServiceLocator.getDatabaseManager();
        }
        return databaseManager;
    }

}
