package com.instabug.apm.cache.handler.fragments

import android.content.ContentValues
import android.database.Cursor
import com.instabug.apm.cache.model.FragmentSpansCacheModel
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.fragment.model.FragmentSpans
import com.instabug.apm.logger.internal.Logger
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.internal.storage.cache.db.InstabugDbContract

interface FragmentSpansCacheHandler {
    fun saveFragment(fragmentSpans: FragmentSpans): Long?
    fun trimToLimit(limit: Int)
    fun trimToLimit(sessionId: String, limit: Int): Int?
    fun deleteFragmentsForSession(sessionId: String)
    fun getFragmentsForSession(sessionID: String): List<FragmentSpansCacheModel>
    fun clearAll()
}

class FragmentSpansCacheHandlerImpl : FragmentSpansCacheHandler {

    private val databaseWrapper
        get() = ServiceLocator.getDatabaseManager()?.openDatabase()

    private val apmLogger: Logger
        get() = ServiceLocator.getApmLogger()

    override fun saveFragment(fragmentSpans: FragmentSpans): Long? {
        return kotlin.runCatching {
            databaseWrapper?.let {
                val values = ContentValues().apply {
                    put(InstabugDbContract.APMFragmentEntry.COLUMN_NAME, fragmentSpans.name)
                    put(
                        InstabugDbContract.APMFragmentEntry.COLUMN_SESSION_ID,
                        fragmentSpans.sessionId
                    )
                }
                it.insert(InstabugDbContract.APMFragmentEntry.TABLE_NAME, null, values)
            }
        }.onFailure {
            apmLogger.logSDKError("Error while inserting fragment ${fragmentSpans.name} into db due to ${it.message}")
            IBGDiagnostics.reportNonFatal(
                it,
                "Error while inserting fragment ${fragmentSpans.name} into db due to ${it.message}"
            )
        }.getOrNull()
    }

    override fun trimToLimit(limit: Int) {
        kotlin.runCatching {
            databaseWrapper?.let {
                val query = ("delete from " + InstabugDbContract.APMFragmentEntry.TABLE_NAME
                        + " where " + InstabugDbContract.APMFragmentEntry.COLUMN_ID + " not in ("
                        + " select " + InstabugDbContract.APMFragmentEntry.COLUMN_ID
                        + " from " + InstabugDbContract.APMFragmentEntry.TABLE_NAME
                        + " order by " + InstabugDbContract.APMFragmentEntry.COLUMN_ID + " desc"
                        + " limit " + limit + " )")
                it.execSQL(query)
            }
        }.onFailure {
            apmLogger.logSDKError("Error while trimming apm fragments due to ${it.message}")
            IBGDiagnostics.reportNonFatal(
                it,
                "Error while trimming apm fragments due to ${it.message}"
            )
        }.getOrNull()
    }


    override fun trimToLimit(sessionId: String, limit: Int): Int? {
        return kotlin.runCatching {
            databaseWrapper?.let {
                val selectByLimitDescendingQuery =
                    ("SELECT " + InstabugDbContract.APMFragmentEntry.COLUMN_ID
                            + " FROM " + InstabugDbContract.APMFragmentEntry.TABLE_NAME
                            + " where " + InstabugDbContract.APMFragmentEntry.COLUMN_SESSION_ID + " = ?"
                            + " ORDER BY " + InstabugDbContract.APMFragmentEntry.COLUMN_ID + " DESC"
                            + " LIMIT ?")

                val whereClause =
                    (InstabugDbContract.APMFragmentEntry.COLUMN_SESSION_ID + " = ? AND "
                            + InstabugDbContract.APMFragmentEntry.COLUMN_ID + " NOT IN (" + selectByLimitDescendingQuery + ")")
                val whereArgs = arrayOf(sessionId, sessionId, limit.toString())
                it.delete(
                    InstabugDbContract.APMFragmentEntry.TABLE_NAME,
                    whereClause,
                    whereArgs
                )
            }
        }.onFailure {
            apmLogger.logSDKError("Error while trimming apm fragments due to ${it.message}")
            IBGDiagnostics.reportNonFatal(
                it,
                "Error while trimming apm fragments due to ${it.message}"
            )
        }.getOrNull()
    }

    override fun deleteFragmentsForSession(sessionId: String) {
        kotlin.runCatching {
            databaseWrapper?.let {
                val whereClause = InstabugDbContract.APMFragmentEntry.COLUMN_SESSION_ID + " = ?"
                val whereArgs = arrayOf(sessionId)
                it.delete(InstabugDbContract.APMFragmentEntry.TABLE_NAME, whereClause, whereArgs)
            }
        }.onFailure {
            apmLogger.logSDKError("Error while deleting apm fragments for session $sessionId db due to ${it.message}")
            IBGDiagnostics.reportNonFatal(
                it,
                "Error while deleting apm fragments for session $sessionId db due to ${it.message}"
            )
        }
    }

    override fun getFragmentsForSession(sessionID: String): List<FragmentSpansCacheModel> {
        val fragments = mutableListOf<FragmentSpansCacheModel>()
        var cursor: Cursor? = null
        kotlin.runCatching {
            cursor = databaseWrapper?.let {
                val selectionClause = InstabugDbContract.APMFragmentEntry.COLUMN_SESSION_ID + " = ?"
                val selectionArgs = arrayOf(sessionID)
                it.query(
                    InstabugDbContract.APMFragmentEntry.TABLE_NAME,
                    null,
                    selectionClause,
                    selectionArgs,
                    null,
                    null,
                    null
                )
            }
            while (cursor?.moveToNext() == true) {
                cursor?.toFragmentCacheModel()?.let(fragments::add)
            }
        }.also {
            cursor?.close()
        }.onFailure {
            apmLogger.logSDKError("Error while getting apm fragments from db db due to ${it.message}")
            IBGDiagnostics.reportNonFatal(
                it,
                "Error while getting apm fragments from db db due to ${it.message}"
            )
        }
        return fragments.toList()
    }

    override fun clearAll() {
        kotlin.runCatching {
            databaseWrapper?.delete(InstabugDbContract.APMFragmentEntry.TABLE_NAME, null, null)
        }.onFailure {
            apmLogger.logSDKError("Error while deleting apm fragments due to ${it.message}")
            IBGDiagnostics.reportNonFatal(
                it,
                "Error while deleting apm fragments due to ${it.message}"
            )
        }
    }

    private fun Cursor.toFragmentCacheModel(): FragmentSpansCacheModel = FragmentSpansCacheModel(
        id = getLong(getColumnIndexOrThrow(InstabugDbContract.APMFragmentEntry.COLUMN_ID)),
        name = getString(getColumnIndexOrThrow(InstabugDbContract.APMFragmentEntry.COLUMN_NAME)),
        sessionId = getLong(getColumnIndexOrThrow(InstabugDbContract.APMFragmentEntry.COLUMN_SESSION_ID))
    )

}
