package com.instabug.library.apm_network_log_repository

import androidx.annotation.VisibleForTesting
import com.instabug.apm.model.APMNetworkLog
import com.instabug.apm.sanitization.Sanitizer
import com.instabug.library.apmokhttplogger.model.OkHttpAPMNetworkLog
import com.instabug.library.apmokhttplogger.model.OkHttpAPMNetworkLogWrapper
import com.instabug.library.diagnostics.IBGDiagnostics
import okhttp3.Call
import java.util.WeakHashMap

class NetworkLogWithCounter(
    val networkLog: OkHttpAPMNetworkLogWrapper,
    var count: Int
)

interface APMNetworkLogRepository {
    /**
     * Start network log that collects data from both interceptor and event listener and link both of them by the unique call object
     * @Param call object that links between interceptor and event listener
     * @return APMNetworkLog that will be used by interceptor and event listener for every one of them to fill the data from its side
     */
    fun start(call: Call): OkHttpAPMNetworkLog

    /**
     * decides whether to wait or end and insert network lof to the database
     * @Param call object that links between interceptor and event listener
     * @param exception passed to get inserted with the NetworkLog if occurred
     */
    fun end(call: Call, exception: Exception? = null)

    /**
     * gets ApmNetworkLog from modelsMap
     * @param call that references the network log
     * @return APMNetworkLog may be null if not present on the map
     */
    operator fun get(call: Call): OkHttpAPMNetworkLog?

    /**
     * gets the injectable external trace header from APMNetworkLog
     * @param call that references the network log
     * @param capturedId the captured external trace id from the original request
     * @return nullable Pair<String, String>
     */
    fun getInjectableHeader(call: Call, capturedId: String?): List<Pair<String, String>>?
}

class APMNetworkLogRepositoryImpl(private val sanitizerProvider: () -> Sanitizer<APMNetworkLog>) :
    APMNetworkLogRepository {

    @VisibleForTesting
    val modelsMap: MutableMap<Call, NetworkLogWithCounter> =
        WeakHashMap()

    override fun start(call: Call): OkHttpAPMNetworkLog = synchronized(call) {
        modelsMap[call]?.let(::getAndIncrementCount) ?: startNew(call)
    }

    private fun startNew(call: Call): OkHttpAPMNetworkLog =
        OkHttpAPMNetworkLogWrapper().also { modelsMap[call] = NetworkLogWithCounter(it, 1) }

    private fun getAndIncrementCount(networkLogWithCounter: NetworkLogWithCounter): OkHttpAPMNetworkLog =
        with(networkLogWithCounter) {
            ++count
            networkLog
        }

    override fun end(call: Call, exception: Exception?) {
        synchronized(call) {
            modelsMap[call]?.apply {
                when {
                    count > 1 -> --count
                    count == 1 -> cacheAndRemoveInMemoryNetworkLog(call, networkLog, exception)
                    else -> reportIllegalStateNonFatal("Illegal NetworkLog callers count: $count")
                }
            } ?: reportIllegalStateNonFatal("Ending NetworkLog without starting it")
        }
    }

    private fun cacheAndRemoveInMemoryNetworkLog(
        call: Call,
        networkLog: OkHttpAPMNetworkLogWrapper,
        exception: Exception?
    ) {
        networkLog.insert(exception, sanitizerProvider())
        modelsMap.remove(call)
    }

    private fun reportIllegalStateNonFatal(message: String) =
        IBGDiagnostics.reportNonFatal(IllegalStateException(message), message)

    override fun get(call: Call): OkHttpAPMNetworkLog? = synchronized(call) {
        modelsMap[call]?.networkLog
    }

    override fun getInjectableHeader(call: Call, capturedId: String?): List<Pair<String, String>>? = synchronized(call){
        modelsMap[call]?.networkLog?.getInjectableHeader(capturedId)
    }
}