/*
 * Decompiled with CFR 0.152.
 */
package com.helger.io.file;

import com.helger.annotation.Nonnegative;
import com.helger.annotation.concurrent.Immutable;
import com.helger.annotation.style.ReturnsMutableCopy;
import com.helger.base.enforce.ValueEnforcer;
import com.helger.base.equals.EqualsHelper;
import com.helger.base.io.EAppend;
import com.helger.base.io.nonblocking.NonBlockingBufferedInputStream;
import com.helger.base.io.nonblocking.NonBlockingBufferedOutputStream;
import com.helger.base.io.nonblocking.NonBlockingBufferedReader;
import com.helger.base.io.nonblocking.NonBlockingBufferedWriter;
import com.helger.base.io.stream.StreamHelper;
import com.helger.base.log.ConditionalLogger;
import com.helger.base.state.EChange;
import com.helger.base.state.EValidity;
import com.helger.base.system.SystemProperties;
import com.helger.collection.commons.CommonsArrayList;
import com.helger.collection.commons.ICommonsList;
import com.helger.io.file.ERandomAccessFileMode;
import com.helger.io.file.FileIOError;
import com.helger.io.file.FileOperations;
import com.helger.io.file.FilenameHelper;
import com.helger.io.misc.SizeHelper;
import com.helger.io.stream.CountingFileInputStream;
import com.helger.io.stream.CountingFileOutputStream;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.RandomAccessFile;
import java.io.Reader;
import java.io.Writer;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.Locale;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Immutable
public final class FileHelper {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileHelper.class);
    private static final ConditionalLogger CONDLOG = new ConditionalLogger(LOGGER, true);
    private static final FileHelper INSTANCE = new FileHelper();

    private FileHelper() {
    }

    public static boolean isWarnOnIssues() {
        return CONDLOG.isEnabled();
    }

    public static void setWarnOnIssues(boolean bl) {
        CONDLOG.setEnabled(bl);
    }

    public static boolean equalFiles(@Nonnull File file3, @Nonnull File file4) {
        return EqualsHelper.equalsCustom((Object)file3, (Object)file4, (file, file2) -> FilenameHelper.getCleanPath(file.getAbsoluteFile()).equals(FilenameHelper.getCleanPath(file2.getAbsoluteFile())));
    }

    public static boolean existsFile(@Nullable File file) {
        return file != null && file.isFile();
    }

    public static boolean existsDir(@Nullable File file) {
        return file != null && file.isDirectory();
    }

    public static boolean canReadAndWriteFile(@Nullable File file) {
        File file2;
        if (file == null) {
            return false;
        }
        return !(file.exists() ? !file.canRead() || !file.canWrite() : (file2 = file.getAbsoluteFile().getParentFile()) == null || !file2.isDirectory() || !file2.canRead() || !file2.canWrite());
    }

    @Nonnull
    public static EChange ensureParentDirectoryIsPresent(@Nonnull File file) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        File file2 = file.getParentFile();
        if (file2 == null || file2.exists()) {
            if (file2 != null && !file2.isDirectory()) {
                CONDLOG.warn("Parent object specified is not a directory: '" + String.valueOf(file2) + "'");
            }
            return EChange.UNCHANGED;
        }
        FileIOError fileIOError = FileOperations.createDirRecursive(file2);
        if (fileIOError.isFailure()) {
            throw new IllegalStateException("Failed to create parent of " + file.getAbsolutePath() + ": " + String.valueOf(fileIOError));
        }
        if (!file2.exists()) {
            throw new IllegalStateException("Parent of " + file.getAbsolutePath() + " is still not existing!");
        }
        return EChange.CHANGED;
    }

    @Nullable
    public static File getCanonicalFile(@Nullable File file) throws IOException {
        return file == null ? null : file.getCanonicalFile();
    }

    @Nullable
    public static File getCanonicalFileOrNull(@Nullable File file) {
        if (file != null) {
            try {
                return file.getCanonicalFile();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return null;
    }

    @Nullable
    public static String getCanonicalPath(@Nullable File file) throws IOException {
        return file == null ? null : file.getCanonicalPath();
    }

    @Nullable
    public static String getCanonicalPathOrNull(@Nullable File file) {
        if (file != null) {
            try {
                return file.getCanonicalPath();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return null;
    }

    public static boolean isParentDirectory(@Nonnull File file, @Nonnull File file2) {
        ValueEnforcer.notNull((Object)file, (String)"SearchDirectory");
        ValueEnforcer.notNull((Object)file2, (String)"StartDirectory");
        File file3 = file.getAbsoluteFile();
        File file4 = file2.getAbsoluteFile();
        try {
            file3 = FileHelper.getCanonicalFile(file3);
            file4 = FileHelper.getCanonicalFile(file4);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        if (!file3.isDirectory()) {
            return false;
        }
        for (File file5 = file4; file5 != null; file5 = file5.getParentFile()) {
            if (!file5.equals(file3)) continue;
            return true;
        }
        return false;
    }

    @Nullable
    public static FileInputStream getInputStream(@Nonnull File file) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        try {
            return new CountingFileInputStream(file);
        }
        catch (FileNotFoundException fileNotFoundException) {
            return null;
        }
    }

    @Nullable
    public static NonBlockingBufferedInputStream getBufferedInputStream(@Nonnull File file) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        FileInputStream fileInputStream = FileHelper.getInputStream(file);
        if (fileInputStream == null) {
            return null;
        }
        return new NonBlockingBufferedInputStream((InputStream)fileInputStream);
    }

    @Nullable
    public static InputStreamReader getReader(@Nonnull File file, @Nonnull Charset charset) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        ValueEnforcer.notNull((Object)charset, (String)"Charset");
        return StreamHelper.createReader((InputStream)FileHelper.getInputStream(file), (Charset)charset);
    }

    @Nullable
    public static NonBlockingBufferedReader getBufferedReader(@Nonnull File file, @Nonnull Charset charset) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        ValueEnforcer.notNull((Object)charset, (String)"Charset");
        InputStreamReader inputStreamReader = FileHelper.getReader(file, charset);
        if (inputStreamReader == null) {
            return null;
        }
        return new NonBlockingBufferedReader((Reader)inputStreamReader);
    }

    @Nullable
    public static FileOutputStream getOutputStream(@Nonnull File file) {
        return FileHelper.getOutputStream(file, EAppend.DEFAULT);
    }

    @Nullable
    public static FileOutputStream getOutputStream(@Nonnull File file, @Nonnull EAppend eAppend) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        ValueEnforcer.notNull((Object)eAppend, (String)"Append");
        if (FileHelper.internalCheckParentDirectoryExistanceAndAccess(file).isInvalid()) {
            return null;
        }
        try {
            return new CountingFileOutputStream(file, eAppend);
        }
        catch (FileNotFoundException fileNotFoundException) {
            CONDLOG.warn("Failed to create output stream for '" + String.valueOf(file) + "'; append: " + String.valueOf(eAppend) + ": " + fileNotFoundException.getClass().getName() + " - " + fileNotFoundException.getMessage());
            return null;
        }
    }

    @Nullable
    public static NonBlockingBufferedOutputStream getBufferedOutputStream(@Nonnull File file) {
        return FileHelper.getBufferedOutputStream(file, EAppend.DEFAULT);
    }

    @Nullable
    public static NonBlockingBufferedOutputStream getBufferedOutputStream(@Nonnull File file, @Nonnull EAppend eAppend) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        ValueEnforcer.notNull((Object)eAppend, (String)"Append");
        FileOutputStream fileOutputStream = FileHelper.getOutputStream(file, eAppend);
        if (fileOutputStream == null) {
            return null;
        }
        return new NonBlockingBufferedOutputStream((OutputStream)fileOutputStream);
    }

    @Nullable
    public static OutputStreamWriter getWriter(@Nonnull File file, @Nonnull Charset charset) {
        return FileHelper.getWriter(file, EAppend.DEFAULT, charset);
    }

    @Nullable
    public static OutputStreamWriter getWriter(@Nonnull File file, @Nonnull EAppend eAppend, @Nonnull Charset charset) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        ValueEnforcer.notNull((Object)charset, (String)"Charset");
        return StreamHelper.createWriter((OutputStream)FileHelper.getOutputStream(file, eAppend), (Charset)charset);
    }

    @Nullable
    public static NonBlockingBufferedWriter getBufferedWriter(@Nonnull File file, @Nonnull Charset charset) {
        return FileHelper.getBufferedWriter(file, EAppend.DEFAULT, charset);
    }

    @Nullable
    public static NonBlockingBufferedWriter getBufferedWriter(@Nonnull File file, @Nonnull EAppend eAppend, @Nonnull Charset charset) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        ValueEnforcer.notNull((Object)charset, (String)"Charset");
        OutputStreamWriter outputStreamWriter = FileHelper.getWriter(file, eAppend, charset);
        if (outputStreamWriter == null) {
            return null;
        }
        return new NonBlockingBufferedWriter((Writer)outputStreamWriter);
    }

    @Nullable
    public static PrintWriter getPrintWriter(@Nonnull File file, @Nonnull Charset charset) {
        return FileHelper.getPrintWriter(file, EAppend.DEFAULT, charset);
    }

    @Nullable
    public static PrintWriter getPrintWriter(@Nonnull File file, @Nonnull EAppend eAppend, @Nonnull Charset charset) {
        return new PrintWriter((Writer)FileHelper.getBufferedWriter(file, eAppend, charset));
    }

    @Nonnull
    static EValidity internalCheckParentDirectoryExistanceAndAccess(@Nonnull File file) {
        try {
            FileHelper.ensureParentDirectoryIsPresent(file);
        }
        catch (IllegalStateException illegalStateException) {
            CONDLOG.warn("Failed to create parent directory of '" + String.valueOf(file) + "'", (Exception)illegalStateException);
            return EValidity.INVALID;
        }
        File file2 = file.getParentFile();
        if (file2 != null && !file2.canWrite()) {
            CONDLOG.warn("Parent directory '" + String.valueOf(file2) + "' of '" + String.valueOf(file) + "' is not writable for current user '" + SystemProperties.getUserName() + "'");
            return EValidity.INVALID;
        }
        return EValidity.VALID;
    }

    @Nullable
    public static RandomAccessFile getRandomAccessFile(@Nonnull String string, @Nonnull ERandomAccessFileMode eRandomAccessFileMode) {
        return FileHelper.getRandomAccessFile(new File(string), eRandomAccessFileMode);
    }

    @Nullable
    public static RandomAccessFile getRandomAccessFile(@Nonnull File file, @Nonnull ERandomAccessFileMode eRandomAccessFileMode) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        ValueEnforcer.notNull((Object)((Object)eRandomAccessFileMode), (String)"Mode");
        try {
            return new RandomAccessFile(file, eRandomAccessFileMode.getMode());
        }
        catch (FileNotFoundException fileNotFoundException) {
            return null;
        }
    }

    public static boolean isFileNewer(@Nonnull File file, @Nonnull File file2) {
        ValueEnforcer.notNull((Object)file, (String)"File1");
        ValueEnforcer.notNull((Object)file2, (String)"aFile2");
        if (file.equals(file2)) {
            return false;
        }
        if (!file.exists()) {
            return false;
        }
        if (!file2.exists()) {
            return true;
        }
        return file.lastModified() > file2.lastModified();
    }

    @Nonnull
    public static String getFileSizeDisplay(@Nullable File file) {
        return FileHelper.getFileSizeDisplay(file, 0);
    }

    @Nonnull
    public static String getFileSizeDisplay(@Nullable File file, @Nonnegative int n) {
        if (file != null && file.exists()) {
            return FileHelper.getFileSizeDisplay(file.length(), n);
        }
        return "";
    }

    @Nonnull
    public static String getFileSizeDisplay(long l) {
        return FileHelper.getFileSizeDisplay(l, 0);
    }

    @Nonnull
    public static String getFileSizeDisplay(@Nonnegative long l, @Nonnegative int n) {
        ValueEnforcer.isGE0((long)l, (String)"FileSize");
        ValueEnforcer.isGE0((int)n, (String)"Decimals");
        return SizeHelper.getSizeHelperOfLocale(Locale.ROOT).getAsMatching(l, n);
    }

    @Nullable
    public static File getSecureFile(@Nullable File file) {
        if (file == null) {
            return null;
        }
        String string = null;
        for (File file2 = new File(FilenameHelper.getCleanPath(file)); file2 != null; file2 = file2.getParentFile()) {
            String string2 = file2.getParent() == null ? file2.getPath() : FilenameHelper.getAsSecureValidFilename(file2.getName());
            string = string == null ? string2 : string2 + "/" + string;
        }
        return new File(string);
    }

    @Nonnegative
    public static int getDirectoryObjectCount(@Nonnull File file) {
        ValueEnforcer.notNull((Object)file, (String)"Directory");
        ValueEnforcer.isTrue((boolean)file.isDirectory(), (String)("Passed object is not a directory: " + String.valueOf(file)));
        int n = 0;
        for (File file2 : FileHelper.getDirectoryContent(file)) {
            if (FilenameHelper.isSystemInternalDirectory(file2.getName())) continue;
            ++n;
        }
        return n;
    }

    @Nonnull
    @ReturnsMutableCopy
    private static ICommonsList<File> _getDirectoryContent(@Nonnull File file, @Nullable File[] fileArray) {
        if (fileArray == null) {
            if (!file.isDirectory()) {
                CONDLOG.warn("Cannot list non-directory: " + file.getAbsolutePath());
            } else if (!file.canExecute()) {
                CONDLOG.warn("Existing directory is missing the listing permission: " + file.getAbsolutePath());
            } else if (!file.canRead()) {
                CONDLOG.warn("Cannot list directory because of no read-rights: " + file.getAbsolutePath());
            } else {
                CONDLOG.warn("Directory listing failed because of IO error: " + file.getAbsolutePath());
            }
        } else if (!file.canExecute()) {
            CONDLOG.warn("Directory is missing the listing permission: " + file.getAbsolutePath());
        }
        return new CommonsArrayList((Object[])fileArray);
    }

    @Nonnull
    @ReturnsMutableCopy
    public static ICommonsList<File> getDirectoryContent(@Nonnull File file) {
        ValueEnforcer.notNull((Object)file, (String)"Directory");
        return FileHelper._getDirectoryContent(file, file.listFiles());
    }

    @Nonnull
    @ReturnsMutableCopy
    public static ICommonsList<File> getDirectoryContent(@Nonnull File file, @Nonnull FilenameFilter filenameFilter) {
        ValueEnforcer.notNull((Object)file, (String)"Directory");
        ValueEnforcer.notNull((Object)filenameFilter, (String)"FilenameFilter");
        return FileHelper._getDirectoryContent(file, file.listFiles(filenameFilter));
    }

    @Nonnull
    @ReturnsMutableCopy
    public static ICommonsList<File> getDirectoryContent(@Nonnull File file, @Nonnull FileFilter fileFilter) {
        ValueEnforcer.notNull((Object)file, (String)"Directory");
        ValueEnforcer.notNull((Object)fileFilter, (String)"FileFilter");
        return FileHelper._getDirectoryContent(file, file.listFiles(fileFilter));
    }

    @Nullable
    public static URL getAsURL(@Nonnull File file) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        try {
            return file.toURI().toURL();
        }
        catch (MalformedURLException malformedURLException) {
            CONDLOG.warn("Failed to convert file to URL: " + String.valueOf(file), (Exception)malformedURLException);
            return null;
        }
    }

    @Nullable
    public static String getAsURLString(@Nonnull File file) {
        ValueEnforcer.notNull((Object)file, (String)"File");
        try {
            return file.toURI().toURL().toExternalForm();
        }
        catch (MalformedURLException malformedURLException) {
            CONDLOG.warn("Failed to convert file to URL: " + String.valueOf(file), (Exception)malformedURLException);
            return null;
        }
    }
}

