/*
 * Copyright 2025 Hazelcast Inc.
 *
 * Licensed under the Hazelcast Community License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://hazelcast.com/hazelcast-community-license
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.sql.impl.expression;

import com.hazelcast.nio.ObjectDataInput;
import com.hazelcast.nio.ObjectDataOutput;

import java.io.IOException;
import java.util.Objects;

/**
 * Base class for expressions acting on two operands.
 */
public abstract class BiExpression<T> implements Expression<T> {

    protected Expression<?> operand1;
    protected Expression<?> operand2;

    protected BiExpression() {
        // No-op.
    }

    protected BiExpression(Expression<?> operand1, Expression<?> operand2) {
        this.operand1 = operand1;
        this.operand2 = operand2;
    }

    @Override
    public void writeData(ObjectDataOutput out) throws IOException {
        out.writeObject(operand1);
        out.writeObject(operand2);
    }

    @Override
    public void readData(ObjectDataInput in) throws IOException {
        operand1 = in.readObject();
        operand2 = in.readObject();
    }

    @Override
    public int hashCode() {
        return Objects.hash(operand1, operand2);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }

        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        BiExpression<?> that = (BiExpression<?>) o;

        return Objects.equals(operand1, that.operand1) && Objects.equals(operand2, that.operand2);
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + "{operand1=" + operand1 + ", operand2=" + operand2 + '}';
    }

    @Override
    public boolean isCooperative() {
        return (operand1 == null || operand1.isCooperative())
                && (operand2 == null || operand2.isCooperative());
    }
}
