/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.internal.cluster.impl;

import com.hazelcast.core.Member;
import com.hazelcast.instance.MemberImpl;
import com.hazelcast.instance.Node;
import com.hazelcast.instance.NodeState;
import com.hazelcast.internal.cluster.fd.ClusterFailureDetector;
import com.hazelcast.internal.cluster.fd.DeadlineClusterFailureDetector;
import com.hazelcast.internal.cluster.fd.PhiAccrualClusterFailureDetector;
import com.hazelcast.internal.cluster.impl.ClusterClockImpl;
import com.hazelcast.internal.cluster.impl.ClusterJoinManager;
import com.hazelcast.internal.cluster.impl.ClusterServiceImpl;
import com.hazelcast.internal.cluster.impl.MemberMap;
import com.hazelcast.internal.cluster.impl.MembersViewMetadata;
import com.hazelcast.internal.cluster.impl.MembershipManager;
import com.hazelcast.internal.cluster.impl.operations.ExplicitSuspicionOp;
import com.hazelcast.internal.cluster.impl.operations.HeartbeatComplaintOp;
import com.hazelcast.internal.cluster.impl.operations.HeartbeatOp;
import com.hazelcast.internal.cluster.impl.operations.MasterConfirmationOp;
import com.hazelcast.internal.metrics.Probe;
import com.hazelcast.logging.ILogger;
import com.hazelcast.nio.Address;
import com.hazelcast.nio.Connection;
import com.hazelcast.spi.impl.NodeEngineImpl;
import com.hazelcast.spi.impl.executionservice.InternalExecutionService;
import com.hazelcast.spi.impl.operationservice.InternalOperationService;
import com.hazelcast.spi.properties.GroupProperty;
import com.hazelcast.spi.properties.HazelcastProperties;
import com.hazelcast.util.Clock;
import com.hazelcast.util.EmptyStatement;
import com.hazelcast.util.StringUtil;
import java.net.ConnectException;
import java.util.Collection;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;

public class ClusterHeartbeatManager {
    private static final long CLOCK_JUMP_THRESHOLD = TimeUnit.MINUTES.toMillis(2L);
    private static final int HEART_BEAT_INTERVAL_FACTOR = 10;
    private static final int MAX_PING_RETRY_COUNT = 5;
    private final ILogger logger;
    private final Lock clusterServiceLock;
    private final Node node;
    private final NodeEngineImpl nodeEngine;
    private final ClusterServiceImpl clusterService;
    private final ClusterClockImpl clusterClock;
    private final ClusterFailureDetector heartbeatFailureDetector;
    private final ClusterFailureDetector masterConfirmationFailureDetector;
    private final long maxNoHeartbeatMillis;
    private final long maxNoMasterConfirmationMillis;
    private final long heartbeatIntervalMillis;
    private final long pingIntervalMillis;
    private final boolean icmpEnabled;
    private final int icmpTtl;
    private final int icmpTimeoutMillis;
    @Probe(name="lastHeartbeat")
    private volatile long lastHeartbeat;
    private volatile long lastClusterTimeDiff;

    ClusterHeartbeatManager(Node node, ClusterServiceImpl clusterService, Lock lock) {
        this.node = node;
        this.clusterService = clusterService;
        this.nodeEngine = node.getNodeEngine();
        this.clusterClock = clusterService.getClusterClock();
        this.logger = node.getLogger(this.getClass());
        this.clusterServiceLock = lock;
        HazelcastProperties hazelcastProperties = node.getProperties();
        this.maxNoHeartbeatMillis = hazelcastProperties.getMillis(GroupProperty.MAX_NO_HEARTBEAT_SECONDS);
        this.maxNoMasterConfirmationMillis = hazelcastProperties.getMillis(GroupProperty.MAX_NO_MASTER_CONFIRMATION_SECONDS);
        this.heartbeatIntervalMillis = ClusterHeartbeatManager.getHeartbeatInterval(hazelcastProperties);
        this.pingIntervalMillis = this.heartbeatIntervalMillis * 10L;
        this.icmpEnabled = hazelcastProperties.getBoolean(GroupProperty.ICMP_ENABLED);
        this.icmpTtl = hazelcastProperties.getInteger(GroupProperty.ICMP_TTL);
        this.icmpTimeoutMillis = (int)hazelcastProperties.getMillis(GroupProperty.ICMP_TIMEOUT);
        this.heartbeatFailureDetector = this.createHeartbeatFailureDetector(hazelcastProperties);
        this.masterConfirmationFailureDetector = new DeadlineClusterFailureDetector(this.maxNoMasterConfirmationMillis);
    }

    private ClusterFailureDetector createHeartbeatFailureDetector(HazelcastProperties properties) {
        String type = properties.getString(GroupProperty.HEARTBEAT_FAILURE_DETECTOR_TYPE);
        if ("deadline".equals(type)) {
            return new DeadlineClusterFailureDetector(this.maxNoHeartbeatMillis);
        }
        if ("phi-accrual".equals(type)) {
            int defaultValue = Integer.parseInt(GroupProperty.MAX_NO_HEARTBEAT_SECONDS.getDefaultValue());
            if (this.maxNoHeartbeatMillis == TimeUnit.SECONDS.toMillis(defaultValue)) {
                this.logger.warning("When using Phi-Accrual Failure Detector, please consider using a lower '" + GroupProperty.MAX_NO_HEARTBEAT_SECONDS.getName() + "' value. Current is: " + defaultValue + " seconds.");
            }
            return new PhiAccrualClusterFailureDetector(this.maxNoHeartbeatMillis, this.heartbeatIntervalMillis, properties);
        }
        throw new IllegalArgumentException("Unknown failure detector type: " + type);
    }

    public long getHeartbeatIntervalMillis() {
        return this.heartbeatIntervalMillis;
    }

    public long getLastHeartbeatTime(Member member) {
        return this.heartbeatFailureDetector.lastHeartbeat(member);
    }

    private static long getHeartbeatInterval(HazelcastProperties hazelcastProperties) {
        long heartbeatInterval = hazelcastProperties.getMillis(GroupProperty.HEARTBEAT_INTERVAL_SECONDS);
        return heartbeatInterval > 0L ? heartbeatInterval : TimeUnit.SECONDS.toMillis(1L);
    }

    void init() {
        InternalExecutionService executionService = this.nodeEngine.getExecutionService();
        HazelcastProperties hazelcastProperties = this.node.getProperties();
        executionService.scheduleWithRepetition("hz:cluster", new Runnable(){

            @Override
            public void run() {
                ClusterHeartbeatManager.this.heartbeat();
            }
        }, this.heartbeatIntervalMillis, this.heartbeatIntervalMillis, TimeUnit.MILLISECONDS);
        long masterConfirmationInterval = hazelcastProperties.getSeconds(GroupProperty.MASTER_CONFIRMATION_INTERVAL_SECONDS);
        masterConfirmationInterval = masterConfirmationInterval > 0L ? masterConfirmationInterval : 1L;
        executionService.scheduleWithRepetition("hz:cluster", new Runnable(){

            @Override
            public void run() {
                ClusterHeartbeatManager.this.sendMasterConfirmation();
            }
        }, masterConfirmationInterval, masterConfirmationInterval, TimeUnit.SECONDS);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void handleHeartbeat(MembersViewMetadata senderMembersViewMetadata, String receiverUuid, long timestamp) {
        Address senderAddress = senderMembersViewMetadata.getMemberAddress();
        try {
            long timeout = Math.min(TimeUnit.SECONDS.toMillis(1L), this.heartbeatIntervalMillis / 2L);
            if (!this.clusterServiceLock.tryLock(timeout, TimeUnit.MILLISECONDS)) {
                this.logger.warning("Cannot handle heartbeat from " + senderAddress + ", could not acquire lock in time.");
                return;
            }
        }
        catch (InterruptedException e) {
            this.logger.warning("Cannot handle heartbeat from " + senderAddress + ", thread interrupted.");
            Thread.currentThread().interrupt();
            return;
        }
        try {
            if (!this.clusterService.isJoined()) {
                if (this.clusterService.getThisUuid().equals(receiverUuid)) {
                    this.logger.fine("Ignoring heartbeat of sender: " + senderMembersViewMetadata + ", because node is not joined!");
                } else {
                    this.logger.fine("Sending explicit suspicion to " + senderAddress + " for heartbeat " + senderMembersViewMetadata + ", because this node has received an invalid heartbeat before it joins to the cluster");
                    InternalOperationService operationService = this.nodeEngine.getOperationService();
                    ExplicitSuspicionOp op = new ExplicitSuspicionOp(senderMembersViewMetadata);
                    operationService.send(op, senderAddress);
                }
                return;
            }
            MembershipManager membershipManager = this.clusterService.getMembershipManager();
            MemberImpl member = membershipManager.getMember(senderAddress, senderMembersViewMetadata.getMemberUuid());
            if (member != null) {
                if (this.clusterService.getThisUuid().equals(receiverUuid)) {
                    this.onHeartbeat(member, timestamp);
                    return;
                }
                this.logger.warning("Local uuid mismatch on received heartbeat. local uuid: " + this.clusterService.getThisUuid() + " received uuid: " + receiverUuid + " with " + senderMembersViewMetadata);
            }
            this.onInvalidHeartbeat(senderMembersViewMetadata);
        }
        finally {
            this.clusterServiceLock.unlock();
        }
    }

    private void onInvalidHeartbeat(MembersViewMetadata senderMembersViewMetadata) {
        Address senderAddress = senderMembersViewMetadata.getMemberAddress();
        if (this.clusterService.isMaster()) {
            if (!this.clusterService.getClusterJoinManager().isMastershipClaimInProgress()) {
                this.logger.fine("Sending explicit suspicion to " + senderAddress + " for heartbeat " + senderMembersViewMetadata + ", because it is not a member of this cluster" + " or its heartbeat cannot be validated!");
                this.clusterService.sendExplicitSuspicion(senderMembersViewMetadata);
            }
        } else {
            Address masterAddress = this.clusterService.getMasterAddress();
            if (this.clusterService.getMembershipManager().isMemberSuspected(masterAddress)) {
                this.logger.fine("Not sending heartbeat complaint for " + senderMembersViewMetadata + " to suspected master: " + masterAddress);
                return;
            }
            this.logger.fine("Sending heartbeat complaint to master " + masterAddress + " for heartbeat " + senderMembersViewMetadata + ", because it is not a member of this cluster" + " or its heartbeat cannot be validated!");
            this.sendHeartbeatComplaintToMaster(senderMembersViewMetadata);
        }
    }

    private void sendHeartbeatComplaintToMaster(MembersViewMetadata senderMembersViewMetadata) {
        if (this.clusterService.isMaster()) {
            this.logger.warning("Cannot send heartbeat complaint for " + senderMembersViewMetadata + " to itself.");
            return;
        }
        Address masterAddress = this.clusterService.getMasterAddress();
        if (masterAddress == null) {
            this.logger.fine("Cannot send heartbeat complaint for " + senderMembersViewMetadata.getMemberAddress() + ", master address is not set.");
            return;
        }
        MembersViewMetadata localMembersViewMetadata = this.clusterService.getMembershipManager().createLocalMembersViewMetadata();
        HeartbeatComplaintOp op = new HeartbeatComplaintOp(localMembersViewMetadata, senderMembersViewMetadata);
        InternalOperationService operationService = this.nodeEngine.getOperationService();
        operationService.send(op, masterAddress);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void handleHeartbeatComplaint(MembersViewMetadata receiverMVMetadata, MembersViewMetadata senderMVMetadata) {
        this.clusterServiceLock.lock();
        try {
            if (!this.clusterService.isJoined()) {
                this.logger.warning("Ignoring heartbeat complaint of receiver: " + receiverMVMetadata + " and sender: " + senderMVMetadata + " because not joined!");
                return;
            }
            MembershipManager membershipManager = this.clusterService.getMembershipManager();
            ClusterJoinManager clusterJoinManager = this.clusterService.getClusterJoinManager();
            if (!this.clusterService.isMaster()) {
                this.logger.warning("Ignoring heartbeat complaint of receiver: " + receiverMVMetadata + " for sender: " + senderMVMetadata + " because this node is not master");
                return;
            }
            if (clusterJoinManager.isMastershipClaimInProgress()) {
                this.logger.fine("Ignoring heartbeat complaint of receiver: " + receiverMVMetadata + " for sender: " + senderMVMetadata + " because mastership claim process is ongoing");
                return;
            }
            if (senderMVMetadata.getMemberAddress().equals(receiverMVMetadata.getMemberAddress())) {
                this.logger.warning("Ignoring heartbeat complaint of receiver: " + receiverMVMetadata + " for sender: " + senderMVMetadata + " because they are same member");
                return;
            }
            if (membershipManager.validateMembersViewMetadata(senderMVMetadata)) {
                if (membershipManager.validateMembersViewMetadata(receiverMVMetadata)) {
                    this.logger.fine("Sending latest member list to " + senderMVMetadata.getMemberAddress() + " and " + receiverMVMetadata.getMemberAddress() + " after heartbeat complaint.");
                    membershipManager.sendMemberListToMember(senderMVMetadata.getMemberAddress());
                    membershipManager.sendMemberListToMember(receiverMVMetadata.getMemberAddress());
                } else {
                    this.logger.fine("Complainer " + receiverMVMetadata.getMemberAddress() + " will explicitly suspect from " + this.node.getThisAddress() + " and " + senderMVMetadata.getMemberAddress());
                    this.clusterService.sendExplicitSuspicion(receiverMVMetadata);
                    this.clusterService.sendExplicitSuspicionTrigger(senderMVMetadata.getMemberAddress(), receiverMVMetadata);
                }
            } else if (membershipManager.validateMembersViewMetadata(receiverMVMetadata)) {
                this.logger.fine("Complainee " + senderMVMetadata.getMemberAddress() + " will explicitly suspect from " + this.node.getThisAddress() + " and " + receiverMVMetadata.getMemberAddress());
                this.clusterService.sendExplicitSuspicion(senderMVMetadata);
                this.clusterService.sendExplicitSuspicionTrigger(receiverMVMetadata.getMemberAddress(), senderMVMetadata);
            } else {
                this.logger.fine("Both complainer " + receiverMVMetadata.getMemberAddress() + " and complainee " + senderMVMetadata.getMemberAddress() + " will explicitly suspect from " + this.node.getThisAddress());
                this.clusterService.sendExplicitSuspicion(senderMVMetadata);
                this.clusterService.sendExplicitSuspicion(receiverMVMetadata);
            }
        }
        finally {
            this.clusterServiceLock.unlock();
        }
    }

    public void onHeartbeat(MemberImpl member, long timestamp) {
        if (member != null) {
            long clusterTime = this.clusterClock.getClusterTime();
            if (this.logger.isFineEnabled()) {
                this.logger.fine(String.format("Received heartbeat from %s (now: %s, timestamp: %s)", member, StringUtil.timeToString(clusterTime), StringUtil.timeToString(timestamp)));
            }
            if (clusterTime - timestamp > this.maxNoHeartbeatMillis / 2L) {
                this.logger.warning(String.format("Ignoring heartbeat from %s since it is expired (now: %s, timestamp: %s)", member, StringUtil.timeToString(clusterTime), StringUtil.timeToString(timestamp)));
                return;
            }
            if (this.isMaster(member)) {
                this.clusterClock.setMasterTime(timestamp);
            }
            this.heartbeatFailureDetector.heartbeat(member, this.clusterClock.getClusterTime());
            MembershipManager membershipManager = this.clusterService.getMembershipManager();
            membershipManager.clearMemberSuspicion(member.getAddress(), "Valid heartbeat");
        }
    }

    void acceptMasterConfirmation(MemberImpl member, long timestamp) {
        if (member != null) {
            long clusterTime;
            if (this.logger.isFineEnabled()) {
                this.logger.fine("MasterConfirmation has been received from " + member);
            }
            if ((clusterTime = this.clusterClock.getClusterTime()) - timestamp > this.maxNoMasterConfirmationMillis / 2L) {
                this.logger.warning(String.format("Ignoring master confirmation from %s, since it is expired (now: %s, timestamp: %s)", member, StringUtil.timeToString(clusterTime), StringUtil.timeToString(timestamp)));
                return;
            }
            this.masterConfirmationFailureDetector.heartbeat(member, clusterTime);
        }
    }

    void heartbeat() {
        if (!this.clusterService.isJoined()) {
            return;
        }
        this.checkClockDrift(this.heartbeatIntervalMillis);
        long clusterTime = this.clusterClock.getClusterTime();
        if (this.clusterService.isMaster()) {
            this.heartbeatWhenMaster(clusterTime);
        } else {
            this.heartbeatWhenSlave(clusterTime);
        }
    }

    private void checkClockDrift(long intervalMillis) {
        long now = Clock.currentTimeMillis();
        if (this.lastHeartbeat != 0L) {
            long clockJump = now - this.lastHeartbeat - intervalMillis;
            long absoluteClockJump = Math.abs(clockJump);
            if (absoluteClockJump > CLOCK_JUMP_THRESHOLD) {
                this.logger.info(String.format("System clock apparently jumped from %s to %s since last heartbeat (%+d ms)", StringUtil.timeToString(this.lastHeartbeat), StringUtil.timeToString(now), clockJump));
                long currentClusterTimeDiff = this.clusterClock.getClusterTimeDiff();
                if (Math.abs(this.lastClusterTimeDiff - currentClusterTimeDiff) < CLOCK_JUMP_THRESHOLD) {
                    this.clusterClock.setClusterTimeDiff(currentClusterTimeDiff - clockJump);
                }
            }
            if (absoluteClockJump >= this.maxNoMasterConfirmationMillis / 2L) {
                this.logger.warning(String.format("Resetting master confirmation timestamps because of huge system clock jump! Clock-Jump: %d ms, Master-Confirmation-Timeout: %d ms", clockJump, this.maxNoMasterConfirmationMillis));
                this.resetMemberMasterConfirmations();
            }
            if (absoluteClockJump >= this.maxNoHeartbeatMillis / 2L) {
                this.logger.warning(String.format("Resetting heartbeat timestamps because of huge system clock jump! Clock-Jump: %d ms, Heartbeat-Timeout: %d ms", clockJump, this.maxNoHeartbeatMillis));
                this.resetHeartbeats();
            }
        }
        this.lastClusterTimeDiff = this.clusterClock.getClusterTimeDiff();
        this.lastHeartbeat = now;
    }

    private void heartbeatWhenMaster(long now) {
        Collection<MemberImpl> members = this.clusterService.getMemberImpls();
        for (MemberImpl member : members) {
            if (member.localMember()) continue;
            try {
                this.logIfConnectionToEndpointIsMissing(now, member);
                if (this.suspectMemberIfNotHeartBeating(now, member) || this.removeMemberIfMasterConfirmationExpired(now, member)) continue;
                this.pingMemberIfRequired(now, member);
                this.sendHeartbeat(member);
            }
            catch (Throwable e) {
                this.logger.severe(e);
            }
        }
    }

    private boolean suspectMemberIfNotHeartBeating(long now, MemberImpl member) {
        if (this.clusterService.getMembershipManager().isMemberSuspected(member.getAddress())) {
            return true;
        }
        long lastHeartbeat = this.heartbeatFailureDetector.lastHeartbeat(member);
        if (!this.heartbeatFailureDetector.isAlive(member, now)) {
            double suspicionLevel = this.heartbeatFailureDetector.suspicionLevel(member, now);
            String reason = String.format("Suspecting %s because it has not sent any heartbeats since %s. Now: %s, heartbeat timeout: %d ms, suspicion level: %.2f", member, StringUtil.timeToString(lastHeartbeat), StringUtil.timeToString(now), this.maxNoHeartbeatMillis, suspicionLevel);
            this.logger.warning(reason);
            this.clusterService.suspectMember(member, reason, true);
            return true;
        }
        if (this.logger.isFineEnabled() && now - lastHeartbeat > this.heartbeatIntervalMillis * 10L) {
            double suspicionLevel = this.heartbeatFailureDetector.suspicionLevel(member, now);
            this.logger.fine(String.format("Not receiving any heartbeats from %s since %s, suspicion level: %.2f", member, StringUtil.timeToString(lastHeartbeat), suspicionLevel));
        }
        return false;
    }

    private boolean removeMemberIfMasterConfirmationExpired(long now, MemberImpl member) {
        if (this.clusterService.getClusterJoinManager().isMastershipClaimInProgress()) {
            return false;
        }
        if (!this.masterConfirmationFailureDetector.isAlive(member, now)) {
            long lastConfirmation = this.masterConfirmationFailureDetector.lastHeartbeat(member);
            String reason = String.format("Removing %s because it has not sent any master confirmation for %d ms.  Clock time: %s. Cluster time: %s. Last confirmation time was %s.", member, this.maxNoMasterConfirmationMillis, StringUtil.timeToString(Clock.currentTimeMillis()), StringUtil.timeToString(now), StringUtil.timeToString(lastConfirmation));
            this.logger.warning(reason);
            this.clusterService.suspectMember(member, reason, true);
            return true;
        }
        return false;
    }

    private void heartbeatWhenSlave(long now) {
        MembershipManager membershipManager = this.clusterService.getMembershipManager();
        Collection<MemberImpl> members = this.clusterService.getMemberImpls();
        for (MemberImpl member : members) {
            if (member.localMember()) continue;
            try {
                this.logIfConnectionToEndpointIsMissing(now, member);
                if (this.suspectMemberIfNotHeartBeating(now, member) || membershipManager.isMemberSuspected(member.getAddress())) continue;
                this.pingMemberIfRequired(now, member);
                this.sendHeartbeat(member);
            }
            catch (Throwable e) {
                this.logger.severe(e);
            }
        }
    }

    private boolean isMaster(MemberImpl member) {
        return member.getAddress().equals(this.clusterService.getMasterAddress());
    }

    private void pingMemberIfRequired(long now, MemberImpl member) {
        if (!this.icmpEnabled) {
            return;
        }
        long lastHeartbeat = this.heartbeatFailureDetector.lastHeartbeat(member);
        if (now - lastHeartbeat >= this.pingIntervalMillis) {
            this.ping(member);
        }
    }

    private void ping(final Member member) {
        this.nodeEngine.getExecutionService().execute("hz:system", new Runnable(){

            @Override
            public void run() {
                try {
                    Address address = member.getAddress();
                    ClusterHeartbeatManager.this.logger.warning(String.format("%s will ping %s", ClusterHeartbeatManager.this.node.getThisAddress(), address));
                    for (int i = 0; i < 5; ++i) {
                        try {
                            if (!address.getInetAddress().isReachable(null, ClusterHeartbeatManager.this.icmpTtl, ClusterHeartbeatManager.this.icmpTimeoutMillis)) continue;
                            ClusterHeartbeatManager.this.logger.info(String.format("%s pinged %s successfully", ClusterHeartbeatManager.this.node.getThisAddress(), address));
                            return;
                        }
                        catch (ConnectException ignored) {
                            EmptyStatement.ignore(ignored);
                        }
                    }
                    String reason = String.format("%s could not ping %s", ClusterHeartbeatManager.this.node.getThisAddress(), address);
                    ClusterHeartbeatManager.this.logger.warning(reason);
                    ClusterHeartbeatManager.this.clusterService.suspectMember(member, reason, true);
                }
                catch (Throwable ignored) {
                    EmptyStatement.ignore(ignored);
                }
            }
        });
    }

    private void sendHeartbeat(Member target) {
        block3: {
            if (target == null) {
                return;
            }
            try {
                MembersViewMetadata membersViewMetadata = this.clusterService.getMembershipManager().createLocalMembersViewMetadata();
                HeartbeatOp op = new HeartbeatOp(membersViewMetadata, target.getUuid(), this.clusterClock.getClusterTime());
                op.setCallerUuid(this.clusterService.getThisUuid());
                this.node.nodeEngine.getOperationService().send(op, target.getAddress());
            }
            catch (Exception e) {
                if (!this.logger.isFineEnabled()) break block3;
                this.logger.fine(String.format("Error while sending heartbeat -> %s[%s]", e.getClass().getName(), e.getMessage()));
            }
        }
    }

    private void logIfConnectionToEndpointIsMissing(long now, MemberImpl member) {
        Connection conn;
        long heartbeatTime = this.heartbeatFailureDetector.lastHeartbeat(member);
        if (!(now - heartbeatTime < this.heartbeatIntervalMillis * 10L || (conn = this.node.connectionManager.getOrConnect(member.getAddress())) != null && conn.isAlive())) {
            this.logger.warning("This node does not have a connection to " + member);
        }
    }

    public void sendMasterConfirmation() {
        if (!this.clusterService.isJoined() || this.node.getState() == NodeState.SHUT_DOWN || this.clusterService.isMaster()) {
            return;
        }
        Address masterAddress = this.clusterService.getMasterAddress();
        if (masterAddress == null) {
            this.logger.fine("Could not send MasterConfirmation, master address is null!");
            return;
        }
        MembershipManager membershipManager = this.clusterService.getMembershipManager();
        MemberMap memberMap = membershipManager.getMemberMap();
        MemberImpl masterMember = memberMap.getMember(masterAddress);
        if (masterMember == null) {
            this.logger.fine("Could not send MasterConfirmation, master member is null! master address: " + masterAddress);
            return;
        }
        if (membershipManager.isMemberSuspected(masterAddress)) {
            this.logger.fine("Not sending MasterConfirmation to " + masterMember + ", since it's suspected.");
            return;
        }
        if (this.logger.isFineEnabled()) {
            this.logger.fine("Sending MasterConfirmation to " + masterMember);
        }
        MembersViewMetadata membersViewMetadata = membershipManager.createLocalMembersViewMetadata();
        MasterConfirmationOp op = new MasterConfirmationOp(membersViewMetadata, this.clusterClock.getClusterTime());
        this.nodeEngine.getOperationService().send(op, masterAddress);
    }

    void resetMemberMasterConfirmations() {
        long now = this.clusterClock.getClusterTime();
        for (MemberImpl member : this.clusterService.getMemberImpls()) {
            this.masterConfirmationFailureDetector.heartbeat(member, now);
        }
    }

    private void resetHeartbeats() {
        long now = this.clusterClock.getClusterTime();
        for (MemberImpl member : this.clusterService.getMemberImpls()) {
            this.heartbeatFailureDetector.heartbeat(member, now);
        }
    }

    void removeMember(MemberImpl member) {
        this.masterConfirmationFailureDetector.remove(member);
        this.heartbeatFailureDetector.remove(member);
    }

    void reset() {
        this.masterConfirmationFailureDetector.reset();
        this.heartbeatFailureDetector.reset();
    }
}

