/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.map.eviction;

import com.hazelcast.config.MapConfig;
import com.hazelcast.config.MaxSizeConfig;
import com.hazelcast.core.EntryEventType;
import com.hazelcast.map.MapContainer;
import com.hazelcast.map.MapEventPublisher;
import com.hazelcast.map.MapService;
import com.hazelcast.map.MapServiceContext;
import com.hazelcast.map.NearCacheProvider;
import com.hazelcast.map.PartitionContainer;
import com.hazelcast.map.RecordStore;
import com.hazelcast.map.record.Record;
import com.hazelcast.nio.Address;
import com.hazelcast.nio.serialization.Data;
import com.hazelcast.partition.InternalPartition;
import com.hazelcast.partition.InternalPartitionService;
import com.hazelcast.spi.NodeEngine;
import java.util.Arrays;
import java.util.Iterator;

public final class EvictionHelper {
    private static final int ONE_HUNDRED_PERCENT = 100;
    private static final int EVICTION_START_THRESHOLD_PERCENTAGE = 95;
    private static final int ONE_KILOBYTE = 1024;

    private EvictionHelper() {
    }

    public static boolean checkEvictable(MapContainer mapContainer) {
        boolean result;
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        MaxSizeConfig.MaxSizePolicy maxSizePolicy = maxSizeConfig.getMaxSizePolicy();
        switch (maxSizePolicy) {
            case PER_NODE: {
                result = EvictionHelper.isEvictablePerNode(mapContainer);
                break;
            }
            case PER_PARTITION: {
                result = EvictionHelper.isEvictablePerPartition(mapContainer);
                break;
            }
            case USED_HEAP_PERCENTAGE: {
                result = EvictionHelper.isEvictableHeapPercentage(mapContainer);
                break;
            }
            case USED_HEAP_SIZE: {
                result = EvictionHelper.isEvictableHeapSize(mapContainer);
                break;
            }
            default: {
                throw new IllegalArgumentException("Not an appropriate max size policy [" + (Object)((Object)maxSizePolicy) + ']');
            }
        }
        return result;
    }

    public static void removeEvictableRecords(RecordStore recordStore, int evictableSize, MapConfig mapConfig, MapServiceContext mapServiceContext) {
        MapConfig.EvictionPolicy evictionPolicy = mapConfig.getEvictionPolicy();
        long[] criterias = EvictionHelper.createAndPopulateEvictionCriteriaArray(recordStore, evictionPolicy);
        if (criterias == null) {
            return;
        }
        Arrays.sort(criterias);
        int evictableBaseIndex = EvictionHelper.getEvictionStartIndex(criterias, evictableSize);
        long criteriaValue = criterias[evictableBaseIndex];
        int evictedRecordCounter = 0;
        Iterator<Record> iterator = recordStore.iterator();
        while (iterator.hasNext()) {
            Record record = iterator.next();
            long value = EvictionHelper.getEvictionCriteriaValue(record, evictionPolicy);
            if (value <= criteriaValue) {
                Data tmpKey = record.getKey();
                Object tmpValue = record.getValue();
                if (EvictionHelper.evictIfNotLocked(tmpKey, recordStore)) {
                    ++evictedRecordCounter;
                    String mapName = mapConfig.getName();
                    EvictionHelper.interceptAndInvalidate(mapServiceContext, value, tmpKey, mapName);
                    EvictionHelper.fireEvent(tmpKey, tmpValue, mapName, mapServiceContext);
                }
            }
            if (evictedRecordCounter < evictableSize) continue;
            break;
        }
    }

    private static long[] createAndPopulateEvictionCriteriaArray(RecordStore recordStore, MapConfig.EvictionPolicy evictionPolicy) {
        int size = recordStore.size();
        long[] criterias = null;
        int index = 0;
        Iterator<Record> iterator = recordStore.iterator();
        while (iterator.hasNext()) {
            Record record = iterator.next();
            if (criterias == null) {
                criterias = new long[size];
            }
            criterias[index] = EvictionHelper.getEvictionCriteriaValue(record, evictionPolicy);
            if (++index != size) continue;
            break;
        }
        if (criterias == null) {
            return null;
        }
        if (index < criterias.length) {
            for (int i = index; i < criterias.length; ++i) {
                criterias[i] = Long.MAX_VALUE;
            }
        }
        return criterias;
    }

    private static int getEvictionStartIndex(long[] criterias, int evictableSize) {
        int length = criterias.length;
        int sizeToEvict = Math.min(evictableSize, length);
        int index = sizeToEvict - 1;
        return index < 0 ? 0 : index;
    }

    private static void interceptAndInvalidate(MapServiceContext mapServiceContext, long value, Data tmpKey, String mapName) {
        mapServiceContext.interceptAfterRemove(mapName, value);
        NearCacheProvider nearCacheProvider = mapServiceContext.getNearCacheProvider();
        if (nearCacheProvider.isNearCacheAndInvalidationEnabled(mapName)) {
            nearCacheProvider.invalidateAllNearCaches(mapName, tmpKey);
        }
    }

    public static void fireEvent(Data key, Object value, String mapName, MapServiceContext mapServiceContext) {
        MapEventPublisher mapEventPublisher = mapServiceContext.getMapEventPublisher();
        NodeEngine nodeEngine = mapServiceContext.getNodeEngine();
        Address thisAddress = nodeEngine.getThisAddress();
        Data dataValue = mapServiceContext.toData(value);
        mapEventPublisher.publishEvent(thisAddress, mapName, EntryEventType.EVICTED, key, dataValue, null);
    }

    public static boolean evictIfNotLocked(Data key, RecordStore recordStore) {
        if (recordStore.isLocked(key)) {
            return false;
        }
        recordStore.evict(key);
        return true;
    }

    public static int getEvictableSize(int currentPartitionSize, MapConfig mapConfig, MapServiceContext mapServiceContext) {
        int evictableSize;
        MaxSizeConfig.MaxSizePolicy maxSizePolicy = mapConfig.getMaxSizeConfig().getMaxSizePolicy();
        int evictionPercentage = mapConfig.getEvictionPercentage();
        switch (maxSizePolicy) {
            case PER_PARTITION: {
                int maxSize = mapConfig.getMaxSizeConfig().getSize();
                int targetSizePerPartition = Double.valueOf((double)maxSize * ((double)(100 - evictionPercentage) / 100.0)).intValue();
                int diffFromTargetSize = currentPartitionSize - targetSizePerPartition;
                int prunedSize = currentPartitionSize * evictionPercentage / 100 + 1;
                evictableSize = Math.max(diffFromTargetSize, prunedSize);
                break;
            }
            case PER_NODE: {
                int maxSize = mapConfig.getMaxSizeConfig().getSize();
                int memberCount = mapServiceContext.getNodeEngine().getClusterService().getMembers().size();
                int maxPartitionSize = maxSize * memberCount / mapServiceContext.getNodeEngine().getPartitionService().getPartitionCount();
                int targetSizePerPartition = Double.valueOf((double)maxPartitionSize * ((double)(100 - evictionPercentage) / 100.0)).intValue();
                int diffFromTargetSize = currentPartitionSize - targetSizePerPartition;
                int prunedSize = currentPartitionSize * evictionPercentage / 100 + 1;
                evictableSize = Math.max(diffFromTargetSize, prunedSize);
                break;
            }
            case USED_HEAP_PERCENTAGE: 
            case USED_HEAP_SIZE: {
                evictableSize = currentPartitionSize * evictionPercentage / 100;
                break;
            }
            default: {
                throw new IllegalArgumentException("Max size policy is not defined [" + (Object)((Object)maxSizePolicy) + "]");
            }
        }
        return evictableSize;
    }

    private static long getEvictionCriteriaValue(Record record, MapConfig.EvictionPolicy evictionPolicy) {
        long value;
        switch (evictionPolicy) {
            case LRU: 
            case LFU: {
                value = record.getEvictionCriteriaNumber();
                break;
            }
            default: {
                throw new IllegalArgumentException("Not an appropriate eviction policy [" + (Object)((Object)evictionPolicy) + ']');
            }
        }
        return value;
    }

    private static boolean isEvictablePerNode(MapContainer mapContainer) {
        int nodeTotalSize = 0;
        MapService mapService = mapContainer.getMapServiceContext().getService();
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        int maxSize = EvictionHelper.getApproximateMaxSize(maxSizeConfig.getSize());
        String mapName = mapContainer.getName();
        NodeEngine nodeEngine = mapService.getMapServiceContext().getNodeEngine();
        InternalPartitionService partitionService = nodeEngine.getPartitionService();
        int partitionCount = partitionService.getPartitionCount();
        for (int i = 0; i < partitionCount; ++i) {
            Address owner = partitionService.getPartitionOwner(i);
            if (!nodeEngine.getThisAddress().equals(owner)) continue;
            PartitionContainer container = mapService.getMapServiceContext().getPartitionContainer(i);
            if (container == null) {
                return false;
            }
            if ((nodeTotalSize += EvictionHelper.getRecordStoreSize(mapName, container)) < maxSize) continue;
            return true;
        }
        return false;
    }

    private static int getRecordStoreSize(String mapName, PartitionContainer partitionContainer) {
        RecordStore existingRecordStore = partitionContainer.getExistingRecordStore(mapName);
        if (existingRecordStore == null) {
            return 0;
        }
        return existingRecordStore.size();
    }

    private static long getRecordStoreHeapCost(String mapName, PartitionContainer partitionContainer) {
        RecordStore existingRecordStore = partitionContainer.getExistingRecordStore(mapName);
        if (existingRecordStore == null) {
            return 0L;
        }
        return existingRecordStore.getHeapCost();
    }

    private static int getApproximateMaxSize(int maxSizeFromConfig) {
        return maxSizeFromConfig * 95 / 100;
    }

    private static boolean isEvictablePerPartition(MapContainer mapContainer) {
        MapService mapService = mapContainer.getMapServiceContext().getService();
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        int maxSize = EvictionHelper.getApproximateMaxSize(maxSizeConfig.getSize());
        String mapName = mapContainer.getName();
        NodeEngine nodeEngine = mapService.getMapServiceContext().getNodeEngine();
        InternalPartitionService partitionService = nodeEngine.getPartitionService();
        for (int i = 0; i < partitionService.getPartitionCount(); ++i) {
            Address owner = partitionService.getPartitionOwner(i);
            if (!nodeEngine.getThisAddress().equals(owner)) continue;
            PartitionContainer container = mapService.getMapServiceContext().getPartitionContainer(i);
            if (container == null) {
                return false;
            }
            int size = EvictionHelper.getRecordStoreSize(mapName, container);
            if (size < maxSize) continue;
            return true;
        }
        return false;
    }

    private static boolean isEvictableHeapSize(MapContainer mapContainer) {
        long usedHeapSize = EvictionHelper.getUsedHeapSize(mapContainer);
        if (usedHeapSize == -1L) {
            return false;
        }
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        int maxSize = EvictionHelper.getApproximateMaxSize(maxSizeConfig.getSize());
        return (long)maxSize < usedHeapSize / 1024L / 1024L;
    }

    private static boolean isEvictableHeapPercentage(MapContainer mapContainer) {
        long total;
        long usedHeapSize = EvictionHelper.getUsedHeapSize(mapContainer);
        if (usedHeapSize == -1L) {
            return false;
        }
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        int maxSize = EvictionHelper.getApproximateMaxSize(maxSizeConfig.getSize());
        return (double)maxSize < 100.0 * (double)usedHeapSize / (double)(total = Runtime.getRuntime().totalMemory());
    }

    private static long getUsedHeapSize(MapContainer mapContainer) {
        long heapCost = 0L;
        MapService mapService = mapContainer.getMapServiceContext().getService();
        String mapName = mapContainer.getName();
        NodeEngine nodeEngine = mapService.getMapServiceContext().getNodeEngine();
        Address thisAddress = nodeEngine.getThisAddress();
        for (int i = 0; i < nodeEngine.getPartitionService().getPartitionCount(); ++i) {
            InternalPartition partition = nodeEngine.getPartitionService().getPartition(i, false);
            if (!partition.isOwnerOrBackup(thisAddress)) continue;
            PartitionContainer container = mapService.getMapServiceContext().getPartitionContainer(i);
            if (container == null) {
                return -1L;
            }
            heapCost += EvictionHelper.getRecordStoreHeapCost(mapName, container);
        }
        return heapCost += mapContainer.getNearCacheSizeEstimator().getSize();
    }
}

