/*
 * Copyright (c) 2008-2013, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.core;

import com.hazelcast.monitor.LocalAtomicNumberStats;

/**
 * AtomicNumber is a backed-up distributed implementation of
 * {@link java.util.concurrent.atomic.AtomicLong java.util.concurrent.atomic.AtomicLong}.
 */
public interface AtomicNumber extends Instance {
    /**
     * Returns the name of this IAtomicLong instance.
     *
     * @return name of this instance
     */
    public String getName();

    /**
     * Atomically adds the given value to the current value.
     *
     * @param delta the value to add
     * @return the updated value
     */
    public long addAndGet(long delta);

    /**
     * Atomically sets the value to the given updated value
     * only if the current value {@code ==} the expected value.
     *
     * @param expect the expected value
     * @param update the new value
     * @return true if successful; or false if the actual value
     *         was not equal to the expected value.
     */
    public boolean compareAndSet(long expect, long update);

    /**
     * Atomically decrements the current value by one.
     *
     * @return the updated value
     */
    public long decrementAndGet();

    /**
     * Gets the current value.
     *
     * @return the current value
     */
    public long get();

    /**
     * Atomically adds the given value to the current value.
     *
     * @param delta the value to add
     * @return the old value before the add
     */
    public long getAndAdd(long delta);

    /**
     * Atomically sets the given value and returns the old value.
     *
     * @param newValue the new value
     * @return the old value
     */
    public long getAndSet(long newValue);

    /**
     * Atomically increments the current value by one.
     *
     * @return the updated value
     */
    public long incrementAndGet();

    /**
     * Atomically sets the given value.
     *
     * @param newValue the new value
     */
    public void set(long newValue);

    /**
     * Not implemented. Use compareAndSet().
     */
    @Deprecated
    boolean weakCompareAndSet(long expect, long update);

    @Deprecated
    /**
     * Not implemented. Use set().
     */
    void lazySet(long newValue);

    LocalAtomicNumberStats getLocalAtomicNumberStats();
}