/*
 * Decompiled with CFR 0.152.
 */
package com.google.maps.android.heatmaps;

import android.graphics.Bitmap;
import android.graphics.Color;
import android.support.v4.util.LongSparseArray;
import com.google.android.gms.maps.model.LatLng;
import com.google.android.gms.maps.model.Tile;
import com.google.android.gms.maps.model.TileProvider;
import com.google.maps.android.geometry.Bounds;
import com.google.maps.android.geometry.Point;
import com.google.maps.android.heatmaps.Gradient;
import com.google.maps.android.heatmaps.WeightedLatLng;
import com.google.maps.android.quadtree.PointQuadTree;
import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

public class HeatmapTileProvider
implements TileProvider {
    public static final int DEFAULT_RADIUS = 20;
    public static final double DEFAULT_OPACITY = 0.7;
    private static final int[] DEFAULT_GRADIENT_COLORS = new int[]{Color.rgb((int)102, (int)225, (int)0), Color.rgb((int)255, (int)0, (int)0)};
    private static final float[] DEFAULT_GRADIENT_START_POINTS = new float[]{0.2f, 1.0f};
    public static final Gradient DEFAULT_GRADIENT = new Gradient(DEFAULT_GRADIENT_COLORS, DEFAULT_GRADIENT_START_POINTS);
    static final double WORLD_WIDTH = 1.0;
    private static final int TILE_DIM = 512;
    private static final int SCREEN_SIZE = 1280;
    private static final int DEFAULT_MIN_ZOOM = 5;
    private static final int DEFAULT_MAX_ZOOM = 11;
    private static final int MAX_ZOOM_LEVEL = 22;
    private static final int MIN_RADIUS = 10;
    private static final int MAX_RADIUS = 50;
    private PointQuadTree<WeightedLatLng> mTree;
    private Collection<WeightedLatLng> mData;
    private Bounds mBounds;
    private int mRadius;
    private Gradient mGradient;
    private int[] mColorMap;
    private double[] mKernel;
    private double mOpacity;
    private double[] mMaxIntensity;

    private HeatmapTileProvider(Builder builder) {
        this.mData = builder.data;
        this.mRadius = builder.radius;
        this.mGradient = builder.gradient;
        this.mOpacity = builder.opacity;
        this.mKernel = HeatmapTileProvider.generateKernel(this.mRadius, (double)this.mRadius / 3.0);
        this.setGradient(this.mGradient);
        this.setWeightedData(this.mData);
    }

    public void setWeightedData(Collection<WeightedLatLng> data) {
        this.mData = data;
        if (this.mData.isEmpty()) {
            throw new IllegalArgumentException("No input points.");
        }
        this.mBounds = HeatmapTileProvider.getBounds(this.mData);
        this.mTree = new PointQuadTree(this.mBounds);
        for (WeightedLatLng l : this.mData) {
            this.mTree.add(l);
        }
        this.mMaxIntensity = this.getMaxIntensities(this.mRadius);
    }

    public void setData(Collection<LatLng> data) {
        this.setWeightedData(HeatmapTileProvider.wrapData(data));
    }

    private static Collection<WeightedLatLng> wrapData(Collection<LatLng> data) {
        ArrayList<WeightedLatLng> weightedData = new ArrayList<WeightedLatLng>();
        for (LatLng l : data) {
            weightedData.add(new WeightedLatLng(l));
        }
        return weightedData;
    }

    public Tile getTile(int x, int y, int zoom) {
        int bucketY;
        int bucketX;
        Point p;
        Bounds overlapBounds;
        double tileWidth = 1.0 / Math.pow(2.0, zoom);
        double padding = tileWidth * (double)this.mRadius / 512.0;
        double tileWidthPadded = tileWidth + 2.0 * padding;
        double bucketWidth = tileWidthPadded / (double)(512 + this.mRadius * 2);
        double minX = (double)x * tileWidth - padding;
        double maxX = (double)(x + 1) * tileWidth + padding;
        double minY = (double)y * tileWidth - padding;
        double maxY = (double)(y + 1) * tileWidth + padding;
        double xOffset = 0.0;
        Collection<Object> wrappedPoints = new ArrayList();
        if (minX < 0.0) {
            overlapBounds = new Bounds(minX + 1.0, 1.0, minY, maxY);
            xOffset = -1.0;
            wrappedPoints = this.mTree.search(overlapBounds);
        } else if (maxX > 1.0) {
            overlapBounds = new Bounds(0.0, maxX - 1.0, minY, maxY);
            xOffset = 1.0;
            wrappedPoints = this.mTree.search(overlapBounds);
        }
        Bounds tileBounds = new Bounds(minX, maxX, minY, maxY);
        Bounds paddedBounds = new Bounds(this.mBounds.minX - padding, this.mBounds.maxX + padding, this.mBounds.minY - padding, this.mBounds.maxY + padding);
        if (!tileBounds.intersects(paddedBounds)) {
            return TileProvider.NO_TILE;
        }
        Collection<WeightedLatLng> points = this.mTree.search(tileBounds);
        if (points.isEmpty()) {
            return TileProvider.NO_TILE;
        }
        double[][] intensity = new double[512 + this.mRadius * 2][512 + this.mRadius * 2];
        for (WeightedLatLng weightedLatLng : points) {
            p = weightedLatLng.getPoint();
            bucketX = (int)((p.x - minX) / bucketWidth);
            bucketY = (int)((p.y - minY) / bucketWidth);
            double[] dArray = intensity[bucketX];
            int n = bucketY;
            dArray[n] = dArray[n] + weightedLatLng.getIntensity();
        }
        for (WeightedLatLng weightedLatLng : wrappedPoints) {
            p = weightedLatLng.getPoint();
            bucketX = (int)((p.x + xOffset - minX) / bucketWidth);
            bucketY = (int)((p.y - minY) / bucketWidth);
            double[] dArray = intensity[bucketX];
            int n = bucketY;
            dArray[n] = dArray[n] + weightedLatLng.getIntensity();
        }
        double[][] convolved = HeatmapTileProvider.convolve(intensity, this.mKernel);
        Bitmap bitmap = HeatmapTileProvider.colorize(convolved, this.mColorMap, this.mMaxIntensity[zoom]);
        return HeatmapTileProvider.convertBitmap(bitmap);
    }

    public void setGradient(Gradient gradient) {
        this.mGradient = gradient;
        this.mColorMap = gradient.generateColorMap(this.mOpacity);
    }

    public void setRadius(int radius) {
        this.mRadius = radius;
        this.mKernel = HeatmapTileProvider.generateKernel(this.mRadius, (double)this.mRadius / 3.0);
        this.mMaxIntensity = this.getMaxIntensities(this.mRadius);
    }

    public void setOpacity(double opacity) {
        this.mOpacity = opacity;
        this.setGradient(this.mGradient);
    }

    private double[] getMaxIntensities(int radius) {
        int i;
        double[] maxIntensityArray = new double[22];
        for (i = 5; i < 11; ++i) {
            maxIntensityArray[i] = HeatmapTileProvider.getMaxValue(this.mData, this.mBounds, radius, (int)(1280.0 * Math.pow(2.0, i - 3)));
            if (i != 5) continue;
            for (int j = 0; j < i; ++j) {
                maxIntensityArray[j] = maxIntensityArray[i];
            }
        }
        for (i = 11; i < 22; ++i) {
            maxIntensityArray[i] = maxIntensityArray[10];
        }
        return maxIntensityArray;
    }

    private static Tile convertBitmap(Bitmap bitmap) {
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        bitmap.compress(Bitmap.CompressFormat.PNG, 100, (OutputStream)stream);
        byte[] bitmapdata = stream.toByteArray();
        return new Tile(512, 512, bitmapdata);
    }

    static Bounds getBounds(Collection<WeightedLatLng> points) {
        Iterator<WeightedLatLng> iter = points.iterator();
        WeightedLatLng first = iter.next();
        double minX = first.getPoint().x;
        double maxX = first.getPoint().x;
        double minY = first.getPoint().y;
        double maxY = first.getPoint().y;
        while (iter.hasNext()) {
            WeightedLatLng l = iter.next();
            double x = l.getPoint().x;
            double y = l.getPoint().y;
            if (x < minX) {
                minX = x;
            }
            if (x > maxX) {
                maxX = x;
            }
            if (y < minY) {
                minY = y;
            }
            if (!(y > maxY)) continue;
            maxY = y;
        }
        return new Bounds(minX, maxX, minY, maxY);
    }

    static double[] generateKernel(int radius, double sd) {
        double[] kernel = new double[radius * 2 + 1];
        for (int i = -radius; i <= radius; ++i) {
            kernel[i + radius] = Math.exp((double)(-i * i) / (2.0 * sd * sd));
        }
        return kernel;
    }

    static double[][] convolve(double[][] grid, double[] kernel) {
        int initial;
        double val;
        int y;
        int x;
        int radius = (int)Math.floor((double)kernel.length / 2.0);
        int dimOld = grid.length;
        int dim = dimOld - 2 * radius;
        int lowerLimit = radius;
        int upperLimit = radius + dim - 1;
        double[][] intermediate = new double[dimOld][dimOld];
        for (x = 0; x < dimOld; ++x) {
            for (y = 0; y < dimOld; ++y) {
                val = grid[x][y];
                if (val == 0.0) continue;
                int xUpperLimit = (upperLimit < x + radius ? upperLimit : x + radius) + 1;
                for (int x2 = initial = lowerLimit > x - radius ? lowerLimit : x - radius; x2 < xUpperLimit; ++x2) {
                    double[] dArray = intermediate[x2];
                    int n = y;
                    dArray[n] = dArray[n] + val * kernel[x2 - (x - radius)];
                }
            }
        }
        double[][] outputGrid = new double[dim][dim];
        for (x = lowerLimit; x < upperLimit + 1; ++x) {
            for (y = 0; y < dimOld; ++y) {
                val = intermediate[x][y];
                if (val == 0.0) continue;
                int yUpperLimit = (upperLimit < y + radius ? upperLimit : y + radius) + 1;
                for (int y2 = initial = lowerLimit > y - radius ? lowerLimit : y - radius; y2 < yUpperLimit; ++y2) {
                    double[] dArray = outputGrid[x - radius];
                    int n = y2 - radius;
                    dArray[n] = dArray[n] + val * kernel[y2 - (y - radius)];
                }
            }
        }
        return outputGrid;
    }

    static Bitmap colorize(double[][] grid, int[] colorMap, double max) {
        int maxColor = colorMap[colorMap.length - 1];
        double colorMapScaling = (double)(colorMap.length - 1) / max;
        int dim = grid.length;
        int[] colors = new int[dim * dim];
        for (int i = 0; i < dim; ++i) {
            for (int j = 0; j < dim; ++j) {
                double val = grid[j][i];
                int index = i * dim + j;
                int col = (int)(val * colorMapScaling);
                if (val != 0.0) {
                    if (col < colorMap.length) {
                        colors[index] = colorMap[col];
                        continue;
                    }
                    colors[index] = maxColor;
                    continue;
                }
                colors[index] = 0;
            }
        }
        Bitmap tile = Bitmap.createBitmap((int)dim, (int)dim, (Bitmap.Config)Bitmap.Config.ARGB_8888);
        tile.setPixels(colors, 0, dim, 0, 0, dim, dim);
        return tile;
    }

    static double getMaxValue(Collection<WeightedLatLng> points, Bounds bounds, int radius, int screenDim) {
        double maxX = bounds.maxX;
        double minX = bounds.minX;
        double maxY = bounds.maxY;
        double minY = bounds.minY;
        double boundsDim = maxX - minX > maxY - minY ? maxX - minX : maxY - minY;
        int nBuckets = (int)((double)(screenDim / (2 * radius)) + 0.5);
        double scale = (double)nBuckets / boundsDim;
        LongSparseArray buckets = new LongSparseArray();
        double max = 0.0;
        for (WeightedLatLng l : points) {
            Double value;
            double x = l.getPoint().x;
            double y = l.getPoint().y;
            int xBucket = (int)((x - minX) * scale);
            int yBucket = (int)((y - minY) * scale);
            LongSparseArray column = (LongSparseArray)buckets.get((long)xBucket);
            if (column == null) {
                column = new LongSparseArray();
                buckets.put((long)xBucket, (Object)column);
            }
            if ((value = (Double)column.get((long)yBucket)) == null) {
                value = 0.0;
            }
            value = value + l.getIntensity();
            column.put((long)yBucket, (Object)value);
            if (!(value > max)) continue;
            max = value;
        }
        return max;
    }

    public static class Builder {
        private Collection<WeightedLatLng> data;
        private int radius = 20;
        private Gradient gradient = DEFAULT_GRADIENT;
        private double opacity = 0.7;

        public Builder data(Collection<LatLng> val) {
            return this.weightedData(HeatmapTileProvider.wrapData(val));
        }

        public Builder weightedData(Collection<WeightedLatLng> val) {
            this.data = val;
            if (this.data.isEmpty()) {
                throw new IllegalArgumentException("No input points.");
            }
            return this;
        }

        public Builder radius(int val) {
            this.radius = val;
            if (this.radius < 10 || this.radius > 50) {
                throw new IllegalArgumentException("Radius not within bounds.");
            }
            return this;
        }

        public Builder gradient(Gradient val) {
            this.gradient = val;
            return this;
        }

        public Builder opacity(double val) {
            this.opacity = val;
            if (this.opacity < 0.0 || this.opacity > 1.0) {
                throw new IllegalArgumentException("Opacity must be in range [0, 1]");
            }
            return this;
        }

        public HeatmapTileProvider build() {
            if (this.data == null) {
                throw new IllegalStateException("No input data: you must use either .data or .weightedData before building");
            }
            return new HeatmapTileProvider(this);
        }
    }
}

