/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.database.connection;

import com.google.firebase.database.connection.ConnectionContext;
import com.google.firebase.database.connection.HostInfo;
import com.google.firebase.database.connection.util.StringListReader;
import com.google.firebase.database.logging.LogWrapper;
import com.google.firebase.database.tubesock.WebSocket;
import com.google.firebase.database.tubesock.WebSocketEventHandler;
import com.google.firebase.database.tubesock.WebSocketException;
import com.google.firebase.database.tubesock.WebSocketMessage;
import com.google.firebase.database.util.JsonMapper;
import java.io.EOFException;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

class WebsocketConnection {
    private static long connectionId = 0L;
    private static final long KEEP_ALIVE_TIMEOUT_MS = 45000L;
    private static final long CONNECT_TIMEOUT_MS = 30000L;
    private static final int MAX_FRAME_SIZE = 16384;
    private WSClient conn;
    private boolean everConnected = false;
    private boolean isClosed = false;
    private long totalFrames = 0L;
    private StringListReader frameReader;
    private Delegate delegate;
    private ScheduledFuture<?> keepAlive;
    private ScheduledFuture<?> connectTimeout;
    private final ConnectionContext connectionContext;
    private final ScheduledExecutorService executorService;
    private final LogWrapper logger;

    public WebsocketConnection(ConnectionContext connectionContext, HostInfo hostInfo, String optCachedHost, String appCheckToken, Delegate delegate, String optLastSessionId) {
        this.connectionContext = connectionContext;
        this.executorService = connectionContext.getExecutorService();
        this.delegate = delegate;
        long connId = connectionId++;
        this.logger = new LogWrapper(connectionContext.getLogger(), "WebSocket", "ws_" + connId);
        this.conn = this.createConnection(hostInfo, optCachedHost, appCheckToken, optLastSessionId);
    }

    private WSClient createConnection(HostInfo hostInfo, String optCachedHost, String appCheckToken, String optLastSessionId) {
        String host = optCachedHost != null ? optCachedHost : hostInfo.getHost();
        URI uri = HostInfo.getConnectionUrl(host, hostInfo.isSecure(), hostInfo.getNamespace(), optLastSessionId);
        HashMap<String, String> extraHeaders = new HashMap<String, String>();
        extraHeaders.put("User-Agent", this.connectionContext.getUserAgent());
        extraHeaders.put("X-Firebase-GMPID", this.connectionContext.getApplicationId());
        extraHeaders.put("X-Firebase-AppCheck", appCheckToken);
        WebSocket ws = new WebSocket(this.connectionContext, uri, null, extraHeaders);
        WSClientTubesock client = new WSClientTubesock(ws);
        return client;
    }

    public void open() {
        this.conn.connect();
        this.connectTimeout = this.executorService.schedule(new Runnable(){

            @Override
            public void run() {
                WebsocketConnection.this.closeIfNeverConnected();
            }
        }, 30000L, TimeUnit.MILLISECONDS);
    }

    public void start() {
    }

    public void close() {
        if (this.logger.logsDebug()) {
            this.logger.debug("websocket is being closed", new Object[0]);
        }
        this.isClosed = true;
        this.conn.close();
        if (this.connectTimeout != null) {
            this.connectTimeout.cancel(true);
        }
        if (this.keepAlive != null) {
            this.keepAlive.cancel(true);
        }
    }

    public void send(Map<String, Object> message) {
        this.resetKeepAlive();
        try {
            String toSend = JsonMapper.serializeJson(message);
            String[] segs = WebsocketConnection.splitIntoFrames(toSend, 16384);
            if (segs.length > 1) {
                this.conn.send("" + segs.length);
            }
            for (int i = 0; i < segs.length; ++i) {
                this.conn.send(segs[i]);
            }
        }
        catch (IOException e) {
            this.logger.error("Failed to serialize message: " + message.toString(), e);
            this.shutdown();
        }
    }

    private void appendFrame(String message) {
        this.frameReader.addString(message);
        --this.totalFrames;
        if (this.totalFrames == 0L) {
            try {
                this.frameReader.freeze();
                Map<String, Object> decoded = JsonMapper.parseJson(this.frameReader.toString());
                this.frameReader = null;
                if (this.logger.logsDebug()) {
                    this.logger.debug("handleIncomingFrame complete frame: " + decoded, new Object[0]);
                }
                this.delegate.onMessage(decoded);
            }
            catch (IOException e) {
                this.logger.error("Error parsing frame: " + this.frameReader.toString(), e);
                this.close();
                this.shutdown();
            }
            catch (ClassCastException e) {
                this.logger.error("Error parsing frame (cast error): " + this.frameReader.toString(), e);
                this.close();
                this.shutdown();
            }
        }
    }

    private void handleNewFrameCount(int numFrames) {
        this.totalFrames = numFrames;
        this.frameReader = new StringListReader();
        if (this.logger.logsDebug()) {
            this.logger.debug("HandleNewFrameCount: " + this.totalFrames, new Object[0]);
        }
    }

    private String extractFrameCount(String message) {
        if (message.length() <= 6) {
            try {
                int frameCount = Integer.parseInt(message);
                if (frameCount > 0) {
                    this.handleNewFrameCount(frameCount);
                }
                return null;
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        this.handleNewFrameCount(1);
        return message;
    }

    private void handleIncomingFrame(String message) {
        if (!this.isClosed) {
            this.resetKeepAlive();
            if (this.isBuffering()) {
                this.appendFrame(message);
            } else {
                String remaining = this.extractFrameCount(message);
                if (remaining != null) {
                    this.appendFrame(remaining);
                }
            }
        }
    }

    private void resetKeepAlive() {
        if (!this.isClosed) {
            if (this.keepAlive != null) {
                this.keepAlive.cancel(false);
                if (this.logger.logsDebug()) {
                    this.logger.debug("Reset keepAlive. Remaining: " + this.keepAlive.getDelay(TimeUnit.MILLISECONDS), new Object[0]);
                }
            } else if (this.logger.logsDebug()) {
                this.logger.debug("Reset keepAlive", new Object[0]);
            }
            this.keepAlive = this.executorService.schedule(this.nop(), 45000L, TimeUnit.MILLISECONDS);
        }
    }

    private Runnable nop() {
        return new Runnable(){

            @Override
            public void run() {
                if (WebsocketConnection.this.conn != null) {
                    WebsocketConnection.this.conn.send("0");
                    WebsocketConnection.this.resetKeepAlive();
                }
            }
        };
    }

    private boolean isBuffering() {
        return this.frameReader != null;
    }

    private void onClosed() {
        if (!this.isClosed) {
            if (this.logger.logsDebug()) {
                this.logger.debug("closing itself", new Object[0]);
            }
            this.shutdown();
        }
        this.conn = null;
        if (this.keepAlive != null) {
            this.keepAlive.cancel(false);
        }
    }

    private void shutdown() {
        this.isClosed = true;
        this.delegate.onDisconnect(this.everConnected);
    }

    private void closeIfNeverConnected() {
        if (!this.everConnected && !this.isClosed) {
            if (this.logger.logsDebug()) {
                this.logger.debug("timed out on connect", new Object[0]);
            }
            this.conn.close();
        }
    }

    private static String[] splitIntoFrames(String src, int maxFrameSize) {
        if (src.length() <= maxFrameSize) {
            return new String[]{src};
        }
        ArrayList<String> segs = new ArrayList<String>();
        for (int i = 0; i < src.length(); i += maxFrameSize) {
            int end = Math.min(i + maxFrameSize, src.length());
            String seg = src.substring(i, end);
            segs.add(seg);
        }
        return segs.toArray(new String[segs.size()]);
    }

    private class WSClientTubesock
    implements WSClient,
    WebSocketEventHandler {
        private WebSocket ws;

        private WSClientTubesock(WebSocket ws) {
            this.ws = ws;
            this.ws.setEventHandler(this);
        }

        @Override
        public void onOpen() {
            WebsocketConnection.this.executorService.execute(new Runnable(){

                @Override
                public void run() {
                    WebsocketConnection.this.connectTimeout.cancel(false);
                    WebsocketConnection.this.everConnected = true;
                    if (WebsocketConnection.this.logger.logsDebug()) {
                        WebsocketConnection.this.logger.debug("websocket opened", new Object[0]);
                    }
                    WebsocketConnection.this.resetKeepAlive();
                }
            });
        }

        @Override
        public void onMessage(WebSocketMessage msg) {
            final String str = msg.getText();
            if (WebsocketConnection.this.logger.logsDebug()) {
                WebsocketConnection.this.logger.debug("ws message: " + str, new Object[0]);
            }
            WebsocketConnection.this.executorService.execute(new Runnable(){

                @Override
                public void run() {
                    WebsocketConnection.this.handleIncomingFrame(str);
                }
            });
        }

        @Override
        public void onClose() {
            String logMessage = "closed";
            WebsocketConnection.this.executorService.execute(new Runnable(){

                @Override
                public void run() {
                    if (WebsocketConnection.this.logger.logsDebug()) {
                        WebsocketConnection.this.logger.debug("closed", new Object[0]);
                    }
                    WebsocketConnection.this.onClosed();
                }
            });
        }

        @Override
        public void onError(final WebSocketException e) {
            WebsocketConnection.this.executorService.execute(new Runnable(){

                @Override
                public void run() {
                    if (e.getCause() != null && e.getCause() instanceof EOFException) {
                        WebsocketConnection.this.logger.debug("WebSocket reached EOF.", new Object[0]);
                    } else {
                        WebsocketConnection.this.logger.debug("WebSocket error.", e, new Object[0]);
                    }
                    WebsocketConnection.this.onClosed();
                }
            });
        }

        @Override
        public void onLogMessage(String msg) {
            if (WebsocketConnection.this.logger.logsDebug()) {
                WebsocketConnection.this.logger.debug("Tubesock: " + msg, new Object[0]);
            }
        }

        @Override
        public void send(String msg) {
            this.ws.send(msg);
        }

        @Override
        public void close() {
            this.ws.close();
        }

        private void shutdown() {
            this.ws.close();
            try {
                this.ws.blockClose();
            }
            catch (InterruptedException e) {
                WebsocketConnection.this.logger.error("Interrupted while shutting down websocket threads", e);
            }
        }

        @Override
        public void connect() {
            try {
                this.ws.connect();
            }
            catch (WebSocketException e) {
                if (WebsocketConnection.this.logger.logsDebug()) {
                    WebsocketConnection.this.logger.debug("Error connecting", e, new Object[0]);
                }
                this.shutdown();
            }
        }
    }

    private static interface WSClient {
        public void connect();

        public void close();

        public void send(String var1);
    }

    public static interface Delegate {
        public void onMessage(Map<String, Object> var1);

        public void onDisconnect(boolean var1);
    }
}

