/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.crashlytics.internal.persistence;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.google.firebase.crashlytics.internal.Logger;
import com.google.firebase.crashlytics.internal.common.CrashlyticsReportWithSessionId;
import com.google.firebase.crashlytics.internal.metadata.UserMetadata;
import com.google.firebase.crashlytics.internal.model.CrashlyticsReport;
import com.google.firebase.crashlytics.internal.model.ImmutableList;
import com.google.firebase.crashlytics.internal.model.serialization.CrashlyticsReportJsonTransform;
import com.google.firebase.crashlytics.internal.persistence.FileStore;
import com.google.firebase.crashlytics.internal.settings.SettingsProvider;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicInteger;

public class CrashlyticsReportPersistence {
    private static final Charset UTF_8 = Charset.forName("UTF-8");
    private static final int MAX_OPEN_SESSIONS = 8;
    private static final String REPORT_FILE_NAME = "report";
    private static final String SESSION_START_TIMESTAMP_FILE_NAME = "start-time";
    private static final String EVENT_FILE_NAME_PREFIX = "event";
    private static final int EVENT_COUNTER_WIDTH = 10;
    private static final String EVENT_COUNTER_FORMAT = "%010d";
    private static final int EVENT_NAME_LENGTH = "event".length() + 10;
    private static final String PRIORITY_EVENT_SUFFIX = "_";
    private static final String NORMAL_EVENT_SUFFIX = "";
    private static final CrashlyticsReportJsonTransform TRANSFORM = new CrashlyticsReportJsonTransform();
    private static final Comparator<? super File> LATEST_SESSION_ID_FIRST_COMPARATOR = (f1, f2) -> f2.getName().compareTo(f1.getName());
    private static final FilenameFilter EVENT_FILE_FILTER = (f, name) -> name.startsWith(EVENT_FILE_NAME_PREFIX);
    private final AtomicInteger eventCounter = new AtomicInteger(0);
    private final FileStore fileStore;
    @NonNull
    private final SettingsProvider settingsProvider;

    public CrashlyticsReportPersistence(FileStore fileStore, SettingsProvider settingsProvider) {
        this.fileStore = fileStore;
        this.settingsProvider = settingsProvider;
    }

    public void persistReport(@NonNull CrashlyticsReport report) {
        CrashlyticsReport.Session session = report.getSession();
        if (session == null) {
            Logger.getLogger().d("Could not get session for report");
            return;
        }
        String sessionId = session.getIdentifier();
        try {
            String json = TRANSFORM.reportToJson(report);
            CrashlyticsReportPersistence.writeTextFile(this.fileStore.getSessionFile(sessionId, REPORT_FILE_NAME), json);
            CrashlyticsReportPersistence.writeTextFile(this.fileStore.getSessionFile(sessionId, SESSION_START_TIMESTAMP_FILE_NAME), NORMAL_EVENT_SUFFIX, session.getStartedAt());
        }
        catch (IOException e) {
            Logger.getLogger().d("Could not persist report for session " + sessionId, e);
        }
    }

    public void persistEvent(@NonNull CrashlyticsReport.Session.Event event, @NonNull String sessionId) {
        this.persistEvent(event, sessionId, false);
    }

    public void persistEvent(@NonNull CrashlyticsReport.Session.Event event, @NonNull String sessionId, boolean isHighPriority) {
        int maxEventsToKeep = this.settingsProvider.getSettingsSync().sessionData.maxCustomExceptionEvents;
        String json = TRANSFORM.eventToJson(event);
        String fileName = CrashlyticsReportPersistence.generateEventFilename(this.eventCounter.getAndIncrement(), isHighPriority);
        try {
            CrashlyticsReportPersistence.writeTextFile(this.fileStore.getSessionFile(sessionId, fileName), json);
        }
        catch (IOException e) {
            Logger.getLogger().w("Could not persist event for session " + sessionId, e);
        }
        this.trimEvents(sessionId, maxEventsToKeep);
    }

    public SortedSet<String> getOpenSessionIds() {
        return new TreeSet<String>(this.fileStore.getAllOpenSessionIds()).descendingSet();
    }

    public long getStartTimestampMillis(String sessionId) {
        File sessionStartTimestampFile = this.fileStore.getSessionFile(sessionId, SESSION_START_TIMESTAMP_FILE_NAME);
        return sessionStartTimestampFile.lastModified();
    }

    public boolean hasFinalizedReports() {
        return !this.fileStore.getReports().isEmpty() || !this.fileStore.getPriorityReports().isEmpty() || !this.fileStore.getNativeReports().isEmpty();
    }

    public void deleteAllReports() {
        this.deleteFiles(this.fileStore.getReports());
        this.deleteFiles(this.fileStore.getPriorityReports());
        this.deleteFiles(this.fileStore.getNativeReports());
    }

    private void deleteFiles(List<File> files) {
        for (File file : files) {
            file.delete();
        }
    }

    public void finalizeReports(@Nullable String currentSessionId, long sessionEndTime) {
        SortedSet<String> sessions = this.capAndGetOpenSessions(currentSessionId);
        for (String sessionId : sessions) {
            Logger.getLogger().v("Finalizing report for session " + sessionId);
            this.synthesizeReport(sessionId, sessionEndTime);
            this.fileStore.deleteSessionFiles(sessionId);
        }
        this.capFinalizedReports();
    }

    public void finalizeSessionWithNativeEvent(String previousSessionId, CrashlyticsReport.FilesPayload ndkPayload) {
        File reportFile = this.fileStore.getSessionFile(previousSessionId, REPORT_FILE_NAME);
        Logger.getLogger().d("Writing native session report for " + previousSessionId + " to file: " + reportFile);
        this.synthesizeNativeReportFile(reportFile, ndkPayload, previousSessionId);
    }

    @NonNull
    public List<CrashlyticsReportWithSessionId> loadFinalizedReports() {
        List<File> allReportFiles = this.getAllFinalizedReportFiles();
        ArrayList<CrashlyticsReportWithSessionId> allReports = new ArrayList<CrashlyticsReportWithSessionId>();
        for (File reportFile : allReportFiles) {
            try {
                CrashlyticsReport jsonReport = TRANSFORM.reportFromJson(CrashlyticsReportPersistence.readTextFile(reportFile));
                allReports.add(CrashlyticsReportWithSessionId.create(jsonReport, reportFile.getName(), reportFile));
            }
            catch (IOException e) {
                Logger.getLogger().w("Could not load report file " + reportFile + "; deleting", e);
                reportFile.delete();
            }
        }
        return allReports;
    }

    private SortedSet<String> capAndGetOpenSessions(@Nullable String currentSessionId) {
        this.fileStore.cleanupLegacyFiles();
        SortedSet<String> openSessionIds = this.getOpenSessionIds();
        if (currentSessionId != null) {
            openSessionIds.remove(currentSessionId);
        }
        if (openSessionIds.size() <= 8) {
            return openSessionIds;
        }
        while (openSessionIds.size() > 8) {
            String id = openSessionIds.last();
            Logger.getLogger().d("Removing session over cap: " + id);
            this.fileStore.deleteSessionFiles(id);
            openSessionIds.remove(id);
        }
        return openSessionIds;
    }

    private void capFinalizedReports() {
        int maxReportsToKeep = this.settingsProvider.getSettingsSync().sessionData.maxCompleteSessionsCount;
        List<File> finalizedReportFiles = this.getAllFinalizedReportFiles();
        int fileCount = finalizedReportFiles.size();
        if (fileCount <= maxReportsToKeep) {
            return;
        }
        List<File> filesToRemove = finalizedReportFiles.subList(maxReportsToKeep, fileCount);
        for (File reportFile : filesToRemove) {
            reportFile.delete();
        }
    }

    private List<File> getAllFinalizedReportFiles() {
        ArrayList<File> sortedPriorityReports = new ArrayList<File>();
        sortedPriorityReports.addAll(this.fileStore.getPriorityReports());
        sortedPriorityReports.addAll(this.fileStore.getNativeReports());
        Collections.sort(sortedPriorityReports, LATEST_SESSION_ID_FIRST_COMPARATOR);
        List<File> sortedReports = this.fileStore.getReports();
        Collections.sort(sortedReports, LATEST_SESSION_ID_FIRST_COMPARATOR);
        sortedPriorityReports.addAll(sortedReports);
        return sortedPriorityReports;
    }

    private void synthesizeReport(String sessionId, long sessionEndTime) {
        List<File> eventFiles = this.fileStore.getSessionFiles(sessionId, EVENT_FILE_FILTER);
        if (eventFiles.isEmpty()) {
            Logger.getLogger().v("Session " + sessionId + " has no events.");
            return;
        }
        Collections.sort(eventFiles);
        ArrayList<CrashlyticsReport.Session.Event> events = new ArrayList<CrashlyticsReport.Session.Event>();
        boolean isHighPriorityReport = false;
        for (File eventFile : eventFiles) {
            try {
                CrashlyticsReport.Session.Event event = TRANSFORM.eventFromJson(CrashlyticsReportPersistence.readTextFile(eventFile));
                events.add(event);
                isHighPriorityReport = isHighPriorityReport || CrashlyticsReportPersistence.isHighPriorityEventFile(eventFile.getName());
            }
            catch (IOException e) {
                Logger.getLogger().w("Could not add event to report for " + eventFile, e);
            }
        }
        if (events.isEmpty()) {
            Logger.getLogger().w("Could not parse event files for session " + sessionId);
            return;
        }
        String userId = UserMetadata.readUserId(sessionId, this.fileStore);
        File reportFile = this.fileStore.getSessionFile(sessionId, REPORT_FILE_NAME);
        this.synthesizeReportFile(reportFile, events, sessionEndTime, isHighPriorityReport, userId);
    }

    private void synthesizeNativeReportFile(@NonNull File reportFile, @NonNull CrashlyticsReport.FilesPayload ndkPayload, @NonNull String previousSessionId) {
        try {
            CrashlyticsReport report = TRANSFORM.reportFromJson(CrashlyticsReportPersistence.readTextFile(reportFile)).withNdkPayload(ndkPayload);
            CrashlyticsReportPersistence.writeTextFile(this.fileStore.getNativeReport(previousSessionId), TRANSFORM.reportToJson(report));
        }
        catch (IOException e) {
            Logger.getLogger().w("Could not synthesize final native report file for " + reportFile, e);
        }
    }

    private void synthesizeReportFile(@NonNull File reportFile, @NonNull List<CrashlyticsReport.Session.Event> events, long sessionEndTime, boolean isHighPriorityReport, @Nullable String userId) {
        try {
            CrashlyticsReport report = TRANSFORM.reportFromJson(CrashlyticsReportPersistence.readTextFile(reportFile)).withSessionEndFields(sessionEndTime, isHighPriorityReport, userId).withEvents(ImmutableList.from(events));
            CrashlyticsReport.Session session = report.getSession();
            if (session == null) {
                return;
            }
            File finalizedReportFile = isHighPriorityReport ? this.fileStore.getPriorityReport(session.getIdentifier()) : this.fileStore.getReport(session.getIdentifier());
            CrashlyticsReportPersistence.writeTextFile(finalizedReportFile, TRANSFORM.reportToJson(report));
        }
        catch (IOException e) {
            Logger.getLogger().w("Could not synthesize final report file for " + reportFile, e);
        }
    }

    private static boolean isHighPriorityEventFile(@NonNull String fileName) {
        return fileName.startsWith(EVENT_FILE_NAME_PREFIX) && fileName.endsWith(PRIORITY_EVENT_SUFFIX);
    }

    private static boolean isNormalPriorityEventFile(@NonNull File dir, @NonNull String name) {
        return name.startsWith(EVENT_FILE_NAME_PREFIX) && !name.endsWith(PRIORITY_EVENT_SUFFIX);
    }

    @NonNull
    private static String generateEventFilename(int eventNumber, boolean isHighPriority) {
        String paddedEventNumber = String.format(Locale.US, EVENT_COUNTER_FORMAT, eventNumber);
        String prioritySuffix = isHighPriority ? PRIORITY_EVENT_SUFFIX : NORMAL_EVENT_SUFFIX;
        return EVENT_FILE_NAME_PREFIX + paddedEventNumber + prioritySuffix;
    }

    private int trimEvents(String sessionId, int maximum) {
        List<File> normalPriorityEventFiles = this.fileStore.getSessionFiles(sessionId, CrashlyticsReportPersistence::isNormalPriorityEventFile);
        Collections.sort(normalPriorityEventFiles, CrashlyticsReportPersistence::oldestEventFileFirst);
        return CrashlyticsReportPersistence.capFilesCount(normalPriorityEventFiles, maximum);
    }

    @NonNull
    private static String getEventNameWithoutPriority(@NonNull String eventFileName) {
        return eventFileName.substring(0, EVENT_NAME_LENGTH);
    }

    private static int oldestEventFileFirst(@NonNull File f1, @NonNull File f2) {
        String name1 = CrashlyticsReportPersistence.getEventNameWithoutPriority(f1.getName());
        String name2 = CrashlyticsReportPersistence.getEventNameWithoutPriority(f2.getName());
        return name1.compareTo(name2);
    }

    private static void writeTextFile(File file, String text) throws IOException {
        try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(file), UTF_8);){
            writer.write(text);
        }
    }

    private static void writeTextFile(File file, String text, long lastModifiedTimestampSeconds) throws IOException {
        try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(file), UTF_8);){
            writer.write(text);
            file.setLastModified(CrashlyticsReportPersistence.convertTimestampFromSecondsToMs(lastModifiedTimestampSeconds));
        }
    }

    @NonNull
    private static String readTextFile(@NonNull File file) throws IOException {
        byte[] readBuffer = new byte[8192];
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        try (FileInputStream fileInput = new FileInputStream(file);){
            int read;
            while ((read = fileInput.read(readBuffer)) > 0) {
                bos.write(readBuffer, 0, read);
            }
            String string = new String(bos.toByteArray(), UTF_8);
            return string;
        }
    }

    private static int capFilesCount(List<File> files, int maximum) {
        int numRetained = files.size();
        for (File f : files) {
            if (numRetained <= maximum) {
                return numRetained;
            }
            FileStore.recursiveDelete(f);
            --numRetained;
        }
        return numRetained;
    }

    private static long convertTimestampFromSecondsToMs(long timestampSeconds) {
        return timestampSeconds * 1000L;
    }
}

