/*
 * Copyright 2010-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ksp.org.jetbrains.kotlin.resolve.calls.inference.components

import ksp.org.jetbrains.kotlin.builtins.KotlinBuiltIns
import ksp.org.jetbrains.kotlin.config.LanguageVersionSettings
import ksp.org.jetbrains.kotlin.descriptors.TypeParameterDescriptor
import ksp.org.jetbrains.kotlin.resolve.calls.components.ClassicTypeSystemContextForCS
import ksp.org.jetbrains.kotlin.resolve.calls.inference.ConstraintSystemBuilder
import ksp.org.jetbrains.kotlin.resolve.calls.inference.model.NewConstraintSystemImpl
import ksp.org.jetbrains.kotlin.resolve.calls.inference.model.SimpleConstraintSystemConstraintPosition
import ksp.org.jetbrains.kotlin.resolve.calls.inference.model.TypeVariableFromCallableDescriptor
import ksp.org.jetbrains.kotlin.resolve.calls.inference.substitute
import ksp.org.jetbrains.kotlin.resolve.calls.results.SimpleConstraintSystem
import ksp.org.jetbrains.kotlin.types.TypeConstructorSubstitution
import ksp.org.jetbrains.kotlin.types.checker.KotlinTypeRefiner
import ksp.org.jetbrains.kotlin.types.model.*
import ksp.org.jetbrains.kotlin.types.typeUtil.asTypeProjection

class SimpleConstraintSystemImpl(
    constraintInjector: ConstraintInjector,
    builtIns: KotlinBuiltIns,
    kotlinTypeRefiner: KotlinTypeRefiner,
    languageVersionSettings: LanguageVersionSettings
) : SimpleConstraintSystem {
    val system = NewConstraintSystemImpl(
        constraintInjector, ClassicTypeSystemContextForCS(builtIns, kotlinTypeRefiner), languageVersionSettings
    )
    val csBuilder: ConstraintSystemBuilder =
        system.getBuilder()

    override fun registerTypeVariables(typeParameters: Collection<TypeParameterMarker>): TypeSubstitutorMarker {

        val substitutionMap = typeParameters.associate {
            requireOrDescribe(it is TypeParameterDescriptor, it)
            val variable = TypeVariableFromCallableDescriptor(it)
            csBuilder.registerVariable(variable)

            it.defaultType.constructor to variable.defaultType.asTypeProjection()
        }
        val substitutor = TypeConstructorSubstitution.createByConstructorsMap(substitutionMap).buildSubstitutor()
        for (typeParameter in typeParameters) {
            requireOrDescribe(typeParameter is TypeParameterDescriptor, typeParameter)
            for (upperBound in typeParameter.upperBounds) {
                addSubtypeConstraint(substitutor.substitute(typeParameter.defaultType), substitutor.substitute(upperBound.unwrap()))
            }
        }
        return substitutor
    }

    override fun addSubtypeConstraint(subType: KotlinTypeMarker, superType: KotlinTypeMarker) {
        csBuilder.addSubtypeConstraint(
            subType,
            superType,
            SimpleConstraintSystemConstraintPosition
        )
    }

    override fun hasContradiction() = csBuilder.hasContradiction
    override val captureFromArgument get() = true

    override val context: TypeSystemInferenceExtensionContext
        get() = system
}
