/*
 * Copyright 2010-2019 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.ir.backend.js.lower

import ksp.org.jetbrains.kotlin.backend.common.BodyLoweringPass
import ksp.org.jetbrains.kotlin.backend.common.DeclarationTransformer
import ksp.org.jetbrains.kotlin.descriptors.ClassKind
import ksp.org.jetbrains.kotlin.descriptors.DescriptorVisibilities
import ksp.org.jetbrains.kotlin.ir.IrStatement
import ksp.org.jetbrains.kotlin.ir.backend.js.JsCommonBackendContext
import ksp.org.jetbrains.kotlin.ir.backend.js.syntheticPrimaryConstructor
import ksp.org.jetbrains.kotlin.ir.builders.declarations.addConstructor
import ksp.org.jetbrains.kotlin.ir.declarations.*
import ksp.org.jetbrains.kotlin.ir.expressions.IrBody
import ksp.org.jetbrains.kotlin.ir.expressions.IrInstanceInitializerCall
import ksp.org.jetbrains.kotlin.ir.expressions.impl.IrDelegatingConstructorCallImpl
import ksp.org.jetbrains.kotlin.ir.expressions.impl.IrInstanceInitializerCallImpl
import ksp.org.jetbrains.kotlin.ir.util.constructors
import ksp.org.jetbrains.kotlin.ir.util.parentAsClass
import ksp.org.jetbrains.kotlin.ir.visitors.IrElementTransformerVoid
import ksp.org.jetbrains.kotlin.ir.visitors.transformChildrenVoid

val IrDeclaration.isSyntheticPrimaryConstructor: Boolean
    get() = origin == PrimaryConstructorLowering.SYNTHETIC_PRIMARY_CONSTRUCTOR

/**
 * Creates a primary constructor if it doesn't exist.
 */
class PrimaryConstructorLowering(val context: JsCommonBackendContext) : DeclarationTransformer {
    override fun transformFlat(declaration: IrDeclaration): List<IrDeclaration>? {
        if (declaration is IrClass && declaration.kind != ClassKind.INTERFACE) {
            val constructors = declaration.constructors

            if (constructors.any { it.isPrimary }) return null

            declaration.syntheticPrimaryConstructor = createPrimaryConstructor(declaration)
        }

        return null
    }

    companion object {
        val SYNTHETIC_PRIMARY_CONSTRUCTOR by IrDeclarationOriginImpl
    }

    private val unitType = context.irBuiltIns.unitType

    private fun createPrimaryConstructor(irClass: IrClass): IrConstructor {
        val declaration = irClass.addConstructor {
            origin = SYNTHETIC_PRIMARY_CONSTRUCTOR
            isPrimary = true
            visibility = DescriptorVisibilities.PRIVATE
        }

        declaration.body = irClass.run {
            factory.createBlockBody(startOffset, endOffset, listOf(IrInstanceInitializerCallImpl(startOffset, endOffset, symbol, unitType)))
        }

        return declaration
    }
}

/**
 * Generates a delegating constructor to the synthetic primary constructor.
 */
class DelegateToSyntheticPrimaryConstructor(context: JsCommonBackendContext) : BodyLoweringPass {
    override fun lower(irBody: IrBody, container: IrDeclaration) {
        if (container is IrConstructor && !container.isPrimary) {
            container.parentAsClass.syntheticPrimaryConstructor?.let { primary ->
                val initializeTransformer = object : IrElementTransformerVoid() {
                    override fun visitDeclaration(declaration: IrDeclarationBase): IrStatement = declaration // optimize visiting

                    override fun visitInstanceInitializerCall(expression: IrInstanceInitializerCall) = expression.run {
                        IrDelegatingConstructorCallImpl(
                            startOffset, endOffset, type,
                            primary.symbol,
                            typeArgumentsCount = primary.typeParameters.size
                        )
                    }
                }

                irBody.transformChildrenVoid(initializeTransformer)
            }
        }
    }
}
