package com.vhall.ims;

import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;

import com.vhall.framework.connect.IVHService;
import com.vhall.framework.VHAPI;
import com.vhall.framework.connect.VhallConnectService;
import com.vhall.framework.VhallSDK;
import com.vhall.ims.message.IVHMessage;
import com.vhall.logmanager.L;
import com.vhall.logmanager.LogReporter;
import com.vhall.message.ConnectServer;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.net.URLDecoder;
import java.util.List;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Response;

/**
 * Created by Hank on 2017/12/20.
 */
public class VHIM implements IVHService {
    private static final String TAG = "VHIM";
    private String mChannelId;
    private String mAccessToken;
    private OnMessageListener mMessageListener;
    private VhallConnectService.OnConnectStateChangedListener mOnConnectChangedListener;
    private Handler mHandler;
    //IM服务
//    public static final String TYPE_CUSTOM = "CustomBroadcast";
//    public static final String TYPE_CHAT = "Chat";
    public static final String TYPE_CUSTOM = "service_custom";
    public static final String TYPE_CHAT = "service_im";
    public static final String TYPE_ONLINE = "service_online";
    public static final String TYPE_JOIN = "Join";
    public static final String TYPE_LEAVE = "Leave";

    public static final String TYPE_TEXT = "text";//文本消息
    public static final String TYPE_IMAGE = "image";//图片消息
    public static final String TYPE_LINK = "link";//URL链接
    public static final String TYPE_VIDEO = "video";//视频消息
    public static final String TYPE_VOICE = "voice";//语音消息
    public static final String TYPE_DISABLE = "disable";//禁言某个用户
    public static final String TYPE_DISABLE_ALL = "disable_all";//全员禁言
    public static final String TYPE_PERMIT = "permit";//取消禁言某个用户
    public static final String TYPE_PERMIT_ALL = "permit_all";//取消全员禁言


    public interface Callback {
        void onSuccess();

        void onFailure(int errorCode, String errorMsg);
    }

    public interface OnMessageListener {
        //聊天消息
        void onMessage(String msg);

        void onChannelStatus(String msg);
    }

    public interface ResultCallback {
        void onSuccess(String data);

        void onFailure(int errorCode, String errorMsg);
    }

    public void setOnMessageListener(OnMessageListener listener) {
        this.mMessageListener = listener;
    }

    public void setOnConnectChangedListener(VhallConnectService.OnConnectStateChangedListener listener) {
        mOnConnectChangedListener = listener;
    }

    public VHIM(String channelId, String accessToken) {
        this.mChannelId = channelId;
        this.mAccessToken = accessToken;
        mHandler = new Handler(Looper.getMainLooper());
        trackInitEvent();
    }

    public boolean join() {
        return VhallSDK.getInstance().join(this);
    }

    public boolean leave() {
        mMessageListener = null;
        mOnConnectChangedListener = null;
        return VhallSDK.getInstance().leave(this);
    }

    @Override
    public String getChannelId() {
        return mChannelId;
    }

    /**
     * hkl 新增
     */
    @Override
    public String getAccessToken() {
        return mAccessToken;
    }

    @Override
    public void onConnectStateChanged(ConnectServer.State state, int serverType) {
        if (mOnConnectChangedListener != null)
            mOnConnectChangedListener.onStateChanged(state, serverType);
    }

    @Override
    public void onMessage(String msg) {
        if (TextUtils.isEmpty(msg))
            return;
        //TODO 是否要解析text，只往外抛text内数据
        try {
            JSONObject obj = new JSONObject(msg);
            String textStr = obj.optString("text");
            if (!TextUtils.isEmpty(textStr)) {
                msg = URLDecoder.decode(textStr);
            }
            L.e(TAG, "onMessage: " + msg);
            JSONObject msgObj = new JSONObject(msg);
            //收到socket Join回调消息
            if (mMessageListener != null) {
                if (msgObj.has("event") && msgObj.optString("event").equals("Join")) {
                    mMessageListener.onChannelStatus(msg);
                } else {
                    if (!("service_document").equals(msgObj.optString("service_type"))) {
                        mMessageListener.onMessage(msg);
                    }
                }
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    public void sendMsg(String msg, final Callback callback) {
        JSONObject obj = new JSONObject();
        try {
            obj.put("type", TYPE_TEXT);
            obj.put("text_content", msg);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        VHAPI.sendMsg(obj.toString(), mAccessToken, TYPE_CHAT, mChannelId, "", new SendCallback(callback));
    }

    public void sendMsg(IVHMessage message,Callback callback){
        if(message.getBody() != null){
            String accessToken = message.getAccessToken();
            String channelId = message.getChannelId();
            VHAPI.sendMsg(message.getBody().getBodyStr(),
                    TextUtils.isEmpty(accessToken)?mAccessToken:accessToken,
                    message.getMessageType(),
                    TextUtils.isEmpty(channelId)?mChannelId:channelId,
                    message.getRoomId(),
                    message.getNoAudit(),
                    message.getContext(),
                    new SendCallback(callback));
        }else{
            L.e(TAG,"body is null,please check your message ");
        }
    }

    /**
     * 发送文本类型的消息
     * @param msg        文字消息
     * @param context    上下文环境
     * @param callback   消息回调
     */
    public void sendMsgWithContext(String msg,String context,Callback callback){
        sendMsg(msg,TYPE_CHAT,context,callback);
    }

    /**
     * 消息发送
     *
     * @param msg      消息内容
     * @param type     消息类型
     * @param context  增加context
     * @param callback
     */
    public void sendMsg(String msg,String type,String context,Callback callback){
        JSONObject obj = new JSONObject();
        try {
            if (!TextUtils.isEmpty(type)) {
                obj.put("type", type);
            } else {
                obj.put("type", TYPE_TEXT);
            }
            switch (type) {
                case TYPE_TEXT:
                    obj.put("text_content", msg);
                    break;
                case TYPE_IMAGE:
                    obj.put("image_url", msg);
                    break;
                case TYPE_LINK:
                    obj.put("link_url", msg);
                    break;
                case TYPE_VIDEO:
                    obj.put("video_url", msg);
                    break;
                case TYPE_VOICE:
                    obj.put("voice_url", msg);
                    break;
                case TYPE_DISABLE:
                    obj.put("target_id", msg);
                    break;
                case TYPE_DISABLE_ALL:
                    break;
                case TYPE_PERMIT:
                    obj.put("target_id", msg);
                    break;
                case TYPE_PERMIT_ALL:
                    break;
                default:
                    break;
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        if(TextUtils.isEmpty(context)){
            VHAPI.sendMsg(obj.toString(), mAccessToken, TYPE_CHAT, mChannelId,"", new SendCallback(callback));
        }else{
            VHAPI.sendMsg(obj.toString(), mAccessToken, TYPE_CHAT, mChannelId, "","0", context,new SendCallback(callback));
        }
    }

    /**
     * 消息发送
     *
     * @param msg      消息内容
     * @param type     消息类型
     * @param callback
     */
    public void sendMsg(String msg, String type, Callback callback) {
        sendMsg(msg,type,"",callback);
    }

    /**
     * 图文消息
     *
     * @param text
     * @param imageUrls
     * @param callback
     */
    public void sendImageText(String text, List<String> imageUrls, Callback callback) {
        JSONObject obj = new JSONObject();
        try {
            JSONArray array = new JSONArray(imageUrls);
            obj.put("type", TYPE_IMAGE);
            obj.put("text_content", text);
            obj.put("image_urls", array);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        VHAPI.sendMsg(obj.toString(), mAccessToken, TYPE_CHAT, mChannelId, "", new SendCallback(callback));
    }

    /**
     * 自定义消息
     *
     * @param data
     * @param callback
     */
    public void sendCustomMsg(String data, Callback callback) {
        VHAPI.sendMsg(data, mAccessToken, TYPE_CUSTOM, mChannelId, "", new SendCallback(callback));
    }

    /**
     * 频道禁言操作
     *
     * @param type
     * @param targetId
     * @param callback
     */
    public void setChannelMsg(String type, String targetId, Callback callback) {
        VHIMApi.setChannel(mChannelId, type, targetId, mAccessToken, new SendCallback(callback));
    }

    /**
     * 获取用户列表
     *
     * @param curPage
     * @param pageSize
     * @param callback
     */
    public void getUserList(int curPage, int pageSize, ResultCallback callback) {
        VHIMApi.getUserIdList(mChannelId, mAccessToken, curPage, pageSize, new GetCallback(callback));
    }

    public void getHistoryList(String type, int curPage, int pageSize, String filterStatus, String startTime, String endTime, String msgType, String auditStatus, ResultCallback callback) {
        VHIMApi.getHistoryList(mChannelId, type, curPage, pageSize, filterStatus, startTime, endTime, msgType, auditStatus, mAccessToken, new GetCallback(callback));
    }

    private class SendCallback implements okhttp3.Callback {
        private Callback callback;

        public SendCallback(Callback callback) {
            this.callback = callback;
        }

        @Override
        public void onFailure(Call call, IOException e) {
            if (callback != null)
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        callback.onFailure(-1, "error network,please try later！");
                    }
                });
        }

        @Override
        public void onResponse(Call call, Response response) throws IOException {
            if (callback != null) {
                final String result = response.body().string();
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (!TextUtils.isEmpty(result)) {
                            try {
                                JSONObject resultObj = new JSONObject(result);
                                int code = resultObj.optInt("code");
                                if (code == 200) {
                                    callback.onSuccess();
                                } else {
                                    String msg = resultObj.optString("msg");
                                    callback.onFailure(code, msg);
                                }
                            } catch (JSONException e) {
                                e.printStackTrace();
                            }
                        }
                    }
                });
            }
        }
    }

    private class GetCallback implements okhttp3.Callback {

        private ResultCallback callback;

        public GetCallback(ResultCallback callback) {
            this.callback = callback;
        }

        @Override
        public void onFailure(Call call, IOException e) {
            if (callback != null)
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        callback.onFailure(-1, "error network,please try later！");
                    }
                });
        }

        @Override
        public void onResponse(Call call, Response response) throws IOException {
            if (callback != null) {
                final String result = response.body().string();
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (!TextUtils.isEmpty(result)) {
                            try {
                                JSONObject resultObj = new JSONObject(result);
                                int code = resultObj.optInt("code");
                                JSONObject data = resultObj.optJSONObject("data");
                                if (code == 200) {
                                    callback.onSuccess(data.toString());
                                } else {
                                    String msg = resultObj.optString("msg");
                                    callback.onFailure(code, msg);
                                }
                            } catch (JSONException e) {
                                e.printStackTrace();
                            }
                        }
                    }
                });
            }
        }
    }

    private void trackInitEvent(){
        JSONObject params = new JSONObject();
        try {
            params.put("channelId",mChannelId);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        LogReporter.getInstance().onCollection(LogReporter.LOG_EVENT_INITIM, params);
    }

}