package com.vhall.rtc;

import android.text.TextUtils;

import com.vhall.framework.common.ICallback;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author：vhall Email：jooperge@163.com
 * 描述：
 * 修改历史:
 * <p>
 * 创建于： 2022/10/18
 */
public class VRTCParams {

    public static final String RTCTYPE_VRTC = "1";
    public static final String RTCTYPE_TRTC = "2";

    /**
     * 流类型
     */
    public static class VRTCStreamType {
        public static final int VhallStreamTypeOnlyAudio = 0;
        public static final int VhallStreamTypeOnlyVideo = 1;
        public static final int VhallStreamTypeAudioAndVideo = 2;
        public static final int VhallStreamTypeScreen = 3;
        public static final int VhallStreamTypeFile = 4;
        public static final int VhallStreamTypeVideoPatrol = 5;
        public static final int VhallStreamTypeCustom = 6;
    }

    /**
     * 视频填充模式
     */
    public static class VRTCVideoFillMode {
        /**
         * VRTC专有模式，拉伸填充
         */
        public static final int RENDER_MODE_NONE = 0;
        /**
         * 适应模式：即按画面长边进行缩放以适应显示区域，短边部分会被填充为黑色，此模式下图像完整但可能留有黑边。
         */
        public static final int RENDER_MODE_FIT = 1;

        /**
         * 填充模式：即将画面内容居中等比缩放以充满整个显示区域，超出显示区域的部分将会被裁剪掉，此模式下画面可能不完整。
         */
        public static final int RENDER_MODE_ASPECT_FILL = 2;
    }

    /**
     * 互动流属性/参数
     */
    public static class VRTCStreamParam {
        /**
         * 业务自定义数据，目前仅VRTC支持，收到远端流时可从 Stream#getAttributes()获取
         */
        public String attributes;
        /**
         * 可选值：{@link VRTCParams.VRTCStreamType}
         * 当发起桌面共享时，无需传入该值，内部会自动维护
         */
        public int streamType;
        /**
         * 用在旁路中显示的昵称
         * 目前仅支持VRTC
         */
        public String nickName;
    }

    /**
     * 互动推流视频参数配置
     */
    public static class VRTCVideoEncodeParam {

        public VRTCVideoEncodeParam() {
        }

        /**
         * 推荐使用该构造器，其他参数由SDK内部自行根据最佳实践配置
         *
         * @param videoResolution     分辨率预定义类型
         * @param videoResolutionMode 横竖屏模式
         */
        public VRTCVideoEncodeParam(int videoResolution, int videoResolutionMode) {
            this.videoResolution = videoResolution;
            this.videoResolutionMode = videoResolutionMode;
        }

        /**
         * 推荐的配置，**但优先级低于 videoWidth/videoHeight**
         * 视频分辨率，参考{@link VRTCCode.RTCVideoProfile#PROFILE_1920_1080_16x9}
         */
        public int videoResolution;

        /**
         * 视频分辨率横竖屏配置，参考{@link VRTCCode#RTC_VIDEO_PROFILE_MODE_LANDSCAPE}
         */
        public int videoResolutionMode;

        /**
         * 弱网下的清晰度/流畅度处理策略，默认平衡模式[当前VRTC仅支持平衡模式]
         */
        public int resolutionStrategy = VRTCCode.RTCDegradationMode.BALANCED;

        /**
         * 有值时，videoResolution不再生效，同时需要手动配置以下其他参数
         */
        public int videoWidth;

        /**
         * 有值时，videoResolution不再生效，同时需要手动配置以下其他参数
         */
        public int videoHeight;

        /**
         * 视频帧率【不建议配置，由SDK根据分辨率自行匹配最优值】
         */
        public int videoFps;

        /**
         * 视频码率【不建议配置，由SDK根据分辨率自行匹配最优值】
         */
        public int videoBitrate;

        /**
         * 视频最小码率【不建议配置，由SDK根据分辨率自行匹配最优值】
         */
        public int minVideoBitrate;

        /**
         * 视频最大码率【不建议配置，由SDK根据分辨率自行匹配最优值】
         */
        public int maxVideoBitrate;
        /**
         * trtc视频分辨率值，外部设置无效，由sdk内部维护
         */
        public int videoResolution2TRTC;
    }

    /**
     * 给业务侧使用的互动房间基本配置信息
     */
    public static class VRTCRoomParams {

        /**
         * 互动房间号
         * vrtc: inav_roomid
         * trtc: strRoomId
         */
        public String inavRoomId = "";
        /**
         * 旁路房间号
         */
        public String broadcastRoomId = "";
        /**
         * token of paas sdk
         */
        public String accessToken = "";
        /**
         * 定义见：{@link VRTCCode#ROLE_HOST}
         */
        public String role = VRTCCode.ROLE_HOST;

        public VRTCRoomParams() {
        }

        /**
         * 不需要推旁路的业务场景可以调该方法快速创建
         */
        public VRTCRoomParams(String inavRoomId, String accessToken, String role) {
            this.inavRoomId = inavRoomId;
            this.accessToken = accessToken;
            this.role = role;
        }

        public VRTCRoomParams(String inavRoomId, String accessToken, String broadcastRoomId, String role) {
            this.inavRoomId = inavRoomId;
            this.accessToken = accessToken;
            this.broadcastRoomId = broadcastRoomId;
            this.role = role;
        }
    }

    /**
     * sdk内部专用
     */
    public static class VRTCRoomParamsInner {

        public VRTCRoomParams roomParams;
        /**
         * 定义见：{@link VRTCCode#MODE_RTC}
         */
        public String mode = VRTCCode.MODE_RTC;

        /**
         * 是否开启软件降噪
         */
        public boolean softEchoCanceller = false;
        /**
         * 是否开启内存/网络资源过载进而降低分辨率/帧率/码率策略【目前没有实现方案，设置无效，默认true】
         */
        public boolean cpuoverusedetection = true;

        /**
         * 房间类型 1:vrtc 2:trtc
         */
        public String type = RTCTYPE_VRTC;
        /**
         * TRTC only
         */
        public int trtcSdkAppId = 0;

        /**
         * token of inav room
         * vrtc: inav_token
         * trtc: userSig
         */
        public String inavToken = "";

        /**
         * 业务层userid，VhallBaseSDK初始化时参数
         * 该参数目前用来辅助生成TRTC专属推旁路时生成stream_id, http://yapi.vhallops.com/project/48/interface/api/54002
         */
        public String userId = "";

        public VRTCRoomParamsInner() {
        }

        public VRTCRoomParamsInner(VRTCRoomParams roomParams) {
            this.roomParams = roomParams;
        }

        /**
         * param copy
         *
         * @param originParam
         */
        public VRTCRoomParamsInner(VRTCRoomParamsInner originParam) {
            this.roomParams = originParam.roomParams;
            this.trtcSdkAppId = originParam.trtcSdkAppId;
            this.inavToken = originParam.inavToken;
            this.type = originParam.type;
            this.softEchoCanceller = originParam.softEchoCanceller;
            this.cpuoverusedetection = originParam.cpuoverusedetection;
            if (!TextUtils.isEmpty(originParam.roomParams.broadcastRoomId)) {
                mode = VRTCCode.MODE_LIVE;
            }
        }
    }

    /**
     * 旁路参数 http://wiki.vhallops.com/pages/viewpage.action?pageId=200933942
     */
    public static class VRTCBroadcastParam {
        /**
         * 旁路视频分辨率，见: {@link VRTCCode.BroadcastVideoProfile}
         */
        public int profile = VRTCCode.BroadcastVideoProfile.BROADCAST_VIDEO_PROFILE_720P_1;

        /**
         * 布局模版，见: {@link VRTCCode.BroadcastLayout}
         */
        public int layoutMode = VRTCCode.BroadcastLayout.CANVAS_LAYOUT_PATTERN_TILED_5_1T4D;
        /**
         * 选填，是否有占坑图标，默认为否。 用于标识填充布局模板种的多余块，图片为统一设计小人头图标，暂不支持自定义图片。该配置项变化时，会立即生效
         */
        public boolean precast_pic_exist = false;

        /**
         * 是否显示旁路昵称[前提：创建本地流时配置nickName]，默认true
         */
        public boolean nickName = true;

        /**
         * 窗格裁剪模式
         * 0: 不裁剪，不同比例下高or宽可能会有黑边
         * 1: 裁剪后铺满
         * 2: 拉伸铺满
         * 仅支持VRTC
         */
        public int cropType = 0;

        /**
         * 背景色，支持格式如： 0x000000、#000000、000000
         */
        public String backgroundColor;
        /**
         * 背景图。
         * <p>
         * VRTC:
         * 1.不支持在开始旁路直播时通过旁路直播参数配置，需要独立调用设置背景图api进行设置 {@link com.vhall.ilss.VHInteractiveV2#setMixBackgroundImage(String, int, ICallback)}
         * 2.仅支持https
         */
        @Deprecated
        public String backgroundImageUrl;

        /**
         * TRTC: 暂不支持
         */
        public Border border;

        /**
         * 背景边框
         */
        public static class Border {
            /**
             * 是否有边框，服务侧default = false
             */
            public boolean exist = false;
            /**
             * 边框宽度，服务侧默认4px，最大8px
             */
            public int width = 4;
            /**
             * 边框颜色，支持格式： 0x000000、#000000、000000；服务侧默认为0x555657
             */
            public String color = "";
            /**
             * 边框是否透明
             */
            public boolean transparent = false;

            public JSONObject convertToJson() {
                JSONObject result = new JSONObject();
                try {
                    result.put("exist", exist);
                    result.put("width", width);
                    result.put("color", color);
                    result.put("transparency", transparent ? 100 : 0);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
                return result;
            }

            public String convertColor2VRTCStyle() {
                return Utils.compatColor2HexStyle(color);
            }

            public Border copy() {
                Border newBorder = new Border();
                newBorder.color = this.color;
                newBorder.width = this.width;
                newBorder.exist = this.exist;
                return newBorder;
            }
        }
    }

    public static class VRTCBroadcastParamInner {
        public VRTCBroadcastParam broadcastParam;
        /**
         * sdk内部使用，兼容处理后的旁路视频分辨率/帧率/码率模版
         */
        public JSONObject compat_profile_json;

        /**
         * sdk内部使用，兼容处理后的旁路布局模版
         */
        public int compat_layout_mode;
        /**
         * 是否是自适应布局模版
         */
        public boolean isAdaptiveLayoutMode = false;

        /**
         * 旁路推流地址
         */
        public String pushUrl;

        /**
         * 互动房间id
         */
        public String inavRoomId;
        /**
         * 旁路房间id
         */
        public String broadcastRoomId;

        public VRTCBroadcastParamInner() {
        }

        public VRTCBroadcastParamInner(VRTCBroadcastParam broadcastParam) {
            this.broadcastParam = broadcastParam;
        }

        /**
         * param copy
         *
         * @param originParam
         */
        public VRTCBroadcastParamInner(VRTCBroadcastParamInner originParam) {
            this.broadcastParam = originParam.broadcastParam;
            this.compat_layout_mode = originParam.compat_layout_mode;
            this.compat_profile_json = originParam.compat_profile_json;
            this.pushUrl = originParam.pushUrl;
        }
    }
}