package com.vhall.framework;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;

import com.vhall.httpclient.utils.OKHttpUtils;
import com.vhall.logmanager.BuildConfig;
import com.vhall.logmanager.VLog;

import java.io.File;
import java.io.IOException;

import okhttp3.Cache;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.FormBody;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okio.Buffer;
import okio.BufferedSink;
import okio.Okio;
import okio.Source;

/**
 * Created by Hank on 2017/11/29.
 */
public final class VHAPI {

    private static final String TAG = "VHAPI";
    private static final String client = "android";
    private static OkHttpClient mOkHttpClient;
    private static OkHttpClient uploadClient;

    public static OkHttpClient getOkHttpClient() {
        if (mOkHttpClient == null) {
            mOkHttpClient = OKHttpUtils.createOkClient();
        }
        return mOkHttpClient;
    }

    public static OkHttpClient getUploadOKHttpClient() {
        if (uploadClient == null) {
            uploadClient = OKHttpUtils.createOkClient();
        }
        return uploadClient;
    }

    public static FormBody.Builder getBaseBuilder(String accessToken) {
        FormBody.Builder builder = new FormBody.Builder()
                .add("client", client)
                .add("app_id", VhallBaseSDK.getInstance().APP_ID)
                .add("sign_type", "2")
                .add("package_check", VhallBaseSDK.getInstance().packageCheck)
                .add("third_party_user_id", TextUtils.isEmpty(VhallBaseSDK.getInstance().mUserId) ? "" : VhallBaseSDK.getInstance().mUserId)
                .add("uuid", TextUtils.isEmpty(VhallBaseSDK.getInstance().mIMEI) ? VhallBaseSDK.getInstance().mUserId : VhallBaseSDK.getInstance().mIMEI)
                .add("version_msg", BuildConfig.VERSION_NAME)
                .add("access_token", null == accessToken ? "" : accessToken);
        return builder;
    }

    public static MultipartBody.Builder getMultipartBodyBuild(String accessToken) {
        MultipartBody.Builder builder = new MultipartBody.Builder()
                .setType(MultipartBody.FORM)
                .addFormDataPart("client", client)
                .addFormDataPart("app_id", VhallBaseSDK.getInstance().APP_ID)
                .addFormDataPart("version_msg", BuildConfig.VERSION_NAME)
                .addFormDataPart("uuid", TextUtils.isEmpty(VhallBaseSDK.getInstance().mIMEI) ? VhallBaseSDK.getInstance().mUserId : VhallBaseSDK.getInstance().mIMEI)
                .addFormDataPart("sign_type", "2")
                .addFormDataPart("package_check", VhallBaseSDK.getInstance().packageCheck)
                .addFormDataPart("third_party_user_id", TextUtils.isEmpty(VhallBaseSDK.getInstance().mUserId) ? "" : VhallBaseSDK.getInstance().mUserId)
                .addFormDataPart("access_token", null == accessToken ? "" : accessToken);
        return builder;
    }

    public static Request getRequest(String command, RequestBody body) {
        Request request = new Request.Builder()
                .addHeader("atom", VhallBaseSDK.getInstance().mAtom)
                .url(VhallBaseSDK.getInstance().mAPIHost + command)
                .post(body)
                .build();
        return request;
    }

    public static Request getRequest(String host, String command, RequestBody body) {
        Request request = new Request.Builder()
                .addHeader("atom", VhallBaseSDK.getInstance().mAtom)
                .url(host + command)
                .post(body)
                .build();
        return request;
    }


    /**
     * 初始化SDK
     * 获取初始化数据
     */
    public static void initSDK(String accessToken, Callback callback) {
        FormBody.Builder builder = getBaseBuilder(accessToken);
        Request request = getRequest("v2/init/start", builder.build());
        getOkHttpClient()
                .newCall(request)
                .enqueue(callback);
    }

    /**
     * 消息链接加入
     *
     */
    public static void joinChannelId(String channelId, String accessToken, Callback callback) {
        FormBody.Builder builder = getBaseBuilder(accessToken);
        if (!TextUtils.isEmpty(channelId))
            builder.add("channel_id", channelId);
        Request request = getRequest("v2/init/start", builder.build());
        getOkHttpClient()
                .newCall(request)
                .enqueue(callback);
    }

    /**
     * 更新消息链接 的用户id
     * <p>
     * 设置用户信息 用户id改变的时候使用
     */
    public static void updateUserId(String channelId, String accessToken, Callback callback) {
        FormBody.Builder builder = getBaseBuilder(accessToken);
        if (!TextUtils.isEmpty(channelId))
            builder.add("channel_id", channelId);
        Request request = getRequest("v2/init/start", builder.build());
        getOkHttpClient()
                .newCall(request)
                .enqueue(callback);
    }


    /**
     * 发消息
     *
     * @param content   消息内容
     * @param type      消息类型
     * @param channelId 频道ID
     * @param callback  回调信息
     */
    public static void sendMsg(String content, String accessToken, String type, String channelId, String roomid, Callback callback) {
        sendMsg(content, accessToken, type, channelId, roomid, "0", callback);
    }

    /**
     * @param content     消息内容
     * @param accessToken
     * @param type        消息类型
     * @param channelId   频道ID
     * @param roomid
     * @param noAudit     消息是否走智能审核
     * @param callback    回调信息
     */
    public static void sendMsg(String content, String accessToken, String type, String channelId, String roomid, String noAudit, Callback callback) {
        sendMsg(content, accessToken, type, channelId, roomid, noAudit, VhallBaseSDK.getInstance().getmUserInfo(), callback);
    }


    /**
     * @param content     消息内容
     * @param accessToken
     * @param type        消息类型
     * @param channelId   频道ID
     * @param roomid
     * @param noAudit     消息是否走智能审核
     * @param context     上下文环境信息
     * @param callback    回调信息
     */
    public static void sendMsg(String content, String accessToken, String type, String channelId, String roomid, String noAudit, String context, Callback callback) {
        FormBody.Builder builder = getBaseBuilder(accessToken)
                .add("body", content)
                .add("type", type)
                .add("context", context)
                .add("no_audit", noAudit);
        if (!TextUtils.isEmpty(channelId)) {
            builder.add("channel_id", channelId);
        }
        if (!TextUtils.isEmpty(roomid))
            builder.add("room_id", roomid);
        Call call = getOkHttpClient().newCall(getRequest("v2/message/send", builder.build()));
        if (callback == null)
            call.enqueue(new DefaultCallback());
        else
            call.enqueue(callback);
    }

    //带缓存的client构建
    public static OkHttpClient mCacheClient;

    public static OkHttpClient getCacheClient(Context context) {
        if (mCacheClient == null) {
            Cache mCache = new Cache(context.getCacheDir(), 10240 * 1024);
            mCacheClient = OKHttpUtils.createOkClient();
        }
        return mCacheClient;
    }

    public static class DefaultCallback implements Callback {

        @Override
        public void onFailure(Call call, IOException e) {
            VLog.e(TAG, e.getMessage());
        }

        @Override
        public void onResponse(Call call, Response response) throws IOException {
            String result = response.body().string();
            VLog.d(TAG, result);
        }
    }

    public static class ProgressRequestBody extends RequestBody {

        public interface ProgressListener {
            void onProgress(long totalBytes, long remainingBytes, boolean done);
        }

        protected File file;
        protected ProgressListener listener;
        protected String contentType;
        private Handler handler;

        public ProgressRequestBody(String contentType, File file, ProgressListener listener) {
            this.file = file;
            this.contentType = contentType;
            this.listener = listener;
            handler = new Handler(Looper.getMainLooper());
        }

        @Override
        public MediaType contentType() {
            return MediaType.parse(contentType);
        }

        @Override
        public long contentLength() throws IOException {
            return file.length();
        }


        @Override
        public void writeTo(BufferedSink sink) throws IOException {
            Source source;
            try {
                source = Okio.source(file);
                Buffer buf = new Buffer();
                final Long[] remaining = {contentLength()};
                final Long total = contentLength();
                for (long readCount; (readCount = source.read(buf, 2048)) != -1; ) {
                    sink.write(buf, readCount);
                    if (listener != null) {
                        final long finalReadCount = readCount;
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                listener.onProgress(total, remaining[0] -= finalReadCount, remaining[0] == 0);
                            }
                        });
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }
}
