package com.vhall.business.widget;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.RectF;
import android.graphics.Typeface;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.widget.ImageView;

import com.vhall.business.MessageServer;
import com.vhall.business.R;
import com.vhall.business.utils.LogManager;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by huanan on 2017/3/24.
 * 文档白板展示基类
 */
public abstract class DocumentView extends ImageView {

    protected static final int TYPE_PATH = 1;
    protected static final int TYPE_TEXT = 4;
    protected static final int TYPE_OVAL = 20;
    protected static final int TYPE_RECT = 22;
    protected static final int TYPE_ARROW = 30;
    protected static final int TYPE_DARROW = 31;
    protected static final int TYPE_ANCHOR = 7;

    private static int defaultStrokeWidth = 2;

    private static final String TAG = "DocumentView";
    protected Context mContext;
    private Paint mPaint;

    float scaleX = 0.0f;
    float scaleY = 0.0f;
    float transX = 0;
    float transY = 0;

    public DocumentView(Context context) {
        super(context);
        init(context);
    }

    public DocumentView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init(context);
    }

    public DocumentView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context);
    }

    protected void init(Context context) {
        mContext = context;
        this.mPaint = new Paint();
        this.mPaint.setAntiAlias(true);
        this.mPaint.setStyle(Paint.Style.STROKE);
        this.setScaleType(ScaleType.FIT_CENTER);
    }

    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        super.onLayout(changed, left, top, right, bottom);
        calcRT();
    }


    public abstract void setSteps(String key, List<MessageServer.MsgInfo> list);

    public abstract void setStep(MessageServer.MsgInfo msgInfo);

    protected void paint(MessageServer.MsgInfo msgInfo, Canvas canvas) {//1 表示白板画笔 4 表示白板文字 7表示白板锚点 20表示圆形 22表示矩形
        if (msgInfo == null || msgInfo.step == null) {
            return;
        }
        MessageServer.Step step = msgInfo.step;
//        mPaint.setColor(color);
//        mPaint.setAlpha(0xB2);
        mPaint.setARGB(0xB2, Color.red(step.color), Color.green(step.color), Color.blue(step.color));
//        mPaint.setARGB(0x3f, (step.color >> 16 & 0xff) / 255, (step.color >> 16 & 0xff) / 255, (step.color >> 16 & 0xff) / 255);
        switch (step.type) {
            case TYPE_PATH://path
                mPaint.setStrokeWidth(step.lineSize);
                Path path = new Path();
                ArrayList<MessageServer.Position> points = step.points;
                if (points != null && points.size() > 0) {
                    for (int j = 0; j < points.size(); j++) {
                        if (j == 0) {
                            path.moveTo((float) points.get(j).x, (float) points.get(j).y);
                        } else {
                            path.lineTo((float) points.get(j).x, (float) points.get(j).y);
                        }
                    }
                }
                canvas.drawPath(path, mPaint);
                break;
            case TYPE_TEXT://text
                mPaint.setStrokeWidth(0);
                Typeface typeface = null;
                if (step.fb == 1 && step.fi == 0) {
                    typeface = Typeface.create(Typeface.DEFAULT_BOLD, Typeface.BOLD);
                } else if (step.fb == 1 && step.fi == 1) {
                    typeface = Typeface.create(Typeface.MONOSPACE, Typeface.BOLD_ITALIC);
                } else if (step.fb == 0 && step.fi == 1) {
                    typeface = Typeface.create(Typeface.MONOSPACE, Typeface.ITALIC);
                } else {
                    typeface = Typeface.create(Typeface.DEFAULT, Typeface.NORMAL);
                }
                mPaint.setTypeface(typeface);
                mPaint.setTextSize(step.fs);
                canvas.drawText(step.ft, (float) step.sPoint.x, (float) step.sPoint.y + step.fs, mPaint);
                break;
            case TYPE_OVAL://oval
                mPaint.setStrokeWidth(defaultStrokeWidth);
                float left, right, top, bottom;
                if (step.sPoint.x < step.tPoint.x) {
                    left = (float) step.sPoint.x;
                    right = (float) step.tPoint.x;
                } else {
                    left = (float) step.tPoint.x;
                    right = (float) step.sPoint.x;
                }
                if (step.sPoint.y < step.tPoint.y) {
                    top = (float) step.sPoint.y;
                    bottom = (float) step.tPoint.y;
                } else {
                    top = (float) step.tPoint.y;
                    bottom = (float) step.sPoint.y;
                }
                RectF rect = new RectF(left, top, right, bottom);
                canvas.drawOval(rect, mPaint);
                break;
            case TYPE_RECT://rectf
                mPaint.setStrokeWidth(defaultStrokeWidth);
                float leftF, rightF, topF, bottomF;
                if (step.sPoint.x < step.tPoint.x) {
                    leftF = (float) step.sPoint.x;
                    rightF = (float) step.tPoint.x;
                } else {
                    leftF = (float) step.tPoint.x;
                    rightF = (float) step.sPoint.x;
                }
                if (step.sPoint.y < step.tPoint.y) {
                    topF = (float) step.sPoint.y;
                    bottomF = (float) step.tPoint.y;
                } else {
                    topF = (float) step.tPoint.y;
                    bottomF = (float) step.sPoint.y;
                }
                RectF rectF = new RectF(leftF, topF, rightF, bottomF);
                canvas.drawRect(rectF, mPaint);
                break;
            case TYPE_ARROW://arrow
                mPaint.setStrokeWidth(defaultStrokeWidth);
                drawAL((int) step.sPoint.x, (int) step.sPoint.y, (int) step.tPoint.x, (int) step.tPoint.y, mPaint, canvas, false);
                break;
            case TYPE_DARROW:
                mPaint.setStrokeWidth(defaultStrokeWidth);
                drawAL((int) step.sPoint.x, (int) step.sPoint.y, (int) step.tPoint.x, (int) step.tPoint.y, mPaint, canvas, true);
                break;
            case TYPE_ANCHOR://anchor
                Bitmap bitmap = BitmapFactory.decodeResource(mContext.getResources(), R.drawable.point);
                if (step.color == 0) {
                    canvas.drawBitmap(bitmap, (float) step.sPoint.x, (float) step.sPoint.y, mPaint);
                } else {
                    canvas.drawBitmap(replaceBitmapColor(bitmap, step.color), (float) step.sPoint.x, (float) step.sPoint.y, mPaint);
                }

                break;

        }

    }

    //calc scale&trans
    private void calcRT() {
        Drawable d = getDrawable();
        if (d != null && d.getIntrinsicHeight() > 0 && d.getIntrinsicWidth() > 0) {
            int dwidth = d.getIntrinsicWidth();
            int dheight = d.getIntrinsicHeight();
            int vwidth = getWidth();
            int vheight = getHeight();
            LogManager.innerLog(TAG, "view.width:" + vwidth + " view.height:" + vheight + "drawable.width:" + dwidth + "drawable.height:" + dheight);
            if (vwidth * 1.0 / vheight > dwidth * 1.0 / dheight) {//fitY
                scaleX = scaleY = vheight * 1.0f / dheight;
                transX = (vwidth - dwidth * scaleX) * 0.5f;
                transY = 0;
            } else {
                scaleX = scaleY = vwidth * 1.0f / dwidth;
                transX = 0;
                transY = (vheight - dheight * scaleY) * 0.5f;
            }
        } else {
            scaleX = scaleY = 0;
            transX = transY = 0;
        }
        LogManager.innerLog(TAG, "scaleX:" + scaleX + " scaleY:" + scaleY + "transX:" + transX + "transY:" + transY);
    }

    /**
     * 绘制单箭头
     *
     * @param sx       start point x
     * @param sy       start point y
     * @param ex       end point x
     * @param ey       end point y
     * @param mPaint   画笔
     * @param canvas   画布
     * @param isdouble 是否双箭头
     */
    protected void drawAL(int sx, int sy, int ex, int ey, Paint mPaint, Canvas canvas, boolean isdouble) {
        if (sx == ex && sy == ey) {
            if (isdouble) {
                ex += 16;
            } else {
                ex += 8;
            }
            drawAL(sx, sy, ex, ey, mPaint, canvas, isdouble);
            return;
        }
        //绘制直线
        canvas.drawLine(sx, sy, ex, ey, mPaint);
        //绘制箭头
        double H = 8; // 箭头高度
        double L = 3.5; // 底边的一半
        double awrad = Math.atan(L / H); // 箭头角度
        double arraow_len = Math.sqrt(L * L + H * H); // 箭头的长度
        //计算下箭头并绘制
        double[] arrXY_1 = rotateVec(ex - sx, ey - sy, awrad, true, arraow_len);
        double[] arrXY_2 = rotateVec(ex - sx, ey - sy, -awrad, true, arraow_len);
        double x_3 = ex - arrXY_1[0]; // (x3,y3)是第一端点
        double y_3 = ey - arrXY_1[1];
        double x_4 = ex - arrXY_2[0]; // (x4,y4)是第二端点
        double y_4 = ey - arrXY_2[1];
        Double X3 = new Double(x_3);
        int x3 = X3.intValue();
        Double Y3 = new Double(y_3);
        int y3 = Y3.intValue();
        Double X4 = new Double(x_4);
        int x4 = X4.intValue();
        Double Y4 = new Double(y_4);
        int y4 = Y4.intValue();
        // 画线
        Path triangle = new Path();
        triangle.moveTo(ex, ey);
        triangle.lineTo(x3, y3);
        triangle.lineTo(x4, y4);
        triangle.close();
        canvas.drawPath(triangle, mPaint);
        if (isdouble) {
            double[] arrXY_3 = rotateVec(sx - ex, sy - ey, awrad, true, arraow_len);
            double[] arrXY_4 = rotateVec(sx - ex, sy - ey, -awrad, true, arraow_len);
            double x_5 = sx - arrXY_3[0];
            double y_5 = sy - arrXY_3[1];
            double x_6 = sx - arrXY_4[0];
            double y_6 = sy - arrXY_4[1];
            Path triangle2 = new Path();
            triangle2.moveTo(sx, sy);
            triangle2.lineTo((float) x_5, (float) y_5);
            triangle2.lineTo((float) x_6, (float) y_6);
            triangle2.close();
            canvas.drawPath(triangle2, mPaint);
        }

    }


    // 计算点坐标
    protected double[] rotateVec(int px, int py, double ang, boolean isChLen, double newLen) {
        double mathstr[] = new double[2];
        // 矢量旋转函数，参数含义分别是x分量、y分量、旋转角、是否改变长度、新长度
        double vx = px * Math.cos(ang) - py * Math.sin(ang);
        double vy = px * Math.sin(ang) + py * Math.cos(ang);
        if (isChLen) {
            double d = Math.sqrt(vx * vx + vy * vy);
            vx = vx / d * newLen;
            vy = vy / d * newLen;
            mathstr[0] = vx;
            mathstr[1] = vy;
        }
        return mathstr;
    }


    /**
     * 将bitmap中的某种颜色值替换成新的颜色
     *
     * @param oldBitmap bitmap
     * @param newColor 替换颜色
     * @return bitmap
     */
    protected Bitmap replaceBitmapColor(Bitmap oldBitmap, int newColor) {
        Bitmap mBitmap = oldBitmap.copy(Bitmap.Config.ARGB_8888, true);
        int mBitmapWidth = mBitmap.getWidth();
        int mBitmapHeight = mBitmap.getHeight();
        for (int i = 0; i < mBitmapHeight; i++) {
            for (int j = 0; j < mBitmapWidth; j++) {
                int color = mBitmap.getPixel(j, i);
                if (color != Color.TRANSPARENT) {
                    mBitmap.setPixel(j, i, Color.argb(Color.alpha(color), Color.red(newColor), Color.green(newColor), Color.blue(newColor)));  //将白色替换成透明色
                }
            }
        }
        return mBitmap;
    }


}
