package com.vhall.business;

import static com.vhall.ims.VHIM.TYPE_CHAT;
import static com.vhall.ims.VHIM.TYPE_CUSTOM;
import static com.vhall.ims.VHIM.TYPE_IMAGE;
import static com.vhall.ims.VHIM.TYPE_TEXT;

import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;

import com.vhall.business.data.RequestCallback;
import com.vhall.business.data.WebinarInfo;
import com.vhall.business.data.WebinarInfoRemoteDataSource;
import com.vhall.business.data.source.UserInfoRepository;
import com.vhall.business.data.source.WebinarInfoRepository;
import com.vhall.business.data.source.local.UserInfoLocalDataSource;
import com.vhall.business.data.source.remote.UserInfoRemoteDataSource;
import com.vhall.business.utils.LogManager;
import com.vhall.logmanager.VLog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.net.URLDecoder;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import okio.ByteString;
import vhall.com.vss2.data.MessageData;

/**
 * Created by huanan on 2016/8/23.
 */
public final class ChatServer {
    private static final String TAG = "ChatServer";
    public static final String eventOnlineKey = "online";// 上线
    public static final String eventOfflineKey = "offline";// 下线
    public static final String eventMsgKey = "msg";// 聊天
    public static final String eventCustomKey = "custom_broadcast";// 自定义消息
    public static final String eventQuestion = "question";//提问和回答
    public static final String eventPayKey = "pay";//打赏
    public static final String eventFavsNum = "favs_num";//打赏
    public static final String eventVirtualUpdate = "base_num_update";//虚拟人数增加

    private WebinarInfo webinarInfo;
    private Handler mDelivery = null;
    private Callback mCallback;

    private WebSocket ws;

    public interface Callback {
        void onChatServerConnected();

        void onConnectFailed();

        void onChatMessageReceived(ChatInfo chatInfo);

        void onChatServerClosed();

        void onChatServerKickOff();
    }


    public static class DefCallback implements Callback {

        @Override
        public void onChatServerConnected() {

        }

        @Override
        public void onConnectFailed() {

        }

        @Override
        public void onChatMessageReceived(ChatInfo chatInfo) {

        }

        @Override
        public void onChatServerClosed() {

        }
        @Override
        public void onChatServerKickOff(){

        }
    }

    public interface ChatRecordCallback {
        void onDataLoaded(List<ChatInfo> list);

        void onFailed(int errorcode, String messaage);
    }

    public ChatServer() {
        mDelivery = new Handler(Looper.getMainLooper());
    }

    protected boolean isAvaliable() {
        if (webinarInfo != null) {
            return true;
        }
        return false;
    }

    public void setWebinarInfo(WebinarInfo webinarInfo) {
        if (webinarInfo == null) {
            return;
        }
        this.webinarInfo = webinarInfo;
        connect();
    }

    private static class TrustAllCerts implements X509TrustManager {

        @Override
        public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {

        }

        @Override
        public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {

        }

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            return new X509Certificate[0];
        }
    }

    private static class TrustAllHostnameVerifier implements HostnameVerifier {
        @Override
        public boolean verify(String hostname, SSLSession session) {
            return true;
        }
    }

    private static SSLSocketFactory createSSLSocketFactory() {
        SSLSocketFactory ssfFactory = null;

        try {
            SSLContext sc = SSLContext.getInstance("TLS");
            sc.init(null, new TrustManager[]{new TrustAllCerts()}, new SecureRandom());
            ssfFactory = sc.getSocketFactory();
        } catch (Exception e) {
        }

        return ssfFactory;
    }


    public void connect() {
        disconnect();
        if (webinarInfo == null || mCallback == null || TextUtils.isEmpty(webinarInfo.chat_server)) {
            return;
        }
        OkHttpClient mOkHttpClient = new OkHttpClient.Builder()
                .sslSocketFactory(createSSLSocketFactory(), new TrustAllCerts())
                .hostnameVerifier(new TrustAllHostnameVerifier())
                .connectTimeout(0, TimeUnit.MILLISECONDS)
                .readTimeout(0, TimeUnit.MILLISECONDS)
                .writeTimeout(0, TimeUnit.MILLISECONDS).build();
        String url = webinarInfo.chat_server + "/ws/" + webinarInfo.webinar_id;
        final Request wsRequest = new Request.Builder().url(url).build();
        ws = mOkHttpClient.newWebSocket(wsRequest, new WebSocketListener() {
            @Override
            public void onOpen(WebSocket webSocket, Response response) {
                super.onOpen(webSocket, response);
                LogManager.innerLog(TAG, "chatServer on Open");
                mDelivery.post(new Runnable() {
                    @Override
                    public void run() {
                        mCallback.onChatServerConnected();
                    }
                });
            }

            @Override
            public void onMessage(WebSocket webSocket, final String text) {
                super.onMessage(webSocket, text);
                LogManager.innerLog(TAG, "chatServer on Message");
                if (TextUtils.isEmpty(text) || mCallback == null) {
                    return;
                }

                mDelivery.post(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            VLog.e("onChatMessageReceived", text);
                            JSONObject obj = new JSONObject(text);
                            ChatInfo chatInfo = getMessageInfo(obj, true);
                            if (chatInfo != null) {
                                if (chatInfo.questionData != null && chatInfo.questionData.answer != null) {
                                    if (chatInfo.questionData.answer.is_open == 1) {
                                        mCallback.onChatMessageReceived(chatInfo);
                                    } else {
                                        if (chatInfo.questionData.join_id.equals(webinarInfo.join_id)) {
                                            mCallback.onChatMessageReceived(chatInfo);
                                        }
                                    }
                                } else {
                                    mCallback.onChatMessageReceived(chatInfo);
                                }
                            }
                        } catch (JSONException e) {
                            e.printStackTrace();
                        }
                    }
                });

            }

            @Override
            public void onMessage(WebSocket webSocket, ByteString bytes) {
                super.onMessage(webSocket, bytes);
                LogManager.innerLog(TAG, "chatServer on Message:bytes");

            }

            @Override
            public void onClosing(WebSocket webSocket, int code, String reason) {
                super.onClosing(webSocket, code, reason);
                LogManager.innerLog(TAG, "chatServer onClosing");
            }

            @Override
            public void onClosed(WebSocket webSocket, int code, String reason) {
                super.onClosed(webSocket, code, reason);
                LogManager.innerLog(TAG, "code:" + code + " reason:" + reason);
                mDelivery.post(new Runnable() {
                    @Override
                    public void run() {
                        mCallback.onChatServerClosed();
                    }
                });
            }

            @Override
            public void onFailure(WebSocket webSocket, Throwable t, Response response) {
                super.onFailure(webSocket, t, response);
                if (!(t instanceof Exception)) {
                    return;
                }
                LogManager.innerLog(TAG, "ChatServer onFailure:" + t.getMessage());
                if (t.getMessage() != null && t.getMessage().equals("Socket closed")) {
                    mDelivery.post(new Runnable() {
                        @Override
                        public void run() {
                            mCallback.onChatServerClosed();
                        }
                    });
                } else {
                    mDelivery.post(new Runnable() {
                        @Override
                        public void run() {
                            mCallback.onConnectFailed();
                        }
                    });
                }

            }
        });
        mOkHttpClient.dispatcher().executorService().shutdown();
    }

    public void disconnect() {
        if (ws != null) {
            try {
                ws.close(1000, "closed by user");
            } catch (IllegalStateException e) {
                // websocket already closed
            }
            ws.cancel();
            ws = null;
        }
    }

    public void setCallback(Callback callback) {
        mCallback = callback;
    }


    //--------------------------------------------------------------------REQ BEGIN----------------------------------------------------------------------------------------
    public static class ChatInfo implements Serializable {

        public String account_id;
        public String third_account_id;
        public String user_name;
        public String avatar;
        public String room;
        public String event;
        public String time;
        public String role; // 角色聊天值：    评论值： host 主持人，user 观众，guest 助理，assistant 嘉宾
        public String roleName;//角色名称： 1(host):主持人 2(user)：观众  3(guest)：嘉宾 4(assistant)：助手
        public String id;   //目前用于评论，用于判断获取的评论是否重复
        public String name;   //目前用于传值
        public String url;   //目前用于传值

        public ChatData msgData;
        public OnlineData onlineData;
        public QuestionData questionData;
        public String unDifinedData;
        //                public PayData payData;
        public FavData favData;
        //        public LoginData loginData;
        public ReplyMsg replyMsg;
        private MessageData messageData;
        //原始消息
        public JSONObject mOriginData;

        //since 6.5.0
        public String msg_id;   //消息id

        public VirtualNumUpdateData virtualNumUpdateData;

        public MessageData getMessageData() {
            return messageData;
        }

        public void setMessageData(MessageData messageData) {
            this.messageData = messageData;
        }

        public static class ReplyMsg {
            public String type;
            public String avatar;
            public String account_id;
            public String user_name;
            public String role;
            public String roleName;
            public Content content;
            public String time;

            public static class Content {
                public String type;
                public String textContent;
            }
        }

        public List<AtListBean> atList;

        public static class AtListBean {
            public String nickName;
            public String accountId;
            public int index;
        }

        public static class ChatData {
            public String text;
            public String type;
            public String resourceUrl;
            public String target_id;//需要被过滤的消息
            public List<String> imageUrls = new ArrayList<>();
        }

        public static class OnlineData {
            public String role;//用户类型 host:主持人 guest：嘉宾 assistant：助手 user：观众
            public String roleName;//1234
            public int concurrent_user;// uv 房间内当前用户数
            public int is_gag;//是否被禁言
            public int attend_count;// pv参会人数
            public int tracksNum;   //pv
        }

        public static class VirtualNumUpdateData {

            public int update_online_num; //虚拟在线人数

            public int update_pv; //虚拟热度
        }

        public static class QuestionData {
            public String id;// 问题的id
            public String nick_name;// 昵称
            public String content;// 提问内容
            public String join_id;// 问题的id
            public String created_at;// 问题的时间11：20
            public String created_time;// 问题的全部时间 2011
            public String role_name="user";// 角色
            public String roleName="2";
            public int is_open;//是否为私密回答
            public String avatar;
            public String type;
            public QuestionData answer;
        }

        public static class FavData {
            public String favs;
        }
    }

    public void sendChat(String content, final RequestCallback callback) {

        if (!isAvaliable() || TextUtils.isEmpty(webinarInfo.msg_token) || TextUtils.isEmpty(webinarInfo.publish_server)) {
            VhallCallback.ErrorCallback(callback, ErrorCode.ERROR_INIT, VhallSDK.mContext.getString(R.string.error_video_info_init));
            return;
        }

        String url = webinarInfo.publish_server;
        final String token = webinarInfo.msg_token;
        UserInfoRepository userInfoRepository = UserInfoRepository.getInstance(UserInfoRemoteDataSource.getInstance(), UserInfoLocalDataSource.getInstance());
        userInfoRepository.sendChat(url, token, content, new RequestCallback() {
            @Override
            public void onSuccess() {
                if (callback != null) {
                    callback.onSuccess();
                }
            }

            @Override
            public void onError(int errorCode, String reason) {
                VhallCallback.ErrorCallback(callback, errorCode, reason);
            }
        });
    }

    public void sendCustom(JSONObject content, final RequestCallback callback) {
        long newMillis = System.currentTimeMillis();


        if (!isAvaliable() || TextUtils.isEmpty(webinarInfo.msg_token) || TextUtils.isEmpty(webinarInfo.publish_server)) {
            VhallCallback.ErrorCallback(callback, ErrorCode.ERROR_INIT, VhallSDK.mContext.getString(R.string.error_video_info_init));
            return;
        }
        String url = webinarInfo.publish_server;
        String token = webinarInfo.msg_token;
        UserInfoRepository userInfoRepository = UserInfoRepository.getInstance(UserInfoRemoteDataSource.getInstance(), UserInfoLocalDataSource.getInstance());
        userInfoRepository.sendCustom(url, token, content, new RequestCallback() {
            @Override
            public void onSuccess() {
                if (callback != null) {
                    callback.onSuccess();
                }
            }

            @Override
            public void onError(int errorCode, String reason) {
                VhallCallback.ErrorCallback(callback, errorCode, reason);
            }
        });
    }

    /**
     * 发送问题
     * 发送问题必须登陆获取userID,允许5分钟发送一次
     *
     * @param user_id  用户标识
     * @param content  问题内容
     * @param callback 回调
     */
    public void sendQuestion(String user_id, String content, final RequestCallback callback) {
        if (!isAvaliable() || TextUtils.isEmpty(user_id) || TextUtils.isEmpty(webinarInfo.webinar_id)) {
            VhallCallback.ErrorCallback(callback, ErrorCode.ERROR_INIT, "获取视频信息失败！");
            return;
        }
        String webinar_id = webinarInfo.webinar_id;
        UserInfoRepository userInfoRepository = UserInfoRepository.getInstance(UserInfoRemoteDataSource.getInstance(), UserInfoLocalDataSource.getInstance());
        userInfoRepository.sendQuestion(user_id, webinar_id, content, new RequestCallback() {
            @Override
            public void onSuccess() {
                if (callback != null) {
                    callback.onSuccess();
                }
            }

            @Override
            public void onError(int errorCode, String reason) {
                VhallCallback.ErrorCallback(callback, errorCode, reason);
            }
        });
    }

    public static ChatInfo getMessageInfo(JSONObject response, boolean text) {
        ChatInfo msgInfo = new ChatInfo();
        try {
            JSONObject obj = null;
            if (text) {
                obj = new JSONObject(URLDecoder.decode(response.optString("text")));
            } else {
                obj = response;
            }
            if (obj == null)
            //非新消息返回null
            {
                return null;
            }
            if (obj.has("service_type")) {//todo 新消息，处理聊天和自定义消息
                msgInfo.event = obj.optString("service_type");
                msgInfo.room = obj.optString("channel");
                msgInfo.time = obj.optString("date_time");
                switch (msgInfo.event) {
                    case TYPE_CHAT:
                        JSONObject context = new JSONObject(obj.optString("context"));
                        msgInfo.account_id = context.optString("account_id");
                        msgInfo.user_name = context.optString("user_name");
                        msgInfo.avatar = context.optString("avatar");
                        msgInfo.role = context.optString("role_name");
                        msgInfo.roleName = changeRoleToRoleName(msgInfo.role);
                        msgInfo.role = changeRoleNameToRole(msgInfo.roleName);
                        JSONObject data = new JSONObject(obj.optString("data"));
                        ChatInfo.ChatData msgData = new ChatInfo.ChatData();
                        msgData.type = data.optString("type");
                        switch (msgData.type) {
                            case TYPE_TEXT:
                                msgData.text = data.optString("text_content");
                                break;
                            case TYPE_IMAGE:
                                JSONArray array = data.optJSONArray("image_urls");
                                if (array != null && array.length() > 0) {
                                    msgData.resourceUrl = array.getString(0);
                                }
                                break;
                            /**
                             * time 20200610 目前仅支持图片类型
                             case TYPE_LINK:
                             msgData.resourceUrl = data.optString("link_url");
                             break;
                             case TYPE_VIDEO:
                             msgData.resourceUrl = data.optString("video_url");
                             break;
                             case TYPE_VOICE:
                             msgData.resourceUrl = data.optString("voice_url");
                             break;*/
                        }
                        msgInfo.msgData = msgData;
                        //todo 为了避免demo层即用户需要修改代码，将消息事件转换成旧消息事件
                        msgInfo.event = eventMsgKey;
                        break;
                    case TYPE_CUSTOM:
                        ChatInfo.ChatData customData = new ChatInfo.ChatData();
                        customData.text = obj.optString("data");
                        msgInfo.msgData = customData;
                        //todo 为了避免demo层即用户需要修改代码，将消息事件转换成旧消息事件
                        msgInfo.event = eventCustomKey;
                        break;
                    default:
                        break;
                }
            } else {//todo 旧消息，处理上下线事件、问答数据、打赏等信息
                msgInfo.account_id = obj.optString("account_id");
                msgInfo.user_name = obj.optString("user_name");
                msgInfo.avatar = obj.optString("avatar");
                msgInfo.event = obj.optString("event");
                msgInfo.room = obj.optString("room");
                msgInfo.time = obj.optString("time");
                msgInfo.role = obj.optString("role");
                msgInfo.roleName = changeRoleToRoleName(msgInfo.role);
                msgInfo.role = changeRoleNameToRole(msgInfo.roleName);
                JSONObject data = obj.optJSONObject("data");
                switch (msgInfo.event) {
                    case eventMsgKey:
                        if (!text) {//todo 获取历史数据解析
                            ChatInfo.ChatData msgData = new ChatInfo.ChatData();
                            msgData.text = data.optString("text");
                            msgData.type = TYPE_TEXT;
                            msgInfo.msgData = msgData;
                        } else {
                            return null;
                        }
                        break;
                    case eventOnlineKey:
                    case eventOfflineKey:
                        ChatInfo.OnlineData onlineData = new ChatInfo.OnlineData();
                        onlineData.role = data.optString("role");
                        onlineData.roleName = changeRoleToRoleName(onlineData.role);
                        onlineData.role = changeRoleNameToRole(onlineData.roleName);
                        onlineData.concurrent_user = data.optInt("concurrent_user");
                        onlineData.is_gag = data.optInt("is_gag");
                        onlineData.attend_count = data.optInt("attend_count");
                        onlineData.tracksNum = data.optInt("tracksNum");
                        msgInfo.onlineData = onlineData;
                        msgInfo.role = onlineData.role;
                        msgInfo.roleName = onlineData.roleName;
                        break;
                    case eventQuestion:
                        ChatInfo.QuestionData questionData = new ChatInfo.QuestionData();
                        questionData.id = data.optString("id");
                        questionData.nick_name = data.optString("nick_name");
                        questionData.content = data.optString("content");
                        questionData.join_id = data.optString("join_id");
                        questionData.created_at = data.optString("created_at");
                        questionData.role_name = data.optString("role_name");
                        questionData.roleName = changeRoleToRoleName(questionData.role_name);
                        questionData.is_open = data.optInt("is_open");
                        questionData.avatar = data.optString("avatar");
                        JSONObject answerObj = data.optJSONObject("answer");
                        if (answerObj != null) {
                            ChatInfo.QuestionData answerData = new ChatInfo.QuestionData();
                            answerData.id = answerObj.optString("id");
                            answerData.avatar = answerObj.optString("avatar");
                            answerData.nick_name = answerObj.optString("nick_name");
                            answerData.content = answerObj.optString("content");
                            answerData.join_id = answerObj.optString("join_id");
                            answerData.created_at = answerObj.optString("created_at");
                            answerData.role_name = answerObj.optString("role_name");
                            answerData.roleName = changeRoleToRoleName(answerData.role_name);
                            answerData.is_open = answerObj.optInt("is_open");
                            questionData.answer = answerData;
                        }
                        msgInfo.questionData = questionData;
                        break;
                    case eventFavsNum:
                        ChatInfo.FavData fav = new ChatInfo.FavData();
                        fav.favs = data.optString("favs");
                        msgInfo.favData = fav;
                        break;
                    case eventCustomKey:
                        ChatInfo.ChatData msgData = new ChatInfo.ChatData();
                        msgData.text = obj.optString("data");
                        msgData.type = TYPE_TEXT;
                        msgInfo.msgData = msgData;
                    default:
                        if (data != null) {
                            msgInfo.unDifinedData = data.toString();
                        } else {
                            msgInfo.unDifinedData = obj.optString("data");
                        }
                        break;
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            /*
             * 如果出现Json解析错误，msgInfo 直接返回null
             * 已发现异常：活动由回放转为直播，正在观看回放的用户发送评论，观看直播用户收到消息格式不正确；
             * 异常原因：返回data为jsonString,非JSONObject；获取data内容出现空指针异常
             */
            return null;
        }
        if (!TextUtils.isEmpty(msgInfo.avatar) && !msgInfo.avatar.startsWith("http")) {
            msgInfo.avatar = String.format("https:%s", msgInfo.avatar);
        }
        return msgInfo;
    }

    public static String changeRoleToRoleName(String role) {
        String roleName = "";
        switch (role) {
            case "1":
            case "host":
                roleName = "1";
                break;
            case "2":
            case "user":
                roleName = "2";
                break;
            case "3":
            case "assistant":
                roleName = "3";
                break;
            case "4":
            case "guest":
                roleName = "4";
                break;
        }
        return roleName;
    }

    public static String changeRoleNameToRole(String roleName) {
        String role = "";
        switch (roleName) {
            case "1":
            case "host":
                role = "host";
                break;
            case "2":
            case "user":
                role = "user";
                break;
            case "3":
            case "assistant":
                role = "assistant";
                break;
            case "4":
            case "guest":
                role = "guest";
                break;
        }
        return role;
    }


    public void acquireChatRecord(boolean show_all, final ChatRecordCallback chatRecordCallback) {
        if (chatRecordCallback == null) {
            throw new IllegalArgumentException(ErrorCode.CALLBACK_ERROR_STR);
        }
        if (!isAvaliable() || TextUtils.isEmpty(webinarInfo.join_id) || TextUtils.isEmpty(webinarInfo.webinar_id)) {
            chatRecordCallback.onFailed(ErrorCode.ERROR_INIT, "获取视频信息失败！");
            return;
        }
        WebinarInfoRepository webinarInfoRepository = WebinarInfoRepository.getInstance(WebinarInfoRemoteDataSource.getInstance());
        webinarInfoRepository.getChatHistory(webinarInfo.join_id, webinarInfo.webinar_id, show_all, new ChatRecordCallback() {
            @Override
            public void onDataLoaded(List<ChatInfo> list) {
                chatRecordCallback.onDataLoaded(list);
            }

            @Override
            public void onFailed(int errorcode, String messaage) {
                chatRecordCallback.onFailed(errorcode, messaage);
            }
        });
    }
}
