package com.vhall.business;


import android.text.TextUtils;

import com.vhall.business.data.RequestCallback;
import com.vhall.business.data.WebinarInfo;
import com.vhall.business.data.WebinarInfoRemoteDataSource;
import com.vhall.business.data.source.WebinarInfoDataSource;
import com.vhall.business.data.source.WebinarInfoRepository;
import com.vhall.business.utils.LogManager;
import com.vhall.player.Constants;
import com.vhall.push.LivePusher;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by zwp on 2018/12/25
 */
class BroadcastFlash extends Push {
    private static final String TAG = "BroadcastFlash";
    private int RECONNECT = 1; //重连次数，暂不开放设置

    //使用模块化Message
    private ChatServer chatServer;
    private MessageServer messageServer;
    //发直播
    private LivePusher mVhallPushLive;
    //dispatch url
    private DispatchParam dispatchParam;

    private int currentTimes = RECONNECT;


    @Override
    public void setWebinarInfo(WebinarInfo webinarInfo) {
        this.webinarInfo = webinarInfo;
        if (this.chatServer != null) {
            this.chatServer.setWebinarInfo(webinarInfo);
        }
        if (this.messageServer != null) {
            this.messageServer.setWebinarInfo(webinarInfo);
        }
    }

    public BroadcastFlash(Builder builder) {
        mVhallPushLive = new LivePusher(builder.videoCapture, builder.audioCapture, builder.config);
        this.streamOnly = builder.streamOnly;
        if (this.streamOnly) {
            //TODO 此处可能需要添加处理逻辑
            // liveParam 模块化中已经替换成VHLivePushConfig
//            this.liveParam = builder.param;
        } else {
            this.videoCapture = builder.videoCapture;
            this.mAudioCapture = builder.audioCapture;
            mVhallPushLive.openNoiseCancelling(builder.denoise);
            if (builder.config.pushReconnectTimes > 0) {
                RECONNECT = builder.config.pushReconnectTimes;//业务层处理重连
            }
            this.listener = builder.listener;
            mVhallPushLive.setListener(listener);
            this.chatServer = new ChatServer();
            chatServer.setCallback(builder.chatCallback);
            this.messageServer = new MessageServer();
            this.messageServer.setCallback(new MessageServer.Callback() {
                @Override
                public void onEvent(MessageServer.MsgInfo messageInfo) {

                }

                @Override
                public void onMsgServerConnected() {

                }

                @Override
                public void onConnectFailed() {

                }

                @Override
                public void onMsgServerClosed() {

                }
            });

        }
    }


    /**
     * 销毁当前直播实例
     */
    @Override
    public void destroy() {
        if (mVhallPushLive.getState() == Constants.State.START) {
            mVhallPushLive.release();
        }
        chatServer.disconnect();
    }

    /**
     * 开始直播
     */
    @Override
    public void start(RequestCallback callback) {
        if (mVhallPushLive.getState() == Constants.State.START) {
            listener.onError(ERROR_PLAYING, 0, VhallSDK.mContext.getString(R.string.living));
            return;
        }
        //推流调度
        refreshPublishURL();
        if (callback!=null){
            callback.onSuccess();
        }
    }

    /**
     * 停止直播
     */
    @Override
    public void stop() {
        if (mVhallPushLive.getState() == Constants.State.START) {
            mVhallPushLive.stop();
        }
    }

    /**
     * @param content  发送内容
     * @param callback 发送回调
     */
    @Override
    public void sendChat(String content, RequestCallback callback) {
        if (chatServer != null) {
            chatServer.sendChat(content, callback);
        }
    }

    @Override
    public void sendCustom(JSONObject content, RequestCallback callback) {
        if (chatServer != null) {
            chatServer.sendCustom(content, callback);
        }
    }

    @Override
    public void connectChatServer() {
        if (chatServer == null) {
            return;
        }
        if (!this.isAvaliable()) {
            return;
        }
        chatServer.connect();
    }

    @Override
    public void disconnectChatServer() {
        if (chatServer == null) {
            return;
        }
        if (!this.isAvaliable()) {
            return;
        }
        this.chatServer.disconnect();
    }

    @Override
    public boolean pushStream(RequestCallback callback){
        //推流调度
        refreshPublishURL();
        if (callback!=null){
            callback.onSuccess();
        }
        return true;
    }

    /**
     * @param show_all           false 显示当次直播聊天最多为20条,true显示所有聊天最条为20条，
     * @param chatRecordCallback 回调
     */
    @Override
    public void acquireChatRecord(boolean show_all, ChatServer.ChatRecordCallback chatRecordCallback) {
        chatServer.acquireChatRecord(show_all, chatRecordCallback);
    }

    @Override
    public void acquireChatRecord(int page, int limit, String msgId, String anchor_path, String is_role, ChatServer.ChatRecordCallback chatRecordCallback) {
        if (chatRecordCallback != null) {
            chatRecordCallback.onFailed(ErrorCode.ERROR_NO_SUPPORT, VhallSDK.mContext.getString(R.string.error_no_support));
        }
    }

    /**
     * push音频数据
     *
     * @param data      音频数据（aac编码的数据）注意要先传音频头
     * @param size      音频数据大小
     * @param type      数据类型 0代表视频头 1代表音频头 2代表音频数据 3代表I帧 4代表p帧 5代表b帧
     * @param timestamp 音频时间戳 单位MS
     * @return 0是成功，非0是失败
     */
    @Override
    public int PushAACDataTs(byte[] data, int size, int type, long timestamp) {
        if (mVhallPushLive != null) {
            return mVhallPushLive.pushAACDataTs(data, size, type, timestamp);
        }
        return -1;
    }

    /**
     * push视频数据
     *
     * @param data      视屏数据(h264编码的数据) 注意要先传视频头
     * @param size      视频数据的大小
     * @param type      数据类型 0代表视频头 1代表音频头 2代表音频数据 3代表I帧 4代表p帧 5代表b帧
     * @param timestamp 视频时间戳 单位MS
     * @return 0是成功，非0是失败
     */
    @Override
    public int PushH264DataTs(byte[] data, int size, int type, long timestamp) {
        if (mVhallPushLive != null) {
            return mVhallPushLive.pushH264DataTs(data, size, type, timestamp);
        }
        return -1;
    }

    @Override
    public int setVolumeAmplificateSize(float size) {
        if (mVhallPushLive != null) {
            return mVhallPushLive.setVolumeAmplificateSize(size);
        }
        return -1;
    }

    private void refreshPublishURL() {
        if (TextUtils.isEmpty(webinarInfo.dispatch_host)) {
            publish();
            return;
        }
        if (!webinarInfo.dispatch_host.endsWith("/")) {
            webinarInfo.dispatch_host = webinarInfo.dispatch_host + "/";
        }
        final String url = webinarInfo.dispatch_host
                + "api/dispatch_publish"
                + "?webinar_id=" + webinarInfo.webinar_id
                + "&stream_name=" + webinarInfo.webinar_id
                + "&webinar_type=1"
                + "&accesstoken=vhall"
                + "&client_type=3"
                + "&appkey=" + VhallSDK.APP_KEY + "&bu=0";
        LogManager.innerLog(TAG, url);
        WebinarInfoRepository repository = WebinarInfoRepository.getInstance(WebinarInfoRemoteDataSource.getInstance());
        repository.getPublishURL(url, new WebinarInfoDataSource.LoadWebinarInfoCallback() {
                    @Override
                    public void onWebinarInfoLoaded(String jsonStr, WebinarInfo webinarInfo) {
                        if (mVhallPushLive.getState() == Constants.State.START) {
                            return;
                        }
                        try {
                            JSONObject obj = new JSONObject(jsonStr);
                            String msg = obj.optString("msg");
                            int code = obj.optInt("code");
                            if (code == 200) {
                                dispatchParam = new DispatchParam();
                                JSONObject dataResult = obj.optJSONObject("data");
                                JSONArray urllist = dataResult.optJSONArray("publish_domainname");
                                if (urllist != null && urllist.length() > 0) {
                                    List<String> urls = new ArrayList<String>();
                                    for (int i = 0; i < urllist.length(); i++) {
                                        urls.add(urllist.optString(i));
                                    }
                                    dispatchParam.urls = urls;
                                }
                                JSONObject args = dataResult.optJSONObject("publish_args");
                                if (args != null) {
                                    dispatchParam.accesstoken = args.optString("accesstoken");
                                    dispatchParam.token = args.optString("token");
                                    dispatchParam.mixer = args.optString("mixer");
                                    dispatchParam.vhost = args.optString("vhost");
                                }
                            }
                        } catch (JSONException e) {
                            e.printStackTrace();
                        }
                        publish();

                    }

                    @Override
                    public void onError(int errorCode, String errorMsg) {
                        publish();
                    }
                }
        );
    }

    private void publish() {
        if (mVhallPushLive != null && mVhallPushLive.getState() == Constants.State.START) {
            return;
        }
        String publicURL = "";
        if (dispatchParam == null || dispatchParam.urls == null || dispatchParam.urls.size() <= 0) {
            publicURL = webinarInfo.media_srv + "?token=" + webinarInfo.streamtoken + "/" + webinarInfo.webinar_id;
        } else {
            publicURL = dispatchParam.urls.get(0)
                    + "?vhost=" + dispatchParam.vhost
                    + "?token=" + dispatchParam.token
                    + "?webinar_id=" + webinarInfo.webinar_id
                    + "?ismix=0?mixserver=" + dispatchParam.mixer
                    + "?accesstoken=" + dispatchParam.accesstoken
                    + "/" + webinarInfo.webinar_id + webinarInfo.user_id;
        }
        LogManager.innerLog(TAG, publicURL);
        mVhallPushLive.setLogParam(getDataCollectionStr());
        mVhallPushLive.start(publicURL);
    }


    private String getDataCollectionStr() {
        String dataCollection = "";
        if (webinarInfo != null && webinarInfo.data_report != null) {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("topic", webinarInfo.data_report.topic);
                jsonObject.put("sessionId", webinarInfo.session_id);
                jsonObject.put("bu", "0"); //区分业务单元，paas=1， saas=0或其它"
                jsonObject.put("guid", webinarInfo.data_report.guid);
                jsonObject.put("pf", 5);
                jsonObject.put("uid", webinarInfo.join_id);
                jsonObject.put("vfid", webinarInfo.data_report.vfid);
                jsonObject.put("vid", webinarInfo.data_report.vid);
                jsonObject.put("vtype", webinarInfo.data_report.vtype);
                jsonObject.put("aid", webinarInfo.webinar_id);
                jsonObject.put("app_id", VhallSDK.APP_KEY); //app_key
                jsonObject.put("host", webinarInfo.data_report.host); //添加默认值
                if (videoCapture != null) {
                    jsonObject.put("ndi", VhallSDK.getmIMEI());
                }
                dataCollection = jsonObject.toString();
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        LogManager.d(TAG, "dataCollection:" + dataCollection);
        return dataCollection;
    }


    private class DispatchParam {
        List<String> urls;
        String token = "";
        String mixer = "";
        String accesstoken = "";
        String vhost = "";
    }

}
