package com.vhall.business;


import android.content.Context;
import android.text.TextUtils;

import com.vhall.business.common.LogReportKs;
import com.vhall.business.common.LogReportManager;
import com.vhall.business.data.RequestCallback;
import com.vhall.business.data.WebinarInfo;
import com.vhall.ilss.VHInteractiveV2;
import com.vhall.lss_v2.VHV2LiveDef;
import com.vhall.lss_v2.push.IVHV2Pusher;
import com.vhall.lss_v2.push.VHV2PusherFactory;
import com.vhall.player.VHPlayerListener;
import com.vhall.push.VHAudioCapture;
import com.vhall.push.VHLivePushConfig;
import com.vhall.push.VHVideoCaptureView;

import org.json.JSONObject;

import static com.vhall.business.common.Constants.TYPE_FLASH;
import static com.vhall.business.common.Constants.TYPE_H5;
import static com.vhall.business.common.Constants.TYPE_H5_NEW;

/**
 * @author hkl
 */
public class Broadcast extends Push {
    private static final String TAG = "Broadcast";
    private int RECONNECT = 1; //重连次数，暂不开放设置

    //dispatch url
    private int currentTimes = RECONNECT;
    private VHPlayerListener listener;
    private Push mVhallPushLive;

    @Override
    public void setWebinarInfo(WebinarInfo webinarInfo) {
        initPush(webinarInfo);
    }

    private void initPush(WebinarInfo webinarInfo) {
        if (webinarInfo == null || mVhallPushLive != null) {
            if (listener != null) {
                listener.onError(ErrorCode.ERROR_INIT, ErrorCode.ERROR_INIT, "error webinarInfo");
            }
            return;
        }
        if (webinarInfo.no_delay_webinar == 1) {
            if (listener != null) {
                listener.onError(ErrorCode.ERROR_NO_DELAY, ErrorCode.ERROR_NO_DELAY, ErrorCode.ERROR_MSG_NO_DELAY);
            }
            return;
        }
        if(isBuilderV2 && webinarInfo.is_new_version != 3){
            if (listener != null) {
                listener.onError(ErrorCode.ERROR_WEBINAR_TYPE_ERROR, ErrorCode.ERROR_WEBINAR_TYPE_ERROR,"is_new_version is not 3");
            }
            return;
        }
        this.webinarInfo = webinarInfo;
        Push.Builder pushBuilder = new Push.Builder();
        pushBuilder.cameraView(this.videoCapture)
                .audioCapture(this.mAudioCapture)
                .callback(this.listener)
                .chatCallback(this.chatCallback)
                .messageCallback(this.messageCallback)
                .config(this.config)
                .stream(this.streamOnly)
                .isBuildV2(this.isBuilderV2)
                .setContext(this.mContext)
                .setInteractiveV2(this.vhLivePusherV2)
                .setLicenseKey(this.licenseKey)
                .setLicenseUrl(this.licenseUrl)
                .denoise(this.denoise);

        String type;
        if (webinarInfo.is_new_version == 3) {
            type = TYPE_H5_NEW;
        } else {
            type = TextUtils.isEmpty(webinarInfo.vss_token) ? TYPE_FLASH : TYPE_H5;
        }

        mVhallPushLive = pushBuilder.buildPush(type);
        if (mVhallPushLive != null) {
            mVhallPushLive.setWebinarInfo(webinarInfo);
        }
    }

    public Broadcast(final Builder builder) {
        this.isBuilderV2 = builder.isBuilderV2;
        this.streamOnly = builder.streamOnly;
        this.mContext = builder.mContext;
        this.config = builder.config;
        this.vhLivePusherV2 = builder.vhInteractiveV2;
        this.denoise = builder.denoise;
        this.chatCallback = builder.chatCallback;
        this.licenseUrl = builder.licenseUrl;
        this.licenseKey = builder.licenseKey;
        this.messageCallback = builder.messageCallback;
        this.videoCapture = builder.videoCapture;
        this.mAudioCapture = builder.audioCapture;
        if (builder.config.pushReconnectTimes > 0) {
            RECONNECT = builder.config.pushReconnectTimes;//业务层处理重连
        }
        this.listener = builder.listener;
        if(!builder.isBuilderV2 && this.videoCapture != null){
            this.videoCapture.init(this.config);
        }
    }


    /**
     * 发直播实例构造器
     */
    public static class Builder {
        VHVideoCaptureView videoCapture;
        IVHV2Pusher vhInteractiveV2;
        VHAudioCapture audioCapture;
        VHPlayerListener listener;
        VHLivePushConfig config;
        ChatServer.Callback chatCallback;
        //暂时没有使用
        MessageServer.Callback messageCallback;
        Context mContext;
        String licenseUrl;
        String licenseKey;
        boolean streamOnly;
        boolean denoise = true;
        boolean isBuilderV2 = false;

        public Builder stream(boolean streamOnly) {
            this.streamOnly = streamOnly;
            return this;
        }

        public Builder denoise(boolean denoise) {
            this.denoise = denoise;
            return this;
        }

        public Builder isBuildV2(boolean v2){
            this.isBuilderV2 = v2;
            return this;
        }

        public Builder cameraView(VHVideoCaptureView videoCapture) {
            this.videoCapture = videoCapture;
            return this;
        }

        public Builder setLicenseUrl(String url){
            this.licenseUrl = url;
            return this;
        }

        public Builder setLicenseKey(String key){
            this.licenseKey = key;
            return this;
        }

        public Builder audioCapture(VHAudioCapture audioCapture) {
            this.audioCapture = audioCapture;
            return this;
        }

        public Builder config(VHLivePushConfig config) {
            this.config = config;
            return this;
        }

        public Builder callback(VHPlayerListener listener) {
            this.listener = listener;
            return this;
        }

        public Builder chatCallback(ChatServer.Callback chatCallback) {
            this.chatCallback = chatCallback;
            return this;
        }

        public Builder messageCallback(MessageServer.Callback messageCallback) {
            this.messageCallback = messageCallback;
            return this;
        }

        public Builder setContext(Context context) {
            this.mContext = context;
            return this;
        }

        public Builder setInteractiveV2(IVHV2Pusher interactive){
            this.vhInteractiveV2 = interactive;
            return this;
        }


        public Broadcast build() {
            if (!streamOnly && (this.videoCapture == null || this.videoCapture.getActivity() == null)) {
                throw new IllegalArgumentException("init CameraView first...");
            }
            if (listener == null) {
                throw new IllegalArgumentException("Broadcast callback should not be null...");
            }
            if (config == null) {
                throw new IllegalArgumentException("PushConfig should not be null...");
            }
            if (mContext == null) {
                throw new IllegalArgumentException("mContext should not be null...");
            }
            if (audioCapture == null) {
                audioCapture = new VHAudioCapture();
            }
            if(isBuilderV2 && vhInteractiveV2 == null){
                vhInteractiveV2 = VHV2PusherFactory.newPusher(mContext, VHV2LiveDef.VHV2LiveBaseType.TXLIVE_TYPE);
            }
            LogReportManager.doReport(LogReportKs.K_LIVE_INIT);
            return new Broadcast(this);
        }
    }


    /**
     * 销毁当前直播实例
     */
    @Override
    public void destroy() {
        if(videoCapture !=null){
            videoCapture.releaseCapture();
        }
        if(vhLivePusherV2 != null){
            vhLivePusherV2.release();
            vhLivePusherV2 = null;
        }
        if (mVhallPushLive != null) {
            mVhallPushLive.destroy();
            LogReportManager.doReport(LogReportKs.K_LIVE_DESTROY);
        }
    }

    /**
     * 开始直播
     */
    @Override
    public void start(boolean rehearsal,RequestCallback callback) {
        if (webinarInfo == null) {
            if (listener != null) {
                listener.onError(ERROR_NOT_INIT, 0, VhallSDK.mContext.getString(R.string.error_video_msg_init));
            }
        }
        if (mVhallPushLive != null) {
            mVhallPushLive.start(rehearsal,callback);
        }
    }

    @Override
    public boolean pushStream(RequestCallback callback){
        if(mVhallPushLive != null){
           return mVhallPushLive.pushStream(callback);
        }
        return false;
    }

    /**
     * 开始直播
     */
    @Override
    public void start(RequestCallback callback) {
        if (webinarInfo == null) {
            if (listener != null) {
                listener.onError(ERROR_NOT_INIT, 0, VhallSDK.mContext.getString(R.string.error_video_msg_init));
            }
        }
        if (mVhallPushLive != null) {
            mVhallPushLive.start(callback);
            LogReportManager.doReport(LogReportKs.K_LIVE_START);
        }
    }

    /**
     * 停止直播
     */
    @Override
    public void stop() {
        if (mVhallPushLive != null) {
            mVhallPushLive.stop();
            LogReportManager.doReport(LogReportKs.K_LIVE_STOP);
        }
    }

    /**
     * @param content  发送内容
     * @param callback 发送回调
     */
    @Override
    public void sendChat(String content, RequestCallback callback) {
        if (!isAvaliable()) {
            if (callback != null) {
                callback.onError(ErrorCode.ERROR_INIT, VhallSDK.mContext.getString(R.string.error_msg_init));
            }
            return;
        }
        //主持人嘉宾不过滤，只过滤观众
//        if (webinarInfo.filters != null) {
//            for (int i = 0; i < webinarInfo.filters.size(); i++) {
//                if (content.contains(webinarInfo.filters.get(i))) {
//                    VhallCallback.ErrorCallback(callback, ErrorCode.ERROR_KEY_FILTERS, VhallSDK.mContext.getString(R.string.error_chat_forbid));
//                    return;
//                }
//            }
//        }
        if (mVhallPushLive != null) {
            mVhallPushLive.sendChat(content, callback);
        }
    }

    @Override
    public void sendCustom(JSONObject content, RequestCallback callback) {
        if (mVhallPushLive != null) {
            mVhallPushLive.sendCustom(content, callback);
        }
    }

    @Override
    public void connectChatServer() {
        if (mVhallPushLive != null) {
            mVhallPushLive.connectChatServer();
        }
    }

    @Override
    public void disconnectChatServer() {
        if (mVhallPushLive != null) {
            mVhallPushLive.disconnectChatServer();
        }
    }


    /**
     * @param show_all           false 显示当次直播聊天最多为20条,true显示所有聊天最条为100条，
     * @param chatRecordCallback 回调
     */
    @Override
    public void acquireChatRecord(boolean show_all, ChatServer.ChatRecordCallback chatRecordCallback) {
        if (mVhallPushLive != null) {
            mVhallPushLive.acquireChatRecord(show_all, chatRecordCallback);
        }
    }

    @Override
    public void acquireChatRecord(int page, int limit, String msgId, String anchor_path, String is_role, ChatServer.ChatRecordCallback chatRecordCallback) {
        if (mVhallPushLive != null)
            mVhallPushLive.acquireChatRecord(page, limit, msgId, anchor_path, is_role, chatRecordCallback);
    }

    /**
     * push音频数据
     *
     * @param data      音频数据（aac编码的数据）注意要先传音频头
     * @param size      音频数据大小
     * @param type      数据类型 0代表视频头 1代表音频头 2代表音频数据 3代表I帧 4代表p帧 5代表b帧
     * @param timestamp 音频时间戳 单位MS
     * @return 0是成功，非0是失败
     */
    @Override
    public int PushAACDataTs(byte[] data, int size, int type, long timestamp) {
        if (mVhallPushLive != null) {
            mVhallPushLive.PushAACDataTs(data, size, type, timestamp);
        }
        return -1;
    }

    /**
     * push视频数据
     *
     * @param data      视屏数据(h264编码的数据) 注意要先传视频头
     * @param size      视频数据的大小
     * @param type      数据类型 0代表视频头 1代表音频头 2代表音频数据 3代表I帧 4代表p帧 5代表b帧
     * @param timestamp 视频时间戳 单位MS
     * @return 0是成功，非0是失败
     */
    @Override
    public int PushH264DataTs(byte[] data, int size, int type, long timestamp) {
        if (mVhallPushLive != null) {
            return mVhallPushLive.PushH264DataTs(data, size, type, timestamp);
        }
        return -1;
    }

    @Override
    public int setVolumeAmplificateSize(float size) {
        if (mVhallPushLive != null) {
            return mVhallPushLive.setVolumeAmplificateSize(size);
        }
        return -1;
    }
}
