package com.proximities.sdk;

import android.content.Context;

import com.proximities.sdk.interfaces.LogDatabaseInterface;
import com.proximities.sdk.interfaces.LogInterface;
import com.proximities.sdk.json.model.log.BaseLogResponse;
import com.proximities.sdk.json.model.log.CampaignLogs;
import com.proximities.sdk.json.model.log.Log;
import com.proximities.sdk.json.model.partner.Poi;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.json.model.transmitter.Transmitter;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;

import java.util.List;

/**
 * Created by william on 13/12/16.
 */

public class LogsManager implements LogInterface, LogDatabaseInterface{

    private static final LogsManager INSTANCE = new LogsManager();

    private DatabaseManager mDbAccessor;
    private ProximitiesNetworkManager mNetworkManager;
    private boolean isSqliteEnabled = true;

    public static LogsManager getInstance() {
        return INSTANCE;
    }

    private LogsManager(){}

    void initLogsManager(Context context) {
        mNetworkManager = ProximitiesNetworkManager.getInstance();
        isSqliteEnabled = ProximitiesPrefs.readEnableSqlite(context);
        if(isSqliteEnabled){
            mDbAccessor = DatabaseManager.getInstance();
        }
    }

    /**
     * Create and send logs for each campaign
     * @param campaigns current campaigns
     * @param action action to send for each campaign
     *                   {@link com.proximities.sdk.util.ProximitiesConstants}
     * @param poiId
     * @param transmitterId
     */
    void generateLogs(List<Campaign> campaigns, String action, String poiId, String transmitterId){
        CampaignLogs campaignLogs = new CampaignLogs();
        for (Campaign campaign : campaigns) {
            String poiIdToSend = null;
            if(poiId != null){
                poiIdToSend = (poiId.equals(ProximitiesConstants.ORIGIN_MULTIPLE_POIS)) ? String.valueOf(campaign.getPoi().getId()) : poiId;
            }
            campaignLogs.addLog(new Log(campaign.getId(), action, Utils.getCurrentDate(), poiIdToSend, transmitterId));
        }
        /*if(isSqliteEnabled){
            mDbAccessor.insertLogsIntoDatabase(campaignLogs);
        }*/
//        LogUtils.LOGD("LogsManager", new Gson().toJson(campaignLogs));
        mNetworkManager.postLogs(this, null, campaignLogs);
    }

    void generateLogsWithoutOrigins(List<Campaign> campaigns, String action){
        CampaignLogs campaignLogs = new CampaignLogs();
        for (Campaign campaign : campaigns) {
            campaignLogs.addLog(new Log(campaign.getId(), action, Utils.getCurrentDate()));
        }
        /*if(isSqliteEnabled){
            mDbAccessor.insertLogsIntoDatabase(campaignLogs);
        }*/
        mNetworkManager.postLogs(this, null, campaignLogs);
    }

    void executeDatabaseReset(){
        if(isSqliteEnabled){
            mDbAccessor.executeDatabaseReset(this);
        }
    }

    void saveCampaignsInDatabase(Poi poi, Transmitter transmitter){
        if(isSqliteEnabled){
            mDbAccessor.saveCampaignInDatabase(poi, transmitter);
        }
    }

    @Override
    public void onSendLogSuccess(BaseLogResponse logs) {
//
    }

    @Override
    public void onSendLogError(CampaignLogs logs) {
        if(isSqliteEnabled){
            mDbAccessor.insertLogsIntoDatabase(logs);
        }
    }

    @Override
    public void onSendLogFromDatabaseSuccess() {
        if(isSqliteEnabled){
            mDbAccessor.deleteLogsFromDatabase();
        }
    }

    @Override
    public void onSendLogFromDatabaseError() {

    }

}
