/**
 * 
 */
package com.eaio.util;

import java.util.Arrays;
import java.util.Collection;

import org.apache.commons.lang.StringUtils;

/**
 * Removes reserved characters from file names. Should work for Windows and Linux.
 * 
 * @author <a href="mailto:jb@eaio.com">Johann Burkard</a>
 * @version $Id: SafeFileName.java 5084 2012-08-09 19:11:04Z johann $
 */
public class SafeFileName {

    private final Collection<String> reserved = Arrays.asList("aux", "com1", "com2", "com3", "com4", "com5", "com6", "com7",
            "com8", "com9", "lpt1", "lpt2", "lpt3", "lpt4", "lpt5", "lpt6", "lpt7", "lpt8", "lpt9", "con", "nul",
            "prn", ".", "..");

    /**
     * Returns a safe file name that should be safe for any file system. Returns a default if the name is <code>null</code> or 
     * really stinks.
     * 
     * @param in the user's crazy input value, may be <code>null</code>
     * @param def a default value, may be <code>null</code>
     * @return a String that's safe for the file system, only <code>null</code> if <tt>def</tt> is <code>null</code>
     */
    public String makeSafe(String in, String def) {
        String work = StringUtils.trimToNull(in);
        if (work == null) {
            return def;
        }

        StringBuilder builder = new StringBuilder(work.length());

        for (int i = 0; i < Math.min(255, work.length()); ++i) {
            char c = work.charAt(i);
            if (c > 0x1f) {
                switch (c) {
                    case '/':
                    case '\\':
                    case '?':
                    case '%':
                    case '*':
                    case ':':
                    case '|':
                    case '"':
                    case '<':
                    case '>':
                    case '^':
                        break;
                    case '-': if (builder.length() == 0) break; // $FALL-THROUGH$
                    case '.':
                    case ' ':
                        if (builder.length() == 0 || i == work.length() - 1 || i == 254)
                            break; // $FALL-THROUGH$
                    default:
                        builder.append(c);
                }
            }
        }
        
        if (builder.length() == 0) {
            return def;
        }
        
        String out = builder.toString();        
        String outLower = out.toLowerCase();
        for (String r : reserved) {
            if (outLower.startsWith(r)) {
                return def;
            }
        }

        return out;

    }

}
