package com.statsig.androidsdk

import com.google.gson.Gson
import kotlinx.coroutines.CoroutineExceptionHandler
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody
import okhttp3.RequestBody.Companion.toRequestBody
import java.lang.RuntimeException
import java.net.URL
import kotlin.math.floor

const val MAX_DIAGNOSTICS_MARKERS = 30
const val SAMPLING_RATE = 10_000

internal class ExternalException(message: String? = null) : Exception(message)

internal class ErrorBoundary() {
    internal var urlString = "https://statsigapi.net/v1/sdk_exception"

    private var apiKey: String? = null
    private var statsigMetadata: StatsigMetadata? = null
    private var seen = HashSet<String>()
    private var diagnostics: Diagnostics? = null

    fun setKey(apiKey: String) {
        this.apiKey = apiKey
    }

    fun setMetadata(statsigMetadata: StatsigMetadata) {
        this.statsigMetadata = statsigMetadata
    }

    fun setDiagnostics(diagnostics: Diagnostics) {
        this.diagnostics = diagnostics
        val sampled = floor(Math.random() * SAMPLING_RATE) == 0.0
        if (sampled) {
            diagnostics.setMaxMarkers(
                ContextType.API_CALL,
                MAX_DIAGNOSTICS_MARKERS,
            )
        } else {
            diagnostics.setMaxMarkers(
                ContextType.API_CALL,
                0,
            )
        }
    }

    private fun handleException(exception: Throwable) {
        println("[Statsig]: An unexpected exception occurred.")
        println(exception)
        if (exception !is ExternalException) {
            this.logException(exception)
        }
    }

    fun getExceptionHandler(): CoroutineExceptionHandler {
        return CoroutineExceptionHandler { _, exception ->
            this.handleException(exception)
        }
    }

    fun capture(task: () -> Unit, tag: String? = null, recover: (() -> Unit)? = null, configName: String? = null) {
        var markerID = ""
        try {
            markerID = startMarker(tag, configName) ?: ""
            task()
            endMarker(tag, markerID, true, configName)
        } catch (e: Exception) {
            endMarker(tag, markerID, false, configName)
            handleException(e)
            recover?.let { it() }
        }
    }

    suspend fun <T> captureAsync(task: suspend () -> T): T? {
        return try {
            task()
        } catch (e: Exception) {
            handleException(e)
            null
        }
    }

    suspend fun <T> captureAsync(task: suspend () -> T, recover: (suspend (e: Exception) -> T)): T {
        return try {
            task()
        } catch (e: Exception) {
            handleException(e)
            recover(e)
        }
    }

    internal fun logException(exception: Throwable) {
        try {
            if (apiKey == null) {
                return
            }

            val name = exception.javaClass.canonicalName ?: exception.javaClass.name
            if (seen.contains(name)) {
                return
            }

            seen.add(name)

            val metadata = statsigMetadata ?: StatsigMetadata("")
            val url = URL(urlString)
            val body = mapOf(
                "exception" to name,
                "info" to RuntimeException(exception).stackTraceToString(),
                "statsigMetadata" to metadata,
            )
            val postData = Gson().toJson(body)

            val clientBuilder = OkHttpClient.Builder()

            clientBuilder.addInterceptor(RequestHeaderInterceptor(apiKey ?: ""))
            clientBuilder.addInterceptor(ResponseInterceptor())

            val httpClient = clientBuilder.build()

            val requestBody: RequestBody = postData.toRequestBody(JSON)
            val request: Request = Request.Builder()
                .url(url)
                .post(requestBody)
                .build()
            httpClient.newCall(request).execute()
        } catch (e: Exception) {
            // noop
        }
    }

    private fun startMarker(tag: String?, configName: String?): String? {
        val diagnostics = this.diagnostics
        val markerKey = KeyType.convertFromString(tag ?: "")
        if (tag == null || diagnostics == null || markerKey == null) {
            return null
        }
        val markerID = tag + "_" + diagnostics.getMarkers(ContextType.API_CALL).count()
        diagnostics.diagnosticsContext = ContextType.API_CALL
        diagnostics.markStart(markerKey, step = null, Marker(markerID = markerID, configName = configName))
        return markerID
    }

    private fun endMarker(tag: String?, markerID: String?, success: Boolean, configName: String?) {
        val diagnostics = this.diagnostics
        val markerKey = KeyType.convertFromString(tag ?: "")
        if (tag == null || diagnostics == null || markerKey == null) {
            return
        }
        diagnostics.markEnd(markerKey, success, step = null, Marker(markerID = markerID, configName = configName))
    }
}
