/*
 * Decompiled with CFR 0.152.
 */
package oshi.hardware.platform.mac;

import com.sun.jna.Pointer;
import com.sun.jna.PointerType;
import com.sun.jna.platform.mac.CoreFoundation;
import com.sun.jna.platform.mac.DiskArbitration;
import com.sun.jna.platform.mac.IOKit;
import com.sun.jna.platform.mac.IOKitUtil;
import com.sun.jna.platform.mac.SystemB;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import javax.annotation.concurrent.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import oshi.hardware.HWDiskStore;
import oshi.hardware.HWPartition;
import oshi.util.ExecutingCommand;
import oshi.util.ParseUtil;

@ThreadSafe
public final class MacDisks {
    private static final CoreFoundation CF = CoreFoundation.INSTANCE;
    private static final DiskArbitration DA = DiskArbitration.INSTANCE;
    private static final Logger LOG = LoggerFactory.getLogger(MacDisks.class);

    private MacDisks() {
    }

    private static Map<CFKey, CoreFoundation.CFStringRef> mapCFKeys() {
        EnumMap<CFKey, CoreFoundation.CFStringRef> keyMap = new EnumMap<CFKey, CoreFoundation.CFStringRef>(CFKey.class);
        for (CFKey cfKey : CFKey.values()) {
            keyMap.put(cfKey, CoreFoundation.CFStringRef.createCFString((String)cfKey.getKey()));
        }
        return keyMap;
    }

    private static Map<String, String> queryMountPointMap() {
        HashMap<String, String> mountPointMap = new HashMap<String, String>();
        int numfs = SystemB.INSTANCE.getfsstat64(null, 0, 0);
        SystemB.Statfs[] fs = new SystemB.Statfs[numfs];
        SystemB.INSTANCE.getfsstat64(fs, numfs * new SystemB.Statfs().size(), 16);
        for (SystemB.Statfs f : fs) {
            String mntFrom = new String(f.f_mntfromname, StandardCharsets.UTF_8).trim();
            mountPointMap.put(mntFrom.replace("/dev/", ""), new String(f.f_mntonname, StandardCharsets.UTF_8).trim());
        }
        return mountPointMap;
    }

    private static Map<String, String> queryLogicalVolumeMap() {
        HashMap<String, String> logicalVolumeMap = new HashMap<String, String>();
        HashSet<String> physicalVolumes = new HashSet<String>();
        boolean logicalVolume = false;
        for (String line : ExecutingCommand.runNative("diskutil cs list")) {
            if (line.contains("Logical Volume Group")) {
                physicalVolumes.clear();
                logicalVolume = false;
                continue;
            }
            if (line.contains("Logical Volume Family")) {
                logicalVolume = true;
                continue;
            }
            if (!line.contains("Disk:")) continue;
            String volume = ParseUtil.parseLastString(line);
            if (logicalVolume) {
                for (String pv : physicalVolumes) {
                    logicalVolumeMap.put(pv, volume);
                }
                physicalVolumes.clear();
                continue;
            }
            physicalVolumes.add(ParseUtil.parseLastString(line));
        }
        return logicalVolumeMap;
    }

    private static boolean updateDiskStats(HWDiskStore diskStore, DiskArbitration.DASessionRef session, Map<String, String> mountPointMap, Map<String, String> logicalVolumeMap, Map<CFKey, CoreFoundation.CFStringRef> cfKeyMap) {
        IOKit.IOIterator driveListIter;
        String bsdName = diskStore.getName();
        CoreFoundation.CFMutableDictionaryRef matchingDict = IOKitUtil.getBSDNameMatchingDict((String)bsdName);
        if (matchingDict != null && (driveListIter = IOKitUtil.getMatchingServices((CoreFoundation.CFDictionaryRef)matchingDict)) != null) {
            IOKit.IORegistryEntry drive = driveListIter.next();
            if (drive != null) {
                if (drive.conformsTo("IOMedia")) {
                    IOKit.IORegistryEntry parent = drive.getParentEntry("IOService");
                    if (parent != null && parent.conformsTo("IOBlockStorageDriver")) {
                        CoreFoundation.CFMutableDictionaryRef properties = parent.createCFProperties();
                        Pointer result = properties.getValue((PointerType)cfKeyMap.get((Object)CFKey.STATISTICS));
                        CoreFoundation.CFDictionaryRef statistics = new CoreFoundation.CFDictionaryRef(result);
                        diskStore.setTimeStamp(System.currentTimeMillis());
                        result = statistics.getValue((PointerType)cfKeyMap.get((Object)CFKey.READ_OPS));
                        CoreFoundation.CFNumberRef stat = new CoreFoundation.CFNumberRef(result);
                        diskStore.setReads(stat.longValue());
                        result = statistics.getValue((PointerType)cfKeyMap.get((Object)CFKey.READ_BYTES));
                        stat.setPointer(result);
                        diskStore.setReadBytes(stat.longValue());
                        result = statistics.getValue((PointerType)cfKeyMap.get((Object)CFKey.WRITE_OPS));
                        stat.setPointer(result);
                        diskStore.setWrites(stat.longValue());
                        result = statistics.getValue((PointerType)cfKeyMap.get((Object)CFKey.WRITE_BYTES));
                        stat.setPointer(result);
                        diskStore.setWriteBytes(stat.longValue());
                        result = statistics.getValue((PointerType)cfKeyMap.get((Object)CFKey.READ_TIME));
                        stat.setPointer(result);
                        long xferTime = stat.longValue();
                        result = statistics.getValue((PointerType)cfKeyMap.get((Object)CFKey.WRITE_TIME));
                        stat.setPointer(result);
                        diskStore.setTransferTime((xferTime += stat.longValue()) / 1000000L);
                        properties.release();
                    } else {
                        LOG.debug("Unable to find block storage driver properties for {}", (Object)bsdName);
                    }
                    ArrayList<HWPartition> partitions = new ArrayList<HWPartition>();
                    CoreFoundation.CFMutableDictionaryRef properties = drive.createCFProperties();
                    Pointer result = properties.getValue((PointerType)cfKeyMap.get((Object)CFKey.BSD_UNIT));
                    CoreFoundation.CFNumberRef bsdUnit = new CoreFoundation.CFNumberRef(result);
                    result = properties.getValue((PointerType)cfKeyMap.get((Object)CFKey.LEAF));
                    CoreFoundation.CFBooleanRef cfFalse = new CoreFoundation.CFBooleanRef(result);
                    CoreFoundation.CFMutableDictionaryRef propertyDict = CF.CFDictionaryCreateMutable(CF.CFAllocatorGetDefault(), new CoreFoundation.CFIndex(0L), null, null);
                    propertyDict.setValue((PointerType)cfKeyMap.get((Object)CFKey.BSD_UNIT), (PointerType)bsdUnit);
                    propertyDict.setValue((PointerType)cfKeyMap.get((Object)CFKey.WHOLE), (PointerType)cfFalse);
                    matchingDict = CF.CFDictionaryCreateMutable(CF.CFAllocatorGetDefault(), new CoreFoundation.CFIndex(0L), null, null);
                    matchingDict.setValue((PointerType)cfKeyMap.get((Object)CFKey.IO_PROPERTY_MATCH), (PointerType)propertyDict);
                    IOKit.IOIterator serviceIterator = IOKitUtil.getMatchingServices((CoreFoundation.CFDictionaryRef)matchingDict);
                    properties.release();
                    propertyDict.release();
                    if (serviceIterator != null) {
                        IOKit.IORegistryEntry sdService = IOKit.INSTANCE.IOIteratorNext(serviceIterator);
                        while (sdService != null) {
                            String partBsdName;
                            String name = partBsdName = sdService.getStringProperty("BSD Name");
                            String type = "";
                            DiskArbitration.DADiskRef disk = DA.DADiskCreateFromBSDName(CF.CFAllocatorGetDefault(), session, partBsdName);
                            if (disk != null) {
                                CoreFoundation.CFDictionaryRef diskInfo = DA.DADiskCopyDescription(disk);
                                if (diskInfo != null) {
                                    result = diskInfo.getValue((PointerType)cfKeyMap.get((Object)CFKey.DA_MEDIA_NAME));
                                    CoreFoundation.CFStringRef volumePtr = new CoreFoundation.CFStringRef(result);
                                    type = volumePtr.stringValue();
                                    if (type == null) {
                                        type = "unknown";
                                    }
                                    if ((result = diskInfo.getValue((PointerType)cfKeyMap.get((Object)CFKey.DA_VOLUME_NAME))) == null) {
                                        name = type;
                                    } else {
                                        volumePtr.setPointer(result);
                                        name = volumePtr.stringValue();
                                    }
                                    diskInfo.release();
                                }
                                disk.release();
                            }
                            String mountPoint = logicalVolumeMap.containsKey(partBsdName) ? "Logical Volume: " + logicalVolumeMap.get(partBsdName) : mountPointMap.getOrDefault(partBsdName, "");
                            Long size = sdService.getLongProperty("Size");
                            Integer bsdMajor = sdService.getIntegerProperty("BSD Major");
                            Integer bsdMinor = sdService.getIntegerProperty("BSD Minor");
                            partitions.add(new HWPartition(partBsdName, name, type, sdService.getStringProperty("UUID"), size == null ? 0L : size, bsdMajor == null ? 0 : bsdMajor, bsdMinor == null ? 0 : bsdMinor, mountPoint));
                            sdService.release();
                            sdService = IOKit.INSTANCE.IOIteratorNext(serviceIterator);
                        }
                        serviceIterator.release();
                    }
                    Collections.sort(partitions);
                    diskStore.setPartitions(partitions.toArray(new HWPartition[0]));
                    if (parent != null) {
                        parent.release();
                    }
                } else {
                    LOG.error("Unable to find IOMedia device or parent for {}", (Object)bsdName);
                }
                drive.release();
            }
            driveListIter.release();
            return true;
        }
        return false;
    }

    public static boolean updateDiskStats(HWDiskStore diskStore) {
        DiskArbitration.DASessionRef session = DA.DASessionCreate(CF.CFAllocatorGetDefault());
        if (session == null) {
            LOG.error("Unable to open session to DiskArbitration framework.");
            return false;
        }
        Map<CFKey, CoreFoundation.CFStringRef> cfKeyMap = MacDisks.mapCFKeys();
        boolean diskFound = MacDisks.updateDiskStats(diskStore, session, MacDisks.queryMountPointMap(), MacDisks.queryLogicalVolumeMap(), cfKeyMap);
        session.release();
        for (CoreFoundation.CFTypeRef cFTypeRef : cfKeyMap.values()) {
            cFTypeRef.release();
        }
        return diskFound;
    }

    public static HWDiskStore[] getDisks() {
        Map<String, String> mountPointMap = MacDisks.queryMountPointMap();
        Map<String, String> logicalVolumeMap = MacDisks.queryLogicalVolumeMap();
        Map<CFKey, CoreFoundation.CFStringRef> cfKeyMap = MacDisks.mapCFKeys();
        ArrayList<HWDiskStore> diskList = new ArrayList<HWDiskStore>();
        DiskArbitration.DASessionRef session = DA.DASessionCreate(CF.CFAllocatorGetDefault());
        if (session == null) {
            LOG.error("Unable to open session to DiskArbitration framework.");
            return new HWDiskStore[0];
        }
        ArrayList<String> bsdNames = new ArrayList<String>();
        IOKit.IOIterator iter = IOKitUtil.getMatchingServices((String)"IOMedia");
        if (iter != null) {
            IOKit.IORegistryEntry media = iter.next();
            while (media != null) {
                Boolean bl = media.getBooleanProperty("Whole");
                if (bl != null && bl.booleanValue()) {
                    DiskArbitration.DADiskRef disk = DA.DADiskCreateFromIOMedia(CF.CFAllocatorGetDefault(), session, (IOKit.IOObject)media);
                    bsdNames.add(DA.DADiskGetBSDName(disk));
                    disk.release();
                }
                media.release();
                media = iter.next();
            }
            iter.release();
        }
        for (String string : bsdNames) {
            String model = "";
            String serial = "";
            long size = 0L;
            String path = "/dev/" + string;
            DiskArbitration.DADiskRef disk = DA.DADiskCreateFromBSDName(CF.CFAllocatorGetDefault(), session, path);
            if (disk == null) continue;
            CoreFoundation.CFDictionaryRef diskInfo = DA.DADiskCopyDescription(disk);
            if (diskInfo != null) {
                Pointer result = diskInfo.getValue((PointerType)cfKeyMap.get((Object)CFKey.DA_DEVICE_MODEL));
                CoreFoundation.CFStringRef modelPtr = new CoreFoundation.CFStringRef(result);
                model = modelPtr.stringValue();
                if (model == null) {
                    model = "unknown";
                }
                result = diskInfo.getValue((PointerType)cfKeyMap.get((Object)CFKey.DA_MEDIA_SIZE));
                CoreFoundation.CFNumberRef sizePtr = new CoreFoundation.CFNumberRef(result);
                size = sizePtr.longValue();
                diskInfo.release();
                if (!"Disk Image".equals(model)) {
                    CoreFoundation.CFStringRef modelNameRef = CoreFoundation.CFStringRef.createCFString((String)model);
                    CoreFoundation.CFMutableDictionaryRef propertyDict = CF.CFDictionaryCreateMutable(CF.CFAllocatorGetDefault(), new CoreFoundation.CFIndex(0L), null, null);
                    propertyDict.setValue((PointerType)cfKeyMap.get((Object)CFKey.MODEL), (PointerType)modelNameRef);
                    CoreFoundation.CFMutableDictionaryRef matchingDict = CF.CFDictionaryCreateMutable(CF.CFAllocatorGetDefault(), new CoreFoundation.CFIndex(0L), null, null);
                    matchingDict.setValue((PointerType)cfKeyMap.get((Object)CFKey.IO_PROPERTY_MATCH), (PointerType)propertyDict);
                    IOKit.IOIterator serviceIterator = IOKitUtil.getMatchingServices((CoreFoundation.CFDictionaryRef)matchingDict);
                    modelNameRef.release();
                    propertyDict.release();
                    if (serviceIterator != null) {
                        IOKit.IORegistryEntry sdService = serviceIterator.next();
                        while (sdService != null) {
                            serial = sdService.getStringProperty("Serial Number");
                            sdService.release();
                            if (serial != null) break;
                            sdService.release();
                            sdService = serviceIterator.next();
                        }
                        serviceIterator.release();
                    }
                    if (serial == null) {
                        serial = "";
                    }
                }
            }
            disk.release();
            if (size <= 0L) continue;
            HWDiskStore diskStore = new HWDiskStore();
            diskStore.setName(string);
            diskStore.setModel(model.trim());
            diskStore.setSerial(serial.trim());
            diskStore.setSize(size);
            MacDisks.updateDiskStats(diskStore, session, mountPointMap, logicalVolumeMap, cfKeyMap);
            diskList.add(diskStore);
        }
        session.release();
        for (CoreFoundation.CFTypeRef cFTypeRef : cfKeyMap.values()) {
            cFTypeRef.release();
        }
        Collections.sort(diskList);
        return diskList.toArray(new HWDiskStore[0]);
    }

    private static enum CFKey {
        IO_PROPERTY_MATCH("IOPropertyMatch"),
        STATISTICS("Statistics"),
        READ_OPS("Operations (Read)"),
        READ_BYTES("Bytes (Read)"),
        READ_TIME("Total Time (Read)"),
        WRITE_OPS("Operations (Write)"),
        WRITE_BYTES("Bytes (Write)"),
        WRITE_TIME("Total Time (Write)"),
        BSD_UNIT("BSD Unit"),
        LEAF("Leaf"),
        WHOLE("Whole"),
        DA_MEDIA_NAME("DAMediaName"),
        DA_VOLUME_NAME("DAVolumeName"),
        DA_MEDIA_SIZE("DAMediaSize"),
        DA_DEVICE_MODEL("DADeviceModel"),
        MODEL("Model");

        private final String key;

        private CFKey(String key) {
            this.key = key;
        }

        public String getKey() {
            return this.key;
        }
    }
}

