/*
 * Decompiled with CFR 0.152.
 */
package com.github.eirslett.maven.plugins.frontend.lib;

import com.github.eirslett.maven.plugins.frontend.lib.AtlassianDevMetricsReporter;
import com.github.eirslett.maven.plugins.frontend.lib.NodeVersionHelper;
import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeVersionDetector {
    private static final Logger logger = LoggerFactory.getLogger(NodeVersionDetector.class);
    private static final String TOOL_VERSIONS_FILENAME = ".tool-versions";

    public static String getNodeVersion(File workingDir, String providedNodeVersion, String genericNodeVersionFile, String artifactId, String forkVersion) throws Exception {
        EventData eventData = new EventData(artifactId, forkVersion);
        if (!Objects.isNull(providedNodeVersion) && !providedNodeVersion.trim().isEmpty()) {
            logger.debug("Looks like a node version was set so using that: {}", (Object)providedNodeVersion);
            NodeVersionDetector.reportFoundVersion("maven", providedNodeVersion, eventData);
            return providedNodeVersion;
        }
        if (!Objects.isNull(genericNodeVersionFile) && !genericNodeVersionFile.trim().isEmpty()) {
            File genericNodeVersionFileFile = new File(genericNodeVersionFile);
            if (!genericNodeVersionFileFile.exists()) {
                throw new Exception("The Node version file doesn't seem to exist: " + genericNodeVersionFileFile);
            }
            if (genericNodeVersionFile.endsWith(".toml") && genericNodeVersionFile.contains("mise")) {
                String trimmedVersion = NodeVersionDetector.readMiseConfigTomlFile(genericNodeVersionFileFile, genericNodeVersionFileFile.toPath());
                NodeVersionDetector.reportFoundVersion("mise", trimmedVersion, eventData);
                return trimmedVersion;
            }
            if (genericNodeVersionFile.endsWith(TOOL_VERSIONS_FILENAME)) {
                String trimmedVersion = NodeVersionDetector.readToolVersionsFile(genericNodeVersionFileFile, genericNodeVersionFileFile.toPath());
                NodeVersionDetector.reportFoundVersion("tool-versions", trimmedVersion, eventData);
                return trimmedVersion;
            }
            String versionLocation = genericNodeVersionFile.contains("nvmrc") ? "nvmrc" : (genericNodeVersionFile.contains("node-version") ? "node-version" : "unknown");
            String trimmedVersion = NodeVersionDetector.readNvmrcFile(genericNodeVersionFileFile, genericNodeVersionFileFile.toPath());
            NodeVersionDetector.reportFoundVersion(versionLocation, trimmedVersion, eventData);
            return trimmedVersion;
        }
        try {
            return NodeVersionDetector.recursivelyFindVersion(workingDir, eventData);
        }
        catch (Throwable throwable) {
            logger.debug("Going to use the configuration node version, failed to find a file with the version because", throwable);
            return providedNodeVersion;
        }
    }

    public static List<String> listMiseConfigFilenames() {
        String miseConfigDir = System.getenv("MISE_CONFIG_DIR");
        String miseEnv = System.getenv("MISE_ENV");
        ArrayList<String> allMiseConfigFilenames = new ArrayList<String>();
        allMiseConfigFilenames.add(String.format("%s/config.%s.toml", miseConfigDir, miseEnv));
        allMiseConfigFilenames.add(String.format("%s/mise.%s.toml", miseConfigDir, miseEnv));
        allMiseConfigFilenames.add(".config/mise/config.toml");
        allMiseConfigFilenames.add("mise/config.toml");
        allMiseConfigFilenames.add("mise.toml");
        allMiseConfigFilenames.add(".mise/config.toml");
        allMiseConfigFilenames.add(".mise.toml");
        allMiseConfigFilenames.add(".config/mise/config.local.toml");
        allMiseConfigFilenames.add("mise/config.local.toml");
        allMiseConfigFilenames.add("mise.local.toml");
        allMiseConfigFilenames.add(".mise/config.local.toml");
        allMiseConfigFilenames.add(".mise.local.toml");
        allMiseConfigFilenames.add(String.format(".config/mise/config.%s.toml", miseEnv));
        allMiseConfigFilenames.add(String.format("mise/config.%s.toml", miseEnv));
        allMiseConfigFilenames.add(String.format("mise.%s.toml", miseEnv));
        allMiseConfigFilenames.add(String.format(".mise/config.%s.toml", miseEnv));
        allMiseConfigFilenames.add(String.format(".mise.%s.toml", miseEnv));
        allMiseConfigFilenames.add(String.format(".config/mise/config.%s.local.toml", miseEnv));
        allMiseConfigFilenames.add(String.format("mise/config.%s.local.toml", miseEnv));
        allMiseConfigFilenames.add(String.format(".mise/config.%s.local.toml", miseEnv));
        allMiseConfigFilenames.add(String.format(".mise.%s.local.toml", miseEnv));
        return allMiseConfigFilenames;
    }

    public static String recursivelyFindVersion(File directory, EventData eventData) throws Exception {
        Object trimmedLine;
        String trimmedLine2;
        String trimmedLine3;
        if (!directory.canRead()) {
            throw new Exception("Tried to find a Node version file but giving up because it's not possible to read " + directory.getPath());
        }
        String directoryPath = directory.getPath();
        Path nodeVersionFilePath = Paths.get(directoryPath, ".node-version");
        File nodeVersionFile = nodeVersionFilePath.toFile();
        if (nodeVersionFile.exists() && (trimmedLine3 = NodeVersionDetector.readNvmrcFile(nodeVersionFile, nodeVersionFilePath)) != null) {
            NodeVersionDetector.reportFoundVersion("node-version", trimmedLine3, eventData);
            return trimmedLine3;
        }
        Path nvmrcFilePath = Paths.get(directoryPath, ".nvmrc");
        File nvmrcFile = nvmrcFilePath.toFile();
        if (nvmrcFile.exists() && (trimmedLine2 = NodeVersionDetector.readNvmrcFile(nvmrcFile, nvmrcFilePath)) != null) {
            NodeVersionDetector.reportFoundVersion("nvmrc", trimmedLine2, eventData);
            return trimmedLine2;
        }
        Path toolVersionsFilePath = Paths.get(directoryPath, TOOL_VERSIONS_FILENAME);
        File toolVersionsFile = toolVersionsFilePath.toFile();
        if (toolVersionsFile.exists() && (trimmedLine = NodeVersionDetector.readToolVersionsFile(toolVersionsFile, toolVersionsFilePath)) != null) {
            NodeVersionDetector.reportFoundVersion("tool-versions", (String)trimmedLine, eventData);
            return trimmedLine;
        }
        for (String miseConfigFilename : NodeVersionDetector.listMiseConfigFilenames()) {
            String trimmedVersion;
            String[] splitMiseConfigFilename = miseConfigFilename.split("/");
            Path potentiallyAbsoluteFilepath = Paths.get("", splitMiseConfigFilename);
            Path miseConfigFilePath = potentiallyAbsoluteFilepath.isAbsolute() ? potentiallyAbsoluteFilepath : Paths.get(directoryPath, splitMiseConfigFilename);
            File miseConfigFile = miseConfigFilePath.toFile();
            if (!miseConfigFile.exists() || (trimmedVersion = NodeVersionDetector.readMiseConfigTomlFile(miseConfigFile, miseConfigFilePath)) == null) continue;
            NodeVersionDetector.reportFoundVersion("mise", trimmedVersion, eventData);
            return trimmedVersion;
        }
        File parent = directory.getParentFile();
        if (Objects.isNull(parent) || directory.equals(parent)) {
            throw new Exception("Reach root-level without finding a suitable file");
        }
        return NodeVersionDetector.recursivelyFindVersion(parent, eventData);
    }

    private static String readNvmrcFile(File nvmrcFile, Path nvmrcFilePath) throws Exception {
        NodeVersionDetector.assertNodeVersionFileIsReadable(nvmrcFile);
        List<String> lines = Files.readAllLines(nvmrcFilePath);
        Optional<String> version = NodeVersionDetector.readNvmrcFileLines(lines);
        if (version.isPresent()) {
            logger.info("Found the version of Node in: " + nvmrcFilePath.normalize());
        }
        return version.orElse(null);
    }

    @VisibleForTesting
    static Optional<String> readNvmrcFileLines(List<String> lines) {
        for (String line : lines) {
            String trimmedLine;
            if (Objects.isNull(line) || (trimmedLine = line.trim()).isEmpty() || trimmedLine.startsWith("#") || trimmedLine.startsWith("/") || trimmedLine.startsWith("!")) continue;
            trimmedLine = trimmedLine.replaceFirst("(\\s*[#!/].*)", "");
            return Optional.of(trimmedLine);
        }
        return Optional.empty();
    }

    @VisibleForTesting
    static String readMiseConfigTomlFile(File miseTomlFile, Path miseTomlFilePath) throws Exception {
        NodeVersionDetector.assertNodeVersionFileIsReadable(miseTomlFile);
        List<String> lines = Files.readAllLines(miseTomlFilePath);
        for (String line : lines) {
            String trimmedLine;
            if (Objects.isNull(line) || (trimmedLine = line.trim()).isEmpty() || !trimmedLine.startsWith("node")) continue;
            logger.info("Found the version of Node in: " + miseTomlFilePath.normalize());
            if (trimmedLine.contains("[")) {
                throw new Exception("mise file support is limited to a single version");
            }
            return trimmedLine.replaceAll("node(js)?\\s*=\\s*", "").replaceAll("\"", "").replaceAll("#.*$", "").trim();
        }
        return null;
    }

    private static String readToolVersionsFile(File toolVersionsFile, Path toolVersionsFilePath) throws Exception {
        NodeVersionDetector.assertNodeVersionFileIsReadable(toolVersionsFile);
        List<String> lines = Files.readAllLines(toolVersionsFilePath);
        for (String line : lines) {
            String trimmedLine;
            if (Objects.isNull(line) || (trimmedLine = line.trim()).isEmpty() || !trimmedLine.startsWith("node")) continue;
            logger.info("Found the version of Node in: " + toolVersionsFilePath.normalize());
            return trimmedLine.replaceAll("node(js)?\\s*", "");
        }
        return null;
    }

    private static void assertNodeVersionFileIsReadable(File file) throws Exception {
        if (!file.canRead()) {
            throw new Exception("Tried to read the node version from the file, but giving up because it's not possible to read" + file.getPath());
        }
    }

    private static void reportFoundVersion(final String location, String nodeVersion, EventData eventData) {
        if (Objects.isNull(nodeVersion) || !NodeVersionHelper.validateVersion(nodeVersion)) {
            return;
        }
        final String standardisedNodeVersion = NodeVersionHelper.getDownloadableVersion(nodeVersion);
        AtlassianDevMetricsReporter.incrementCount("runtime.version", eventData.artifactId, eventData.forkVersion, (Map<String, String>)new HashMap<String, String>(){
            {
                this.put("runtime-version", standardisedNodeVersion);
                this.put("version-location", location);
            }
        });
    }

    public static interface NodeVersionLocations {
        public static final String NVMRC = "nvmrc";
        public static final String NODE_VERSION = "node-version";
        public static final String MAVEN = "maven";
        public static final String MISE = "mise";
        public static final String TOOL_VERSIONS = "tool-versions";
        public static final String UNKNOWN = "unknown";
    }

    @VisibleForTesting
    static class EventData {
        private final String artifactId;
        private final String forkVersion;

        EventData(String artifactId, String forkVersion) {
            this.artifactId = artifactId;
            this.forkVersion = forkVersion;
        }
    }
}

