/*
 * Decompiled with CFR 0.152.
 */
package com.bastiaanjansen.otp;

import com.bastiaanjansen.otp.HMACAlgorithm;
import com.bastiaanjansen.otp.helpers.URIHelper;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.codec.binary.Base32;

public final class HOTPGenerator {
    private static final String URL_SCHEME = "otpauth";
    private static final int DEFAULT_PASSWORD_LENGTH = 6;
    private static final HMACAlgorithm DEFAULT_HMAC_ALGORITHM = HMACAlgorithm.SHA1;
    private static final String OTP_TYPE = "hotp";
    private final int passwordLength;
    private final HMACAlgorithm algorithm;
    private final byte[] secret;

    private HOTPGenerator(Builder builder) {
        this.passwordLength = builder.passwordLength;
        this.algorithm = builder.algorithm;
        this.secret = builder.secret;
    }

    public static HOTPGenerator fromURI(URI uri) throws URISyntaxException {
        Map<String, String> query = URIHelper.queryItems(uri);
        byte[] secret = Optional.ofNullable(query.get("secret")).map(s -> s.getBytes(StandardCharsets.UTF_8)).orElseThrow(() -> new IllegalArgumentException("Secret query parameter must be set"));
        Builder builder = new Builder(secret);
        try {
            Optional.ofNullable(query.get("digits")).map(Integer::valueOf).ifPresent(builder::withPasswordLength);
            Optional.ofNullable(query.get("algorithm")).map(String::toUpperCase).map(HMACAlgorithm::valueOf).ifPresent(builder::withAlgorithm);
        }
        catch (Exception e) {
            throw new URISyntaxException(uri.toString(), "URI could not be parsed");
        }
        return builder.build();
    }

    public static HOTPGenerator withDefaultValues(byte[] secret) {
        return new Builder(secret).build();
    }

    public URI getURI(int counter, String issuer) throws URISyntaxException {
        return this.getURI(counter, issuer, "");
    }

    public URI getURI(int counter, String issuer, String account) throws URISyntaxException {
        HashMap<String, String> query = new HashMap<String, String>();
        query.put("counter", String.valueOf(counter));
        return this.getURI(OTP_TYPE, issuer, account, query);
    }

    public int getPasswordLength() {
        return this.passwordLength;
    }

    public HMACAlgorithm getAlgorithm() {
        return this.algorithm;
    }

    public boolean verify(String code, long counter) {
        return this.verify(code, counter, 0);
    }

    public boolean verify(String code, long counter, int delayWindow) {
        if (code.length() != this.passwordLength) {
            return false;
        }
        for (int i = -delayWindow; i <= delayWindow; ++i) {
            String currentCode = this.generate(counter + (long)i);
            if (!code.equals(currentCode)) continue;
            return true;
        }
        return false;
    }

    public String generate(long counter) throws IllegalStateException {
        byte[] hash;
        if (counter < 0L) {
            throw new IllegalArgumentException("Counter must be greater than or equal to 0");
        }
        byte[] secretBytes = this.decodeBase32(this.secret);
        byte[] counterBytes = this.longToBytes(counter);
        try {
            hash = this.generateHash(secretBytes, counterBytes);
        }
        catch (InvalidKeyException | NoSuchAlgorithmException e) {
            throw new IllegalStateException();
        }
        return this.getCodeFromHash(hash);
    }

    public URI getURI(String type, String issuer, String account, Map<String, String> query) throws URISyntaxException {
        query.put("digits", String.valueOf(this.passwordLength));
        query.put("algorithm", this.algorithm.name());
        query.put("secret", new String(this.secret, StandardCharsets.UTF_8));
        query.put("issuer", issuer);
        String path = account.isEmpty() ? URIHelper.encode(issuer) : String.format("%s:%s", URIHelper.encode(issuer), URIHelper.encode(account));
        return URIHelper.createURI(URL_SCHEME, type, path, query);
    }

    private byte[] decodeBase32(byte[] value) {
        Base32 codec = new Base32();
        return codec.decode(value);
    }

    private byte[] longToBytes(long value) {
        return ByteBuffer.allocate(8).putLong(value).array();
    }

    private byte[] generateHash(byte[] secret, byte[] data) throws InvalidKeyException, NoSuchAlgorithmException {
        SecretKeySpec signKey = new SecretKeySpec(secret, "RAW");
        Mac mac = Mac.getInstance(this.algorithm.getHMACName());
        mac.init(signKey);
        return mac.doFinal(data);
    }

    private String getCodeFromHash(byte[] hash) {
        int mask = 15;
        byte lastByte = hash[hash.length - 1];
        int offset = lastByte & mask;
        byte[] truncatedHashInBytes = new byte[]{hash[offset], hash[offset + 1], hash[offset + 2], hash[offset + 3]};
        ByteBuffer byteBuffer = ByteBuffer.wrap(truncatedHashInBytes);
        long truncatedHash = byteBuffer.getInt();
        truncatedHash &= Integer.MAX_VALUE;
        truncatedHash = (long)((double)truncatedHash % Math.pow(10.0, this.passwordLength));
        return String.format("%0" + this.passwordLength + "d", truncatedHash);
    }

    public static final class Builder {
        private int passwordLength;
        private HMACAlgorithm algorithm;
        private final byte[] secret;

        public Builder(byte[] secret) {
            if (secret.length == 0) {
                throw new IllegalArgumentException("Secret must not be empty");
            }
            this.secret = secret;
            this.passwordLength = 6;
            this.algorithm = DEFAULT_HMAC_ALGORITHM;
        }

        public Builder(String secret) {
            this(secret.getBytes(StandardCharsets.UTF_8));
        }

        public Builder withPasswordLength(int passwordLength) {
            if (!this.passwordLengthIsValid(passwordLength)) {
                throw new IllegalArgumentException("Password length must be between 6 and 8 digits");
            }
            this.passwordLength = passwordLength;
            return this;
        }

        public Builder withAlgorithm(HMACAlgorithm algorithm) {
            this.algorithm = algorithm;
            return this;
        }

        public HOTPGenerator build() {
            return new HOTPGenerator(this);
        }

        private boolean passwordLengthIsValid(int passwordLength) {
            return passwordLength >= 6 && passwordLength <= 8;
        }
    }
}

