/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2019-2020 TheRandomLabs
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.therandomlabs.curseapi.forgesvc;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import com.therandomlabs.curseapi.CurseAPIProvider;
import com.therandomlabs.curseapi.CurseException;
import com.therandomlabs.curseapi.file.CurseFile;
import com.therandomlabs.curseapi.file.CurseFiles;
import com.therandomlabs.curseapi.game.CurseCategory;
import com.therandomlabs.curseapi.game.CurseGame;
import com.therandomlabs.curseapi.project.CurseProject;
import com.therandomlabs.curseapi.project.CurseSearchQuery;
import com.therandomlabs.curseapi.util.JsoupUtils;
import com.therandomlabs.curseapi.util.RetrofitUtils;
import okhttp3.HttpUrl;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;

/**
 * A {@link CurseAPIProvider} that uses the API at
 * <a href="https://addons-ecs.forgesvc.net/">https://addons-ecs.forgesvc.net/</a>
 * used by the Twitch launcher.
 * <p>
 * This provider falls back on the methods declared in
 * {@link com.therandomlabs.curseapi.CurseAPI} wherever possible so that default behaviors
 * may be overridden. For example, {@link CurseProject#files()} is implemented by
 * calling {@link com.therandomlabs.curseapi.CurseAPI#files(int)} rather than directly
 * calling {@link #files(int)}.
 * <p>
 * Where possible, this class should not be accessed directly, and the methods declared in
 * {@link com.therandomlabs.curseapi.CurseAPI} should be favored.
 *
 * @see com.therandomlabs.curseapi.cfwidget.CFWidgetProvider
 */
public final class ForgeSvcProvider implements CurseAPIProvider {
	/**
	 * The singleton instance of {@link ForgeSvcProvider}.
	 */
	public static final ForgeSvcProvider instance = new ForgeSvcProvider();

	private static final ForgeSvc forgeSvc =
			RetrofitUtils.get("https://addons-ecs.forgesvc.net/").create(ForgeSvc.class);

	private ForgeSvcProvider() {}

	/**
	 * {@inheritDoc}
	 */
	@Nullable
	@Override
	public CurseProject project(int id) throws CurseException {
		return RetrofitUtils.execute(forgeSvc.getProject(id));
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("NullAway")
	@Nullable
	@Override
	public Element projectDescription(int id) throws CurseException {
		final Element element = RetrofitUtils.getElement(forgeSvc.getDescription(id));
		//If the description is empty, we assume that the project does not exist.
		return JsoupUtils.isEmpty(element) ? null : replaceLinkouts(element);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<CurseProject> searchProjects(CurseSearchQuery query) throws CurseException {
		final List<ForgeSvcProject> projects = RetrofitUtils.execute(forgeSvc.searchProjects(
				query.gameID(), query.categorySectionID(), query.categoryID(),
				query.gameVersionString(), query.pageIndex(), query.pageSize(),
				query.searchFilter(), query.sortingMethod().id()
		));

		if (projects == null) {
			throw new CurseException("Failed to search projects: " + query);
		}

		return new ArrayList<>(projects);
	}

	/**
	 * {@inheritDoc}
	 */
	@Nullable
	@Override
	public CurseFiles<CurseFile> files(int projectID) throws CurseException {
		final Set<ForgeSvcFile> files = RetrofitUtils.execute(forgeSvc.getFiles(projectID));

		if (files == null) {
			return null;
		}

		for (ForgeSvcFile file : files) {
			file.setProjectID(projectID);
		}

		return new CurseFiles<>(files);
	}

	/**
	 * {@inheritDoc}
	 */
	@Nullable
	@Override
	public CurseFile file(int projectID, int fileID) throws CurseException {
		final ForgeSvcFile file = RetrofitUtils.execute(forgeSvc.getFile(projectID, fileID));

		if (file == null) {
			return null;
		}

		file.setProjectID(projectID);
		return file;
	}

	/**
	 * {@inheritDoc}
	 *
	 * @param projectID a project ID. This is apparently not necessary, so {@code 0} will suffice.
	 */
	@Nullable
	@Override
	public Element fileChangelog(int projectID, int fileID) throws CurseException {
		final Element changelog =
				RetrofitUtils.getElement(forgeSvc.getChangelog(projectID, fileID));
		return changelog == null ? null : replaceLinkouts(changelog);
	}

	/**
	 * {@inheritDoc}
	 *
	 * @param projectID a project ID. This is apparently not necessary, so {@code 0} will suffice.
	 */
	@Nullable
	@Override
	public HttpUrl fileDownloadURL(int projectID, int fileID) throws CurseException {
		final String url = RetrofitUtils.getString(forgeSvc.getFileDownloadURL(projectID, fileID));
		return url == null ? null : HttpUrl.get(url);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Set<CurseGame> games() throws CurseException {
		final Set<ForgeSvcGame> games = RetrofitUtils.execute(forgeSvc.getGames(false));

		if (games == null) {
			throw new CurseException("Failed to retrieve games");
		}

		return new TreeSet<>(games);
	}

	/**
	 * {@inheritDoc}
	 */
	@Nullable
	@Override
	public CurseGame game(int id) throws CurseException {
		return RetrofitUtils.execute(forgeSvc.getGame(id));
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Set<CurseCategory> categories() throws CurseException {
		final Set<ForgeSvcCategory> categories = RetrofitUtils.execute(forgeSvc.getCategories());

		if (categories == null) {
			throw new CurseException("Failed to retrieve categories");
		}

		return new TreeSet<>(categories);
	}

	/**
	 * {@inheritDoc}
	 */
	@Nullable
	@Override
	public Set<CurseCategory> categories(int sectionID) throws CurseException {
		final Set<ForgeSvcCategory> categories =
				RetrofitUtils.execute(forgeSvc.getCategories(sectionID));
		return categories == null ? null : new TreeSet<>(categories);
	}

	/**
	 * {@inheritDoc}
	 */
	@Nullable
	@Override
	public CurseCategory category(int id) throws CurseException {
		return RetrofitUtils.execute(forgeSvc.getCategory(id));
	}

	private static Element replaceLinkouts(Element element) {
		final Elements links = element.getElementsByTag("a");

		for (Element link : links) {
			if (link.attr("href").startsWith("/linkout?remoteUrl=")) {
				final String encoded = link.attr("href").substring("/linkout?remoteUrl=".length());
				link.attr("href", decode(decode(encoded)));
			}
		}

		return element;
	}

	private static String decode(String encoded) {
		try {
			return URLDecoder.decode(
					encoded.replace("+", "%2B"), StandardCharsets.UTF_8.name()
			).replace("%2B", "+");
		} catch (UnsupportedEncodingException ex) {
			throw new RuntimeException(ex);
		}
	}
}
