package com.amity.socialcloud.sdk.infra.mqtt.listener

import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.chat.data.channel.ChannelRepository
import com.amity.socialcloud.sdk.chat.data.marker.subchannel.SubChannelMarkerRepository
import com.amity.socialcloud.sdk.chat.data.message.MessageCreatedEventPersister
import com.amity.socialcloud.sdk.chat.domain.marker.channel.UpdateChannelUnreadLastMentionedSegmentUseCase
import com.amity.socialcloud.sdk.chat.domain.marker.channel.UpdateChannelUnreadLastSegmentUseCase
import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.core.MarkerEvent
import com.amity.socialcloud.sdk.core.MessagePreviewEvent
import com.amity.socialcloud.sdk.core.ObjectResolverEngine
import com.amity.socialcloud.sdk.core.mention.AmityMentionType
import com.amity.socialcloud.sdk.core.session.eventbus.MarkerEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.MessagePreviewEventBus
import com.amity.socialcloud.sdk.log.AmityLog
import com.ekoapp.ekosdk.internal.api.dto.MessageQueryDto
import com.jakewharton.rxrelay3.PublishRelay
import com.jakewharton.rxrelay3.Relay
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.functions.Consumer
import io.reactivex.rxjava3.functions.Predicate
import io.reactivex.rxjava3.schedulers.Schedulers
import java.util.concurrent.TimeUnit

internal class MessageCreated : MessageEventListener() {

    private val BUFFER_TIME_SPAN_IN_MILLIS: Long = 500
    private val BUFFER_MAX_COUNT = 100
    private val relay: Relay<MessageQueryDto> = PublishRelay.create()
    private val hasItem = Predicate { list: List<MessageQueryDto> -> list.isNotEmpty() }
    private val handleItems = Consumer { list: List<MessageQueryDto> -> handleBufferedItems(list) }

    init {
        relay.buffer(
            BUFFER_TIME_SPAN_IN_MILLIS,
            TimeUnit.MILLISECONDS,
            Schedulers.io(),
            BUFFER_MAX_COUNT
        )
            .filter(hasItem)
            .doOnNext(handleItems)
            .subscribe()
    }

    override fun getEventName(): String {
        return "message.created"
    }

    override fun shouldProcessEvent(dto: MessageQueryDto): Boolean {
        return true
    }

    override fun processEvent(dto: MessageQueryDto) {
        relay.accept(dto)
    }

    private fun handleBufferedItems(list: List<MessageQueryDto>) {
        MessageCreatedEventPersister().persist(list)
            .andThen(Completable.fromAction {
                list.map {  dto ->
                    dto.messages
                }.map { messageList ->
                    // Emit message created event to event bus
                    messageList.map { message ->
                        MarkerEvent.NewMessage(message).let(MarkerEventBus::publish)
                        UpdateChannelUnreadLastSegmentUseCase().execute(message.channelId, message.segment)
                        val isMentionedAll = message.mentionees
                                .mapNotNull { it.type }
                                .any { it == AmityMentionType.CHANNEL.apiKey }
                        val isMentioned = if (isMentionedAll) {
                            true
                        } else {
                            message.mentionees
                                    .mapNotNull { it.userPublicIds }
                                    .flatten()
                                    .contains(AmityCoreClient.getUserId())
                        }
                        if (isMentioned) {
                            UpdateChannelUnreadLastMentionedSegmentUseCase().execute(message.channelId, message.segment)
                        }
                        list.map { dto -> dto.subChannels }.map { subChannelList ->
                            subChannelList.find { it.subChannelId == message.subChannelId }
                        }.firstOrNull()?.let { subChannel ->
                            MessagePreviewEvent.MessageCreated(message,subChannel).let(MessagePreviewEventBus::publish)
                        }
                    }
                    // Fetch channels if not exists
                    val channelRepository = ChannelRepository()
                    messageList
                        .map { it.channelId }
                        .distinct()
                        .map { channelId ->
                            if (!channelRepository.isChannelCacheExists(channelId)) {
                                CoreClient.resolve(channelId, ObjectResolverEngine.Companion.ReferenceType.CHANNEL)
                            }
                        }
                    // Fetch subchannel unread info if not exists
                    val subChannelMarkerRepository = SubChannelMarkerRepository()
                    messageList
                        .mapNotNull { it.subChannelId }
                        .distinct()
                        .map { subChannelId ->
                            if (subChannelMarkerRepository.getSubChannelUnreadInfo(subChannelId) == null) {
                               CoreClient.resolve(subChannelId, ObjectResolverEngine.Companion.ReferenceType.SUBCHANNEL_UNREAD_INFO)
                            }
                        }
                }
            })
            .subscribeOn(Schedulers.io())
            .doOnError {
                AmityLog.e(it)
            }
            .subscribe()
    }

}