package com.amity.socialcloud.sdk.social.domain.story

import com.amity.socialcloud.sdk.core.domain.ComposerUseCase
import com.amity.socialcloud.sdk.core.domain.file.ImageGetUseCase
import com.amity.socialcloud.sdk.core.domain.file.VideoGetUseCase
import com.amity.socialcloud.sdk.core.domain.reaction.MyReactionsGetUseCase
import com.amity.socialcloud.sdk.core.domain.user.UserGetUseCase
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionReferenceType
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.amity.socialcloud.sdk.model.social.story.AmityStoryImageDisplayMode
import com.amity.socialcloud.sdk.model.social.story.AmityStoryItem
import com.amity.socialcloud.sdk.social.domain.storytarget.get.GetStoryTargetUseCase
import com.google.gson.JsonArray
import com.google.gson.JsonObject


internal class StoryComposerUseCase : ComposerUseCase() {
    fun execute(story: AmityStory): AmityStory {
        addTarget(story)
        addData(story)
        addCreator(story)
        addStoryItems(story)
        addMyReactions(story)
        return story
    }

    private fun addTarget(story: AmityStory) {
        story.target = GetStoryTargetUseCase().execute(story.getTargetType(), story.getTargetId())
    }

    private fun addData(story: AmityStory) {
        when (story.getDataType()) {
            AmityStory.DataType.IMAGE -> {
                story.data = AmityStory.Data.IMAGE(
                    storyId = story.getStoryId(),
                    rawData = story.rawData,
                    imageDisplayMode = AmityStoryImageDisplayMode.enumOf(story.rawData?.get("imageDisplayMode")?.asString ?: ""),
                    image = ImageGetUseCase().execute(getImageFileId(story.rawData))
                )
            }
            AmityStory.DataType.VIDEO -> {
                story.data = AmityStory.Data.VIDEO(
                    storyId = story.getStoryId(),
                    rawData = story.rawData,
                    thumbnail = ImageGetUseCase().execute(getThumbnailFileId(story.rawData)),
                    video = VideoGetUseCase().execute(getVideoFileId(story.rawData))
                )
            }
            else -> {
                // do nothing
            }
        }
    }

    private fun getImageFileId(rawData: JsonObject?): String {
        return rawData?.get("fileId")?.asString ?: ""
    }

    private fun getVideoFileId(rawData: JsonObject?): String {
        return rawData?.getAsJsonObject("videoFileId")?.get("original")?.asString ?: ""
    }

    private fun getThumbnailFileId(rawData: JsonObject?): String {
        return rawData?.get("thumbnailFileId")?.asString ?: ""
    }

    private fun addStoryItems(story: AmityStory) {
        story.storyItems = mapStoryItems(story.rawStoryItems)
    }

    private fun mapStoryItems(rawStoryItems: JsonArray?) : List<AmityStoryItem> {
        return rawStoryItems?.map {
            val item = it.asJsonObject
            val itemType = item.get("type")?.asString
            val data = item.get("data")?.asJsonObject
            val url = data?.get("url")?.asString ?: ""
            val customText = data?.get("customText")?.asString
            val storyItem = when (itemType) {
                "hyperlink" -> {
                    AmityStoryItem.HYPERLINK(
                        url = url,
                        customText = customText
                    )
                }
                else -> {
                    AmityStoryItem.UNKNOWN
                }
            }
            storyItem
        } ?: emptyList()
    }

    private fun addCreator(story: AmityStory) {
        story.creator = UserGetUseCase().execute(story.getCreatorId())
    }

    private fun addMyReactions(story: AmityStory) {
        story.myReactions =
            MyReactionsGetUseCase().execute(AmityReactionReferenceType.STORY, story.getStoryId())
    }

}