package com.amity.socialcloud.sdk.chat.data.marker.message

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.chat.data.marker.reader.MarkReadRemoteDataStore
import com.amity.socialcloud.sdk.chat.data.marker.reader.SubChannelReadingPersister
import com.amity.socialcloud.sdk.chat.data.marker.readuser.MessageDeliveredUsersRequest
import com.amity.socialcloud.sdk.chat.data.marker.readuser.MessageReadUsersRequest
import com.amity.socialcloud.sdk.chat.data.marker.readuser.paging.MessageDeliveredUsersMediator
import com.amity.socialcloud.sdk.chat.data.marker.readuser.paging.MessageReadUsersMediator
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.core.data.user.UserModelMapper
import com.amity.socialcloud.sdk.model.chat.message.MessageDeliveredMembershipFilter
import com.amity.socialcloud.sdk.model.chat.message.MessageReadMembershipFilter
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.ekoapp.ekosdk.internal.MessageMarkerEntity
import com.ekoapp.ekosdk.internal.api.dto.MessageDeliveredUsersQueryDto
import com.ekoapp.ekosdk.internal.api.dto.MessageReadUsersQueryDto
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

internal class MessageMarkerRepository {
    
    fun fetchMessageMarker(messageIds: List<String>): Completable {
        return if(CoreClient.isUnreadCountEnable()) {
            MessageMarkerRemoteDataStore().queryMessageMarkers(messageIds)
                .flatMapCompletable {
                    try {
                        MessageMarkerQueryPersister().persist(it)
                    } catch (e: Exception) {
                        Completable.complete()
                    }
                }
        } else {
            Completable.complete()
        }
    }

    fun getReadCount(messageId: String): Int {
        return MessageMarkerLocalDataStore().getMessageMarker(messageId)?.readCount ?: 0
    }
    
    fun getDeliveredCount(messageId: String): Int {
        return MessageMarkerLocalDataStore().getMessageMarker(messageId)?.deliveredCount ?: 0
    }
    
    fun saveLocalMessageMarkers(markers: List<MessageMarkerEntity>): Completable {
        return Completable.fromAction {
            MessageMarkerLocalDataStore().saveLocalMessageMarkers(markers)
        }
    }
    
    fun markMessageDelivered(subChannel: String, messageId: String): Completable {
        return MarkReadRemoteDataStore().markMessageDelivered(subChannel, messageId)
                .flatMapCompletable {
                    SubChannelReadingPersister().persist(it)
                }
                .subscribeOn(Schedulers.io())
    }
    
    fun markMessageRead(subChannel: String, readToSegment: Int): Completable {
        return MarkReadRemoteDataStore().markMessageRead(subChannel, readToSegment)
            .ignoreElement()
            .subscribeOn(Schedulers.io())
    }
    
    fun fetchMessageReadUsers(request: MessageReadUsersRequest): Single<MessageReadUsersQueryDto> {
        return MessageMarkerRemoteDataStore().queryMessageReadUsers(request)
    }
    
    @ExperimentalPagingApi
    fun getMessageReadUsers(
        messageId: String,
        memberships: List<MessageReadMembershipFilter>
    ): Flowable<PagingData<AmityUser>> {
        val pagerCreator = QueryStreamPagerCreator(
                pagingConfig = PagingConfig(
                    pageSize = AmityObjectRepository.DEFAULT_PAGE_SIZE,
                    enablePlaceholders = false
                ),
                queryStreamMediator = MessageReadUsersMediator(messageId, memberships),
                pagingSourceFactory = { MessageMarkerLocalDataStore().getMessageReadUsersPagingSource(messageId, memberships) },
                modelMapper = UserModelMapper()
        )
        return pagerCreator.create().toRx3()
    }
    
    fun fetchMessageDeliveredUsers(request: MessageDeliveredUsersRequest): Single<MessageDeliveredUsersQueryDto> {
        return MessageMarkerRemoteDataStore().queryMessageDeliveredUsers(request)
    }
    
    @ExperimentalPagingApi
    fun getMessageDeliveredUsers(
        messageId: String,
        memberships: List<MessageDeliveredMembershipFilter>
    ): Flowable<PagingData<AmityUser>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = AmityObjectRepository.DEFAULT_PAGE_SIZE,
                enablePlaceholders = false
            ),
            queryStreamMediator = MessageDeliveredUsersMediator(messageId, memberships),
            pagingSourceFactory = { MessageMarkerLocalDataStore().getMessageDeliveredUsersPagingSource(messageId, memberships) },
            modelMapper = UserModelMapper()
        )
        return pagerCreator.create().toRx3()
    }
}