package com.amity.socialcloud.sdk.api.social.comment.create

import com.amity.socialcloud.sdk.core.mention.AmityMentionType
import com.amity.socialcloud.sdk.model.core.mention.AmityMentioneeTarget
import com.amity.socialcloud.sdk.model.social.comment.AmityComment
import com.google.gson.JsonArray
import com.google.gson.JsonObject

class AmityAttachmentCommentCreator private constructor(
    referenceType: String,
    referenceId: String,
    parentId: String?,
    commentId: String?,
    private val text: String?,
    private val attachments: List<AmityComment.Attachment>?,
    dataTypes: List<String>,
    metadata: JsonObject?,
    mentionees: List<AmityMentioneeTarget>?
) : AmityCommentCreator(
    referenceType,
    referenceId,
    parentId,
    commentId,
    metadata,
    mentionees,
    dataTypes
) {

    override fun getData(): JsonObject? {
        if (text.isNullOrEmpty()) return null

        val data = JsonObject()
        data.addProperty("text", text)
        return data
    }

    override fun getAttachments(): JsonArray? {
        if (attachments.isNullOrEmpty()) return null

        val data = JsonArray()
        attachments.forEach { attachment ->
            with(JsonObject()) {
                when (attachment) {
                    is AmityComment.Attachment.IMAGE -> {
                        addProperty("type", attachment.getDataType().apiKey)
                        addProperty("fileId", attachment.getFileId())
                        data.add(this)
                    }
                }
            }
        }
        return data
    }

    class Builder internal constructor() {
        private var referenceType: String = ""
        private var referenceId: String = ""
        private var parentId: String? = null
        private var commentId: String? = null
        private var metadata: JsonObject? = null
        private var mentionees: List<AmityMentioneeTarget>? = null
        private var text: String? = null
        private var attachments: List<AmityComment.Attachment>? = null
        private var dataTypes: MutableSet<String> = mutableSetOf()

        internal fun referenceType(referenceType: String): Builder {
            return apply { this.referenceType = referenceType }
        }

        internal fun referenceId(referenceId: String): Builder {
            return apply { this.referenceId = referenceId }
        }

        internal fun parentId(parentId: String?): Builder {
            return apply { this.parentId = parentId }
        }

        internal fun commentId(commentId: String?): Builder {
            return apply {
                this.commentId = commentId
            }
        }

        /**
         * @param [metadata] to add metadata to the comment
         * @return the same instance of AmityAttachmentCommentCreator.Builder
         */
        fun metadata(metadata: JsonObject): Builder {
            this.metadata = metadata
            return this
        }

        /**
         * @param [userIds] list of userId of users to be mentioned by this comment
         * @return the same instance of AmityAttachmentCommentCreator.Builder
         */
        fun mentionUsers(userIds: List<String>): Builder {
            this.mentionees = listOf(AmityMentioneeTarget(AmityMentionType.USER.apiKey, userIds))
            return this
        }

        /**
         * @param [text] to add text to the comment
         * @return the same instance of AmityAttachmentCommentCreator.Builder
         */
        fun text(text: String): Builder {
            return apply {
                this.text = text
                if (text.isNotEmpty()) {
                    dataTypes.add(AmityComment.DataType.TEXT.apiKey)
                }
            }
        }

        internal fun attachments(vararg commentAttachments: AmityComment.Attachment): Builder {
            return apply {
                this.attachments = commentAttachments.toList()
                attachments?.map { attachment ->
                    when (attachment) {
                        is AmityComment.Attachment.IMAGE -> {
                            dataTypes.add(attachment.getDataType().apiKey)
                        }
                    }
                }
            }
        }

        /**
         * Instantiates AmityAttachmentCommentCreator with built params.
         * @return new instance of AmityAttachmentCommentCreator.
         */
        fun build(): AmityAttachmentCommentCreator {
            return AmityAttachmentCommentCreator(
                referenceType,
                referenceId,
                parentId,
                commentId,
                text,
                attachments,
                dataTypes.toList(),
                metadata,
                mentionees
            )
        }
    }

}