package com.amity.socialcloud.sdk.social.domain.storytarget.composer

import com.amity.socialcloud.sdk.core.domain.ComposerUseCase
import com.amity.socialcloud.sdk.model.social.story.AmityStoryTarget
import com.amity.socialcloud.sdk.social.data.story.StoryRepository
import com.amity.socialcloud.sdk.social.data.storytarget.StoryTargetRepository
import com.amity.socialcloud.sdk.social.domain.community.CommunityGetUseCase
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime

internal class StoryTargetComposerUseCase : ComposerUseCase() {

    fun execute(storyTarget: AmityStoryTarget): AmityStoryTarget {
        addSyncingCount(storyTarget)
        addFailedCount(storyTarget)
        addTarget(storyTarget)
        // need to be added after addSyncingCount and addFailedCount
        addHighestSeen(storyTarget)
        addHasUnseen(storyTarget)
        return storyTarget
    }

    private fun addSyncingCount(storyTarget: AmityStoryTarget) {
        storyTarget.syncingStoriesCount = StoryRepository().getSyncingStoriesCount(
            storyTarget.getTargetType(),
            storyTarget.getTargetId()
        )
    }

    private fun addFailedCount(storyTarget: AmityStoryTarget) {
        storyTarget.failedStoriesCount = StoryRepository().getFailedStoriesCount(
            storyTarget.getTargetType(),
            storyTarget.getTargetId()
        )
    }

    private fun addTarget(storyTarget: AmityStoryTarget) {
        when (storyTarget) {
            is AmityStoryTarget.COMMUNITY -> {
                storyTarget.community = CommunityGetUseCase().execute(storyTarget.getTargetId())
            }

            else -> {}
        }
    }

    private fun addHighestSeen(storyTarget: AmityStoryTarget) {
        storyTarget.highestSeen = getHighestSeen(storyTarget)
    }

    private fun getHighestSeen(storyTarget: AmityStoryTarget): DateTime? {
        return if (storyTarget.localLastStorySeenExpiresAt == null && storyTarget.lastStorySeenExpiresAt == null) {
            null
        } else if (storyTarget.localLastStorySeenExpiresAt == null) {
            storyTarget.lastStorySeenExpiresAt
        } else if (storyTarget.lastStorySeenExpiresAt == null) {
            storyTarget.localLastStorySeenExpiresAt
        } else {
            if (storyTarget.localLastStorySeenExpiresAt.isAfter(storyTarget.lastStorySeenExpiresAt)) {
                storyTarget.localLastStorySeenExpiresAt
            } else {
                storyTarget.lastStorySeenExpiresAt
            }
        }
    }

    private fun addHasUnseen(storyTarget: AmityStoryTarget) {
        val now = DateTime.now()
        val computedHasUnseen: Boolean
        val localSortingDate: DateTime?
        if (storyTarget.syncingStoriesCount + storyTarget.failedStoriesCount > 0) {
            computedHasUnseen = (storyTarget.localLastStoryExpiresAt?.isAfter(now) ?: false)
                    && (storyTarget.localLastStoryExpiresAt?.isAfter(
                getHighestSeen(storyTarget) ?: now
            ) ?: false)
            localSortingDate = storyTarget.localLastStoryExpiresAt
        } else {
            computedHasUnseen = (storyTarget.lastStoryExpiresAt?.isAfter(now) ?: false)
                    && (storyTarget.lastStoryExpiresAt?.isAfter(getHighestSeen(storyTarget) ?: now)
                ?: false)
            localSortingDate = storyTarget.lastStoryExpiresAt
        }
        if (storyTarget.hasUnseen != computedHasUnseen
            || storyTarget.localSortingDate != localSortingDate) {
            storyTarget.hasUnseen = computedHasUnseen
            storyTarget.localSortingDate = localSortingDate
            updateStoryTarget(
                storyTarget = storyTarget,
                hasUnseen = computedHasUnseen,
                localSortingDate = localSortingDate
            )
        }
    }

    private fun updateStoryTarget(
        storyTarget: AmityStoryTarget,
        hasUnseen: Boolean,
        localSortingDate: DateTime?
    ) {
        StoryTargetRepository().updateStoryTargetHasUnseen(
            targetType = storyTarget.getTargetType(),
            targetId = storyTarget.getTargetId(),
            hasUnseen = hasUnseen,
            localSortingDate = localSortingDate
        ).subscribeOn(Schedulers.io())
            .onErrorComplete()
            .subscribe()
    }

}