package com.ekoapp.ekosdk.internal.data.dao

import androidx.room.Dao
import androidx.room.Delete
import androidx.room.Query
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.ekoapp.ekosdk.internal.PostEntity
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import org.joda.time.DateTime

@Dao
abstract class PostDao : EkoObjectDao<PostEntity>() {
	@Delete
	abstract override fun delete(ekoPost: PostEntity)
	@Query("DELETE from post")
	abstract override fun deleteAll()
	@Query("DELETE from post where parentPostId = :postId or postId = :postId")
	abstract fun deleteById(postId: String): Completable
	@Query("UPDATE post set commentCount = commentCount + 1 where postId = :postId")
	abstract fun incrementCommentCount(postId: String)
	@Query("UPDATE post set commentCount = commentCount - 1 where postId = :postId")
	abstract fun decrementCommentCount(postId: String)

	// dummy update post
	@Query("UPDATE post set postId = postId where postId = :postId")
	abstract fun updatePost(postId: String)

	// dummy update by target
	@Query("UPDATE post set postId = postId where targetType = :targetType and targetId = :targetId")
	abstract fun notifyPostByTarget(targetType: String, targetId: String)

	@Query("SELECT *" +
			" from post" +
			" where post.postId IN (:postIds)")
	abstract fun getByIdsNowImpl(postIds: List<String>): List<PostEntity>
	override fun getByIdsNow(postIds: List<String>): List<PostEntity> {
		return getByIdsNowImpl(postIds)
	}

	@Query("SELECT *" +
			" from post" +
			" where post.postId IN (:postIds)" +
			" LIMIT 1")
	abstract fun getByIdNowImpl(postIds: String): PostEntity?
	override fun getByIdNow(postId: String): PostEntity? {
		return getByIdNowImpl(postId)
	}

	@Query("SELECT *" +
			" from post, amity_paging_id" +
			" where post.targetType = :targetType" +
			" and post.targetId = :targetId " +
			" and post.isDeleted =" +
			" (case when :includeDeleted is null then post.isDeleted else :includeDeleted end)" +
			" and post.feedType =" +
			" (case when :feedType is null then post.feedType else :feedType end)" +
			" and post.postDataType in (:postTypes) " +
			" and post.updatedAt > :now" +
			" and post.postId not in " +
			"(" +
			"SELECT amity_paging_id.id" +
			" from amity_paging_id" +
			" where amity_paging_id.hash = (:hash)" +
			" and amity_paging_id.nonce = (:nonce) " +
			")" +
			" order by post.updatedAt  desc" +
			" limit 1")
	abstract fun getLatestChildrenPostImpl(targetId: String,
	                                       targetType: String,
	                                       postTypes: List<String>,
	                                       hash: Int,
	                                       nonce: Int,
	                                       now: DateTime,
	                                       includeDeleted: Boolean?,
	                                       feedType: String?): Flowable<PostEntity>

	@Query("SELECT *" +
			" from post" +
			" where post.targetType = :targetType" +
			" and post.targetId = :targetId " +
			" and post.parentPostId is null" +
			" and post.isDeleted =" +
			" (case when :isDeleted is null then post.isDeleted else :isDeleted end)" +
			" and post.feedType =" +
			" (case when :feedType is null then post.feedType else :feedType end)" +
			" and post.updatedAt > :now" +
			" and post.postId not in " +
			"(" +
			"SELECT amity_paging_id.id" +
			" from amity_paging_id" +
			" where amity_paging_id.hash = (:hash)" +
			" and amity_paging_id.nonce = (:nonce) " +
			")" +
			" order by post.updatedAt  desc" +
			" limit 1")
	abstract fun getLatestParentPostImpl(targetId: String,
	                                     targetType: String,
	                                     hash: Int,
	                                     nonce: Int,
	                                     now: DateTime,
										 isDeleted: Boolean?,
	                                     feedType: String?): Flowable<PostEntity>

	fun getLatestPost(targetId: String,
	                  targetType: String,
	                  includeDeleted: Boolean?,
	                  postTypes: List<String>?,
	                  hash: Int,
	                  nonce: Int,
	                  feedType: AmityFeedType?): Flowable<PostEntity> {
		val feedTypeKey = feedType?.apiKey
		return if (postTypes.isNullOrEmpty()) {
			getLatestParentPostImpl(targetId, targetType, hash, nonce,
					DateTime.now(), includeDeleted, feedTypeKey)
		} else {
			getLatestChildrenPostImpl(targetId, targetType, postTypes,
					hash, nonce, DateTime.now(), includeDeleted, feedTypeKey)
		}
	}

	@Query("UPDATE post SET isDeleted = :isDeleted WHERE postId = :postId or parentPostId = :postId")
	abstract fun updateDeletedImpl(postId: String, isDeleted: Boolean): Completable
	fun softDeleteById(postId: String): Completable {
		return updateDeletedImpl(postId, true)
	}

	@Query("UPDATE post SET feedType = :feedType WHERE postId = :postId ")
	abstract fun updateFeedTypeImpl(postId: String, feedType: String)
	fun updateFeedType(postId: String, feedType: String) {
		updateFeedTypeImpl(postId, feedType)
	}

	@Query("SELECT *" +
			" from post" +
			" where post.postId IN (:postIds)" +
			" LIMIT 1")
	abstract fun getPostImpl(postIds: String): Flowable<PostEntity>
	fun getPost(postId: String): Flowable<PostEntity> {
		return getPostImpl(postId)
	}

	@Query("SELECT *" +
			" from post" +
			" where post.postId IN (:postId)")
	abstract fun observePostImpl(postId: String): Flowable<List<PostEntity>>
	fun observePost(postId: String): Flowable<List<PostEntity>> {
		return observePostImpl(postId)
	}

	fun getByPostIds(ids: List<String>): Flowable<List<PostEntity>> {
		return getByPostIdsImpl(ids)
	}

	@Query("SELECT *" +
			" from post" +
			" where post.postId IN (:postIds)")
	abstract fun getByPostIdsImpl(postIds: List<String>): Flowable<List<PostEntity>>

	@Query("DELETE from post where parentPostId = :postId and postId not in (:childPostIds)")
	abstract fun invalidateChildPosts(postId: String, childPostIds: List<String>): Completable
}