package com.ekoapp.ekosdk.internal.mediator

import com.ekoapp.ekosdk.ReactorObject
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.schedulers.Schedulers

abstract class DynamicQueryStreamMediator<Entity : Any,
        ResponseDto : Any,
        ReactorModel : ReactorObject>(
    nonce: Int,
    var dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
    var reactorDisposable: CompositeDisposable? = null
) : QueryStreamMediator<Entity, ResponseDto>(nonce, dynamicQueryStreamKeyCreator) {
    
    private var isReactorStarted = false
    fun setup(reactorDisposable: CompositeDisposable) {
        this.reactorDisposable = reactorDisposable
        if (!isReactorStarted) {
            startReactor()
            isReactorStarted = true
        }
    }

    abstract fun provideReactorPublisher(): Flowable<ReactorModel>

    private fun startReactor() {
        //dispose previous compositeDisposable if needed
        reactorDisposable?.add(
            //the reactor is responsible to observe
            //object with the correct filter
            //the filter should be similar to what we send
            //to the server (query params)
            provideReactorPublisher()
                //differentiate if item is new by uniqueId and updatedAt
                .distinct { Pair(it.uniqueId(), it.updatedAt()) }
                .doOnNext { reactorObject ->
                    //once reactor object meets all condition
                    //insert to pagingIds directly
                    insertPagingIds(reactorObject.uniqueId())
                }
                //using single thread to make sure that, similar reactors
                //from other callers will insert pagingIds in the same thread
                //avoiding race condition that could occur
                .subscribeOn(Schedulers.single())
                .subscribe()
        )
    }

    override fun onFirstPageLoaded(dto: ResponseDto) : Completable {
        return super.onFirstPageLoaded(dto)
    }

}