package com.ekoapp.ekosdk.internal.data.dao;

import androidx.room.Dao;
import androidx.room.Query;
import androidx.room.Transaction;

import com.ekoapp.ekosdk.internal.data.model.EkoUserFollowStatusEntity;

import org.joda.time.DateTime;

import java.util.List;

import io.reactivex.rxjava3.core.Flowable;

@Dao
public abstract class EkoFollowDao extends EkoObjectDao<EkoUserFollowStatusEntity> implements AmityPagingDao<EkoUserFollowStatusEntity> {

    @Query("UPDATE user_follow set status = :status where targetUserId = :userId")
    public abstract void updateFollowing(String userId, String status);

    @Query("UPDATE user_follow set status = :status where sourceUserId = :userId")
    public abstract void updateFollower(String userId, String status);

    @Query("DELETE from user_follow")
    public abstract void deleteAll();

    @Query("DELETE from user_follow where sourceUserId = :sourceUserId and targetUserId = :targetUserId")
    public abstract void deleteBySourceAndTarget(String sourceUserId, String targetUserId);

    @Query("SELECT * from user_follow where id = :id LIMIT 1")
    abstract EkoUserFollowStatusEntity getByIdNowImpl(String id);

    public EkoUserFollowStatusEntity getByIdNow(String id) {
        return getByIdNowImpl(id);
    }

    @Query("SELECT * from user_follow where id IN (:ids)")
    abstract List<EkoUserFollowStatusEntity> getByIdsNowImpl(List<String> ids);

    public List<EkoUserFollowStatusEntity> getByIdsNow(List<String> ids) {
        return getByIdsNowImpl(ids);
    }

    @Query("SELECT * from user_follow" +
            " where sourceUserId = :sourceUserId" +
            " and targetUserId = :targetUserId" +
            " LIMIT 1")
    abstract Flowable<EkoUserFollowStatusEntity> getAllByIdImpl(String sourceUserId, String targetUserId);

    public Flowable<EkoUserFollowStatusEntity> getUserFollowById(String sourceUserId, String targetUserId) {
        return getAllByIdImpl(sourceUserId, targetUserId);
    }

    public Flowable<EkoUserFollowStatusEntity> getLatestFollowing(String userId, List<String> statuses) {
        return getLatestFollowingImpl(userId, statuses, DateTime.now());
    }

    @Query("SELECT * from user_follow" +
            " where user_follow.sourceUserId = :userId" +
            " and user_follow.status in (:statuses)" +
            " and user_follow.updatedAt > :now" +
            " order by user_follow.updatedAt  desc" +
            " limit 1"
    )
    abstract Flowable<EkoUserFollowStatusEntity> getLatestFollowingImpl(String userId, List<String> statuses, DateTime now);


    public Flowable<EkoUserFollowStatusEntity> getLatestFollower(String userId, List<String> statuses) {
        return getLatestFollowerImpl(userId, statuses, DateTime.now());
    }

    @Query("SELECT * from user_follow" +
            " where user_follow.targetUserId = :userId" +
            " and user_follow.status in (:statuses)" +
            " and user_follow.updatedAt > :now" +
            " order by user_follow.updatedAt  desc" +
            " limit 1"
    )
    abstract Flowable<EkoUserFollowStatusEntity> getLatestFollowerImpl(String userId, List<String> statuses, DateTime now);

    @Transaction //dummy update, for triggering user update tied to follow.
    public void updateUser(String userId) {
        updateSourceUserImpl(userId);
        updateTargetUserImpl(userId);
    }

    @Query("UPDATE user_follow set sourceUserId = :userId where sourceUserId = :userId")
    abstract void updateSourceUserImpl(String userId);

    @Query("UPDATE user_follow set targetUserId = :userId where targetUserId = :userId")
    abstract void updateTargetUserImpl(String userId);

}