package com.amity.socialcloud.sdk.core.engine.analytics

import com.amity.socialcloud.sdk.core.data.analytics.AnalyticsRepository
import com.amity.socialcloud.sdk.model.core.error.AmityException
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.disposables.CompositeDisposable
import io.reactivex.rxjava3.schedulers.Schedulers
import java.util.concurrent.TimeUnit

internal class AnalyticsEventSyncer {

    private val syncDisposable = CompositeDisposable()

    fun start() {
        syncDisposable.clear()
        createHighPrioritySyncTimer()
        createLowPrioritySyncTimer()
    }

    fun stop() {
        syncDisposable.clear()
    }

    fun deleteAllAnalyticsEvents() {
        AnalyticsRepository().deleteAllAnalyticsEvents()
            .subscribeOn(Schedulers.io())
            .doOnError {

            }
            .subscribe()
    }

    private fun syncCapturedEvent(syncPriority: AnalyticsEventSyncPriority) {
        val repository = AnalyticsRepository()
        repository.sendAnalyticsEvents(syncPriority)
            .subscribeOn(Schedulers.io())
            .doOnComplete {

            }
            .doOnError {
                handleSyncError(it)
            }
            .subscribe()
    }

    private fun handleSyncError(error: Throwable) {
        if (error is AmityException) {
            if (error.httpStatusCode == 422) {
                stop()
                deleteAllAnalyticsEvents()
            } else if (error.httpStatusCode in 500..599) {
                stop()
            }
        }
    }

    private fun createHighPrioritySyncTimer() {
        Flowable.interval(
            1000,
            1000 * 10,
            TimeUnit.MILLISECONDS
        )
            .subscribeOn(Schedulers.io())
            .map {
                syncCapturedEvent(AnalyticsEventSyncPriority.HIGH)
            }
            .subscribe()
            .let(syncDisposable::add)
    }

    private fun createLowPrioritySyncTimer() {
        Flowable.interval(
            1000,
            1000 * 60,
            TimeUnit.MILLISECONDS
        )
            .subscribeOn(Schedulers.io())
            .map {
                syncCapturedEvent(AnalyticsEventSyncPriority.LOW)
            }
            .subscribe()
            .let(syncDisposable::add)
    }

}