package com.amity.socialcloud.sdk.chat.data.message

import android.net.Uri
import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.chat.message.query.AmityMessageQuerySortOption
import com.amity.socialcloud.sdk.chat.data.marker.message.MessageMarkerRepository
import com.amity.socialcloud.sdk.chat.data.message.flag.MessageFlagLocalDataStore
import com.amity.socialcloud.sdk.chat.data.message.flag.MessageFlagRemoteDataStore
import com.amity.socialcloud.sdk.chat.data.message.paging.MessageMediator
import com.amity.socialcloud.sdk.chat.domain.marker.message.OptimisticCreateMessageMarkerUseCase
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.core.MarkerSyncEngine
import com.amity.socialcloud.sdk.core.MessagePreviewEvent
import com.amity.socialcloud.sdk.core.MessageSyncEngine
import com.amity.socialcloud.sdk.core.data.file.FileLocalDataStore
import com.amity.socialcloud.sdk.core.session.eventbus.MessagePreviewEventBus
import com.amity.socialcloud.sdk.model.chat.message.AmityMessage
import com.amity.socialcloud.sdk.model.chat.message.AmityMessageAttachment
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.file.AmityFileInfo
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadResult
import com.amity.socialcloud.sdk.model.core.flag.AmityContentFlagReason
import com.amity.socialcloud.sdk.model.core.mention.AmityMentioneeTarget
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.ekoapp.ekosdk.internal.EkoMessageEntity
import com.ekoapp.ekosdk.internal.api.dto.MessageQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.FlagContentRequest
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.repository.comment.CommentLoadResult
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import java.util.UUID

@OptIn(ExperimentalPagingApi::class)
internal class MessageRepository : AmityObjectRepository<EkoMessageEntity, AmityMessage>() {


    override fun fetchAndSave(objectId: String): Completable {
        return MessageRemoteDataStore().getMessage(objectId)
            .flatMapCompletable { dto ->
                fetchMessageMarker(dto)
                    .andThen(MessageQueryPersister().persist(dto))
            }
    }

    override fun queryFromCache(objectId: String): EkoMessageEntity? {
        val message = MessageLocalDataStore().getMessage(objectId)
        // We intentionally don't support unsynced object as a live object
        // such as optimistic object (pre-created)
        if (message != null
            && AmityMessage.State.enumOf(message.syncState) != AmityMessage.State.SYNCED
        ) {
            throw   AmityException.create(
                message = "Observing unsynced object is not supported by Live Object.",
                cause = null,
                error = AmityError.UNSUPPORTED
            )
        }
        return message
    }

    override fun mapper(): ModelMapper<EkoMessageEntity, AmityMessage> {
        return MessageModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<EkoMessageEntity> {
        return MessageLocalDataStore().observeMessage(objectId)
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE * 2
    }

    fun createMessage(
        subChannelId: String,
        parentId: String?,
        fileUri: Uri?,
        fileId: String? = null,
        type: String,
        data: JsonObject,
        tags: AmityTags,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>
    ): Completable {
        return createPreviewMessage(
            subChannelId = subChannelId,
            parentId = parentId,
            type = type,
            fileUri = fileUri,
            data = data,
            tags = tags,
            metadata = metadata,
            mentionees = mentionees
        ).flatMapCompletable { message ->
            Completable.create { emitter ->
                if (fileId != null) {
                    CoreClient.createAttachmentMessage(message, emitter, AmityMessageAttachment.FILE_ID(fileId))
                } else if (fileUri != null) {
                    MessageSyncEngine.grantPersistableUriPermissionIfNeeded(fileUri)
                    CoreClient.createAttachmentMessage(message, emitter, AmityMessageAttachment.URL(fileUri))
                } else {
                    CoreClient.createTextMessage(message, emitter)
                }
            }
        }
    }

    private fun <T : AmityFileInfo> handleUploadResult(
        messageId: String,
        subChannelId: String,
        parentId: String?,
        type: String,
        data: JsonObject,
        tags: AmityTags,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>,
        uploadResult: AmityUploadResult<T>
    ): Completable {
        return when (uploadResult) {
            is AmityUploadResult.COMPLETE<T> -> {
                val fileId = uploadResult.getFile().getFileId()
                syncMessage(
                    messageId = messageId,
                    subChannelId = subChannelId,
                    parentId = parentId,
                    type = type,
                    fileId = fileId,
                    data = data,
                    tags = tags,
                    metadata = metadata,
                    mentionees = mentionees
                )
            }
            is AmityUploadResult.ERROR -> {
                MessageLocalDataStore().updateMessageState(messageId, AmityMessage.State.FAILED)
                    .andThen(Completable.error(uploadResult.getError()))
            }
            is AmityUploadResult.CANCELLED -> {
                MessageLocalDataStore().updateMessageState(messageId, AmityMessage.State.FAILED)
                    .andThen(
                        Completable.error(
                            AmityException.create(
                                message = "${type.capitalize()} upload cancelled",
                                null,
                                AmityError.UNKNOWN
                            )
                        )
                    )
            }
            else -> {
                Completable.complete()
            }
        }
    }

    fun syncMessage(
        messageId: String,
        subChannelId: String,
        parentId: String?,
        type: String,
        data: JsonObject,
        tags: AmityTags,
        fileId: String? = null,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>
    ): Completable {
        if (data.has("fileId")) {
            data.remove("fileId")
        }
        return MessageLocalDataStore().updateMessageState(messageId, AmityMessage.State.SYNCING)
            .andThen(MessageRemoteDataStore().createMessage(
                messageId = messageId,
                subChannelId = subChannelId,
                parentId = parentId,
                type = type,
                fileId = fileId,
                data = data,
                tags = tags,
                metadata = metadata,
                mentionees = mentionees
            )
                .flatMapCompletable { dto ->
                    // Optimistic create message marker to avoid missing message marker
                    // when marker service doesn't finish message marker creation
                    OptimisticCreateMessageMarkerUseCase()
                        .execute(dto)
                        .onErrorComplete()
                        .andThen(MessageQueryPersister().persist(dto))
                        .andThen(
                            MessageLocalDataStore().updateMessageState(
                                messageId,
                                AmityMessage.State.SYNCED
                            )
                        )
                        .andThen(
                            Completable.fromCallable {
                                dto.messages.map { message ->
                                    dto.subChannels.find { it.subChannelId == message.subChannelId }?.let { subChannel ->
                                        MessagePreviewEventBus.publish(MessagePreviewEvent.MessageCreated(message, subChannel))
                                    }
                                }
                            }.onErrorComplete()
                        )
                })
    }

    private fun createPreviewMessage(
        subChannelId: String,
        parentId: String?,
        type: String,
        data: JsonObject?,
        tags: AmityTags,
        fileUri: Uri?,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>
    ): Single<EkoMessageEntity> {
        return MessageLocalDataStore().createMessage(
            subChannelId = subChannelId,
            parentId = parentId,
            type = type,
            data = data,
            tags = tags,
            metadata = metadata,
            fileUri = fileUri,
            mentionees = mentionees
        ).flatMap { message ->
            MessageFlagLocalDataStore().createFlag(message.messageId)
                .andThen(
                    if (!fileUploadingTypes.contains(message.getDataType())) {
                        Completable.complete()
                    } else {
                        val fileType = when (message.getDataType()) {
                            AmityMessage.DataType.IMAGE -> {
                                AmityMessage.DataType.IMAGE.apiKey
                            }
                            AmityMessage.DataType.VIDEO -> {
                                AmityMessage.DataType.VIDEO.apiKey
                            }
                            AmityMessage.DataType.AUDIO -> {
                                AmityMessage.DataType.AUDIO.apiKey
                            }
                            else -> {
                                AmityMessage.DataType.FILE.apiKey
                            }
                        }
                        val path = fileUri?.toString() ?: ""
                        FileLocalDataStore().createLocalFile(message.messageId, fileType, path)
                    }
                )
                .andThen(Single.just(message))
        }
    }

    fun observeLatestMessage(subChannelId: String, isDeleted: Boolean?): Flowable<AmityMessage> {
        return MessageLocalDataStore().observeLatestMessage(subChannelId, isDeleted)
            .map {
                MessageModelMapper().map(it)
            }
    }

    fun updateMessage(
        messageId: String,
        data: JsonObject?,
        tags: AmityTags?,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?
    ): Completable {
        return MessageRemoteDataStore().updateMessage(
            messageId,
            data,
            tags,
            metadata,
            mentionees
        ).flatMapCompletable { dto ->
                MessageQueryPersister().persist(dto)
                    .andThen(
                        Completable.fromCallable {
                            dto.messages.map { message ->
                                dto.subChannels.find { it.subChannelId == message.subChannelId }?.let { subChannel ->
                                    MessagePreviewEventBus.publish(MessagePreviewEvent.MessageUpdated(message, subChannel))
                                }
                            }
                        }.onErrorComplete()
                    )
            }
    }

    fun deleteMessage(messageId: String): Completable {
        return Single.fromCallable {
            var deletingId = ""
            val message = MessageLocalDataStore().getMessage(messageId)
            if (message == null || message.syncState == AmityMessage.State.SYNCED.stateName) {
                deletingId = messageId
            }
            deletingId
        }
            .flatMapCompletable { deletingId ->
                if (deletingId.isNotEmpty()) {
                    MessageRemoteDataStore().deleteMessage(messageId)
                        .flatMapCompletable { dto ->
                            MessageQueryPersister().persist(dto)
                                .andThen(
                                    Completable.fromCallable {
                                        dto.messages.map { message ->
                                            dto.subChannels.find { it.subChannelId == message.subChannelId }?.let { subChannel ->
                                                MessagePreviewEventBus.publish(MessagePreviewEvent.MessageDeleted(message, subChannel))
                                            }
                                        }
                                    }.onErrorComplete()
                            )
                        }
                } else {
                    MessageLocalDataStore().hardDeleteMessage(messageId)
                }
            }
    }

    fun flagMessage(messageId: String, reason: AmityContentFlagReason): Completable {
        val request = when(reason){
            is AmityContentFlagReason.Others -> {
               FlagContentRequest(
                   reason = reason.reason,
                   details = reason.details
               )
            }
            else -> {
                FlagContentRequest(
                    reason = reason.reason,
                )
            }
        }
        return MessageFlagRemoteDataStore().flagMessage(messageId, request)
            .map {
                MessageQueryPersister().persist(it)
            }
            .ignoreElement()
    }

    fun unflagMessage(messageId: String): Completable {
        return MessageFlagRemoteDataStore().unflagMessage(messageId)
            .map {
                MessageQueryPersister().persist(it)
            }
            .ignoreElement()
    }

    fun observeMessages(
        subChannelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        type: String?,
        sortOption: AmityMessageQuerySortOption
    ): Flowable<List<AmityMessage>> {
        return MessageLocalDataStore().observeMessages(
            subChannelId,
            isFilterByParentId,
            parentId,
            includingTags,
            excludingTags,
            isDeleted,
            type,
            sortOption
        ).map {
            it.map {
                MessageModelMapper().map(it)
            }
        }
    }

    fun loadFirstPageMessages(
        subChannelId: String,
        sortOption: AmityMessageQuerySortOption,
        parentId: String?,
        isFilterByParentId: Boolean,
        isDeleted: Boolean?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        type: String?,
        limit: Int
    ): Single<CommentLoadResult> {
        return MessageRemoteDataStore().queryMessages(
            subChannelId = subChannelId,
            filterByParentId = isFilterByParentId,
            parentId = parentId,
            isDeleted = isDeleted,
            includingTags = includingTags,
            excludingTags = excludingTags,
            dataType = type,
            sortBy = sortOption.apiKey,
            limit = limit,
        )
            .flatMap {
                MessageLocalDataStore().hardDeleteAllFromSubChannel(subChannelId)
                    .andThen(fetchMessageMarker(it))
                    .andThen(MessageQueryPersister().persist(it))
                    .andThen(Single.just(it))
            }
            .map { dto ->
                val ids = dto.messages.map { it.messageId }
                val token = dto.token?.next ?: ""
                CommentLoadResult(token, ids)
            }
    }

    fun loadMessages(
        subChannelId: String,
        sortOption: AmityMessageQuerySortOption,
        parentId: String?,
        isFilterByParentId: Boolean,
        isDeleted: Boolean?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        type: String?,
        token: String
    ): Single<CommentLoadResult> {
        return MessageRemoteDataStore().queryMessages(
            subChannelId = subChannelId,
            filterByParentId = isFilterByParentId,
            parentId = parentId,
            isDeleted = isDeleted,
            includingTags = includingTags,
            excludingTags = excludingTags,
            dataType = type,
            sortBy = sortOption.apiKey,
            token = token,
        )
            .flatMap { dto ->
                fetchMessageMarker(dto)
                    .andThen(MessageQueryPersister().persist(dto))
                    .andThen(Single.just(dto))
            }
            .map { dto ->
                val ids = dto.messages.map { it.messageId }
                val token = dto.token?.next ?: ""
                CommentLoadResult(token, ids)
            }
    }

    fun getMessagePagingData(
        subChannelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        aroundMessageId: String?,
        sortOption: AmityMessageQuerySortOption,
        type: AmityMessage.DataType?
    ): Flowable<PagingData<AmityMessage>> {
        val uniqueId = if (aroundMessageId != null) UUID.randomUUID().toString() else null
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = MessageMediator(
                subChannelId = subChannelId,
                isFilterByParentId = isFilterByParentId,
                parentId = parentId,
                includingTags = includingTags,
                excludingTags = excludingTags,
                isDeleted = isDeleted,
                sortOption = sortOption,
                type = type?.apiKey,
                aroundMessageId = aroundMessageId,
                uniqueId = uniqueId
            ),
            pagingSourceFactory = {
                MessageLocalDataStore().getMessagePagingSource(
                    subChannelId = subChannelId,
                    isFilterByParentId = isFilterByParentId,
                    parentId = parentId,
                    includingTags = includingTags,
                    excludingTags = excludingTags,
                    isDeleted = isDeleted,
                    sortOption = sortOption,
                    dataType = type?.apiKey,
                    aroundMessageId = aroundMessageId,
                    uniqueId = uniqueId,
                )
            },
            modelMapper = MessageModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getLatestMessage(
        subChannelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        type: String?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int,
        isUnsyncedOnly: Boolean = false,
    ): Flowable<AmityMessage> {
        return MessageLocalDataStore().getLatestMessage(
            subChannelId = subChannelId,
            isFilterByParentId = isFilterByParentId,
            parentId = parentId,
            includingTags = includingTags,
            excludingTags = excludingTags,
            isDeleted = isDeleted,
            type = type,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce,
            isUnsyncedOnly = isUnsyncedOnly
        )
            .map {
                MessageModelMapper().map(it)
            }
    }

    fun isFlaggedByMe(messageId: String): Single<Boolean> {
        return MessageFlagRemoteDataStore().isFlaggedByMe(messageId)
            .map { it.get("isFlagByMe").asBoolean }
    }

    internal fun updateMarkerHash(messageId: String, hash: Int) {
        MessageLocalDataStore().updateMarkerHash(messageId, hash)
    }

    private fun fetchMessageMarker(dto: MessageQueryDto): Completable {
        return dto.messages
                .filter { MarkerSyncEngine.isMarkerSyncSupport(it.channelType) }
                .map { it.messageId }
                .let { messageIds ->
                    when {
                        messageIds.isEmpty() -> Completable.complete()
                        else -> MessageMarkerRepository().fetchMessageMarker(messageIds)
                    }
                }
                .onErrorComplete()
    }

    fun deleteFailedMessages(): Completable {
        return Completable.fromCallable {
            MessageLocalDataStore().cleanUpFailedMessages()
        }
    }

    fun findCacheAroundMessageId(messageId: String) : Single<List<String>> {
        return MessageLocalDataStore().findCacheAroundMessage(messageId)
    }

    companion object {
        private val fileUploadingTypes = listOf(
            AmityMessage.DataType.IMAGE,
            AmityMessage.DataType.FILE,
            AmityMessage.DataType.AUDIO,
            AmityMessage.DataType.VIDEO
        )
    }

}
