package com.amity.socialcloud.sdk.model.chat.channel

import android.os.Parcelable
import com.amity.socialcloud.sdk.api.chat.member.AmityChannelModeration
import com.amity.socialcloud.sdk.api.chat.member.AmityChannelParticipation
import com.amity.socialcloud.sdk.api.chat.message.latest.AmityChannelLatestMessageQuery
import com.amity.socialcloud.sdk.api.chat.notification.AmityChannelNotification
import com.amity.socialcloud.sdk.api.core.events.AmityTopicSubscription
import com.amity.socialcloud.sdk.chat.domain.marker.channel.CheckMarkerServiceSupportedUseCase
import com.amity.socialcloud.sdk.chat.domain.marker.reader.MarkChannelAsReadUseCase
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.model.chat.message.AmityMessagePreview
import com.amity.socialcloud.sdk.model.core.events.AmityTopic
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.ekoapp.ekosdk.ReactorObject
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import kotlinx.parcelize.Parcelize
import kotlinx.parcelize.TypeParceler
import org.joda.time.DateTime

@Parcelize
@TypeParceler<JsonObject?, JsonObjectParceler>
data class AmityChannel internal constructor(
        private val channelId: String,
        private val defaultSubChannelId: String,
        private val displayName: String,
        private val metadata: JsonObject? = null,
        private val messageCount: Int,
        private val isRateLimited: Boolean,
        private val isMuted: Boolean,
        private val lastActivity: DateTime,
        private val memberCount: Int,
        private val moderatorMemberCount: Int,
        private val tags: AmityTags,
        internal var unreadCount: Int = 0,
        internal var subChannelsUnreadCount: Int = 0,
        internal var hasMentioned: Boolean = false,
        internal var isMentioned: Boolean = false,
        internal var defaultSubChannelUnreadCount: Int = 0,
        internal var hasDefaultSubChannelUnreadMention: Boolean = false,
        private val type: Type,
        internal val avatarFileId: String? = null,
        internal var avatar: AmityImage? = null,
        internal val messagePreviewId: String? = null,
        internal var messagePreview: AmityMessagePreview? = null,
        private val isDeleted: Boolean,
        private val isPublic: Boolean,
        private val createdAt: DateTime,
        private val updatedAt: DateTime,
        internal val path: String
) : Parcelable, ReactorObject {

    fun getChannelId(): String {
        return channelId
    }

    fun getChannelType(): Type {
        return type
    }

    fun getDefaultSubChannelId(): String {
        return defaultSubChannelId
    }

    fun getDisplayName(): String {
        return displayName
    }

    fun getMetadata(): JsonObject? {
        return metadata
    }

    fun getMemberCount(): Int {
        return memberCount
    }

    fun getModeratorMemberCount(): Int {
        return moderatorMemberCount
    }

    /* begin_public_function
	  id: channel.unread_count
	*/
    fun getSubChannelsUnreadCount(): Int {
        return subChannelsUnreadCount
    }
    /* end_public_function */

    fun isMentioned(): Boolean {
        return isMentioned
    }

    fun isRateLimited(): Boolean {
        return isRateLimited
    }

    fun isMuted(): Boolean {
        return isMuted
    }

    fun getLastActivity(): DateTime {
        return lastActivity
    }

    fun getTags(): AmityTags {
        return tags
    }

    fun getAvatar(): AmityImage? {
        return avatar
    }

    fun isDeleted(): Boolean {
        return isDeleted
    }

    fun getCreatedAt(): DateTime {
        return createdAt
    }

    fun getUpdatedAt(): DateTime {
        return updatedAt
    }

    fun membership(): AmityChannelParticipation {
        return AmityChannelParticipation(getChannelId())
    }

    fun moderation(): AmityChannelModeration {
        return AmityChannelModeration(getChannelId())
    }

    fun notification(): AmityChannelNotification {
        return AmityChannelNotification(getChannelId())
    }

    fun latestMessage(): AmityChannelLatestMessageQuery.Builder {
        return AmityChannelLatestMessageQuery.Builder(getChannelId())
    }

    override fun updatedAt(): DateTime {
        return updatedAt
    }

    override fun uniqueId(): String {
        return channelId
    }

    fun subscription(): AmityTopicSubscription {
        return AmityTopicSubscription(AmityTopic.CHANNEL(this))
    }

    fun isUnreadCountSupport(): Boolean {
        return CheckMarkerServiceSupportedUseCase().execute(type)
    }

    fun markAsRead(): Completable {
        return MarkChannelAsReadUseCase().execute(channelId)
    }

    fun getMessagePreviewId(): String? {
        return messagePreviewId
    }

    fun getMessagePreview(): AmityMessagePreview? {
        return messagePreview
    }

    fun isPublic(): Boolean {
        return isPublic
    }

    fun getUnreadCount(): Int {
        return unreadCount
    }

    enum class Type(val apiKey: String) {
        COMMUNITY("community"),
        LIVE("live"),
        BROADCAST("broadcast"),
        CONVERSATION("conversation"),

        @Deprecated(message = "This channel type is deprecated")
        STANDARD("standard"),

        @Deprecated(message = "This channel type is deprecated")
        PRIVATE("private"), ;

        companion object {
            fun enumOf(value: String?): Type {
                return values().find { it.apiKey == value } ?: COMMUNITY
            }
        }
    }

}