package com.amity.socialcloud.sdk.social.data.community.paging

import androidx.paging.ExperimentalPagingApi
import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityMembershipStatusFilter
import com.amity.socialcloud.sdk.social.data.community.CommunityQueryPersister
import com.amity.socialcloud.sdk.social.data.community.CommunityRemoteDataStore
import com.ekoapp.ekosdk.internal.api.dto.CommunityListQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.entity.CommunityEntity
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

@ExperimentalPagingApi
internal class CommunitySemanticSearchMediator(
    private val query: String,
    private val filter: AmityCommunityMembershipStatusFilter?,
    private val tags: AmityTags?,
    private val categoryIds: List<String>?,
    private val includeDiscoverablePrivateCommunity: Boolean
) : DynamicQueryStreamMediator<CommunityEntity, CommunityListQueryDto, AmityCommunity>(
    nonce = AmityNonce.SEARCH_COMMUNITY_LIST,
    dynamicQueryStreamKeyCreator = CommunitySemanticSearchKeyCreator(
        query = query,
        tags = tags,
        filter = filter,
        categoryIds = categoryIds,
        //includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityCommunity> {
        return Flowable.never()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<CommunityListQueryDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<CommunityListQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: CommunityListQueryDto): Completable {
        return CommunityQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: CommunityListQueryDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.paging?.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.communities?.map { it.communityId ?: "" } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<CommunityListQueryDto> {
        return CommunityRemoteDataStore().searchCommunities(
            query = query,
            tags = tags,
            filter = filter,
            includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity,
            categoryIds = categoryIds,
            limit = limit,
            token = token
        ).toRx2()
    }

}