package com.amity.socialcloud.sdk.social.data.community

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.social.community.query.AmityCommunitySortOption
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityFilter
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityMembershipStatusFilter
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityStorySettings
import com.amity.socialcloud.sdk.model.social.community.AmityJoinRequest
import com.amity.socialcloud.sdk.model.social.community.AmityJoinRequestStatus
import com.amity.socialcloud.sdk.model.social.community.AmityJoinResult
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.social.data.community.paging.CommunityJoinRequestQueryMediator
import com.amity.socialcloud.sdk.social.data.community.paging.CommunityQueryMediator
import com.amity.socialcloud.sdk.social.data.community.paging.CommunitySearchMediator
import com.amity.socialcloud.sdk.social.data.community.paging.CommunitySemanticSearchMediator
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.entity.CommunityEntity
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagerCreator
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

@OptIn(ExperimentalPagingApi::class)
internal class CommunityRepository : AmityObjectRepository<CommunityEntity, AmityCommunity>() {

    override fun fetchAndSave(objectId: String): Completable {
        return CommunityRemoteDataStore().getCommunity(objectId)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    override fun queryFromCache(objectId: String): CommunityEntity? {
        return CommunityLocalDataStore().getCommunityById(objectId)
    }

    override fun mapper(): ModelMapper<CommunityEntity, AmityCommunity> {
        return CommunityModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<CommunityEntity> {
        return CommunityLocalDataStore().observeCommunity(objectId)
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    fun getCommunityPagingData(
        keyword: String,
        categoryId: String,
        filter: AmityCommunityFilter,
        sortBy: AmityCommunitySortOption,
        isDeleted: Boolean?,
        includeDiscoverablePrivateCommunity: Boolean,
    ): Flowable<PagingData<AmityCommunity>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = EkoObjectRepository.DEFAULT_PAGE_SIZE,
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = CommunityQueryMediator(
                keyword = keyword,
                categoryId = categoryId,
                filter = filter,
                sortOption = sortBy,
                isDeleted = isDeleted,
                includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity
            ),
            pagingSourceFactory = {
                CommunityLocalDataStore().getCommunityPagingSource(
                    keyword = keyword,
                    categoryId = categoryId,
                    filter = filter,
                    sortOption = sortBy,
                    isDeleted = isDeleted,
                    includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity
                )
            },
            modelMapper = CommunityModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun searchCommunityPagingData(
        keyword: String,
        categoryId: String,
        filter: AmityCommunityFilter,
        sortBy: AmityCommunitySortOption,
        isDeleted: Boolean?,
        includeDiscoverablePrivateCommunity: Boolean,
    ): Flowable<PagingData<AmityCommunity>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = DEFAULT_PAGE_SIZE,
                enablePlaceholders = true
            ),
            queryStreamMediator = CommunitySearchMediator(
                keyword = keyword,
                categoryId = categoryId,
                filter = filter,
                sortOption = sortBy,
                isDeleted = isDeleted,
                includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity
            ),
            pagingSourceFactory = {
                CommunityLocalDataStore().searchCommunityPagingSource(
                    keyword = keyword,
                    categoryId = categoryId,
                    filter = filter,
                    sortOption = sortBy,
                    isDeleted = isDeleted,
                    //includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity
                )
            },
            modelMapper = CommunityModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getRecommendedCommunities(includeDiscoverablePrivateCommunity: Boolean): Flowable<List<AmityCommunity>> {
        return CommunityRemoteDataStore().getRecommendedCommunities(
            QueryOptionsRequestParams(limit = DEFAULT_PAGE_SIZE),
            includeDiscoverablePrivateCommunity
        )
            .flatMap {
                CommunityQueryPersister().persist(it)
                    .andThen(
                        Single.just(it)
                            .map { it.communities?.map { it.communityId!! } ?: emptyList() })
            }
            .flatMapPublisher {
                CommunityLocalDataStore().getCommunities(it)
                    .map { entities ->
                        entities.map {
                            CommunityModelMapper().map(it)
                        }
                    }
            }
    }

    fun getTrendingCommunities(includeDiscoverablePrivateCommunity: Boolean): Flowable<List<AmityCommunity>> {
        return CommunityRemoteDataStore().getTrendingCommunities(
            QueryOptionsRequestParams(limit = DEFAULT_PAGE_SIZE),
            includeDiscoverablePrivateCommunity
        )
            .flatMap {
                CommunityQueryPersister().persist(it)
                    .andThen(
                        Single.just(it)
                            .map { it.communities?.map { it.communityId!! } ?: emptyList() })
            }
            .flatMapPublisher {
                CommunityLocalDataStore().getCommunities(it)
                    .map { entities ->
                        entities.map {
                            CommunityModelMapper().map(it)
                        }
                    }
            }
    }

    fun createCommunity(
        displayName: String,
        description: String?,
        categoryIds: List<String>?,
        isPublic: Boolean?,
        metadata: JsonObject?,
        userIds: List<String>?,
        avatarFileId: String?,
        needApprovalOnPostCreation: Boolean?,
        onlyAdminCanPost: Boolean?,
        storySettings: AmityCommunityStorySettings?,
        isDiscoverable: Boolean? = null,
        requiresJoinApproval: Boolean? = null,
    ): Single<AmityCommunity> {
        return CommunityRemoteDataStore().createCommunity(
            displayName = displayName,
            description = description,
            categoryIds = categoryIds,
            isPublic = isPublic,
            metadata = metadata,
            userIds = userIds,
            avatarFileId = avatarFileId,
            needApprovalOnPostCreation = needApprovalOnPostCreation,
            onlyAdminCanPost = onlyAdminCanPost,
            storySettings = storySettings,
            isDiscoverable = isDiscoverable,
            requiresJoinApproval = requiresJoinApproval
        ).flatMap { dto ->
            CommunityQueryPersister().persist(dto)
                .andThen(
                    dto.communities?.firstOrNull()?.communityId?.let { communityId ->
                        CommunityLocalDataStore().observeCommunity(communityId).firstOrError()
                            .map { CommunityModelMapper().map(it) }
                    } ?: Single.error(
                        AmityException.create(
                            "corrupted payload",
                            null,
                            AmityError.UNKNOWN
                        )
                    )
                )
        }
    }


    fun updateCommunity(
        communityId: String,
        displayName: String?,
        description: String?,
        categoryIds: List<String>?,
        isPublic: Boolean?,
        metadata: JsonObject?,
        avatarFileId: String?,
        needApprovalOnPostCreation: Boolean?,
        onlyAdminCanPost: Boolean?,
        storySettings: AmityCommunityStorySettings?,
        isDiscoverable: Boolean? = null,
        requiresJoinApproval: Boolean? = null,
    ): Single<AmityCommunity> {
        return CommunityRemoteDataStore().updateCommunity(
            communityId = communityId,
            displayName = displayName,
            description = description,
            categoryIds = categoryIds,
            isPublic = isPublic,
            metadata = metadata,
            avatarFileId = avatarFileId,
            needApprovalOnPostCreation = needApprovalOnPostCreation,
            onlyAdminCanPost = onlyAdminCanPost,
            storySettings = storySettings,
            isDiscoverable = isDiscoverable,
            requiresJoinApproval = requiresJoinApproval
        ).flatMap { dto ->
            CommunityQueryPersister().persist(dto)
                .andThen(dto.communities?.firstOrNull()?.communityId?.let { communityId ->
                    CommunityLocalDataStore().observeCommunity(communityId)
                        .firstOrError()
                        .map { CommunityModelMapper().map(it) }
                } ?: Single.error(
                    AmityException.create(
                        "corrupted payload",
                        null,
                        AmityError.UNKNOWN
                    )
                ))
        }
    }

    // To rename to "getCommunity"
    fun getCommunityById(communityId: String): AmityCommunity? {
        val community = CommunityLocalDataStore().getCommunityById(communityId)
        return if (community == null) null else CommunityModelMapper().map(community)
    }


    fun joinCommunity(communityId: String): Completable {
        return CommunityRemoteDataStore().joinCommunity(communityId)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun getJoinRequestPagingData(
        communityId: String,
        status: AmityJoinRequestStatus,
        sortBy: AmityCommunitySortOption? = null,
        targetType: String,
        type: String,
    ): Flowable<PagingData<AmityJoinRequest>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = DEFAULT_PAGE_SIZE,
                enablePlaceholders = true,
            ),
            queryStreamMediator = CommunityJoinRequestQueryMediator(
                communityId = communityId,
                status = status.name,
                sortBy = sortBy,
                targetType = targetType,
                type = type
            ),
            pagingSourceFactory = {
                CommunityJoinRequestLocalDataStore().getCommunityJoinRequestsPagingSource(
                    communityId = communityId,
                    status = status.name,
                    sortBy = sortBy,
                    targetType = targetType,
                    type = type
                )
            },
            modelMapper = CommunityJoinRequestModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getLatestJoinRequest(
        communityIds: List<String>,
    ): Single<AmityJoinRequest> {
        return CommunityRemoteDataStore().getLatestJoinRequest(communityIds).flatMap { dto ->
            CommunityJoinRequestPersister().persist(dto)
                .andThen(
                    dto.joinRequests?.let { joinRequests ->
                        CommunityJoinRequestEntityMapper().map(joinRequests).firstOrNull()?.let {
                            Single.just(
                                CommunityJoinRequestModelMapper().map(it)
                            )
                        } ?: run {
                            Single.error(
                                AmityException.create(
                                    "No join requests found",
                                    null,
                                    AmityError.UNKNOWN
                                )
                            )
                        }
                    } ?: Single.error(
                        AmityException.create(
                            "corrupted payload",
                            null,
                            AmityError.UNKNOWN
                        )
                    )
                )
        }
    }

    fun getMyJoinRequest(
        communityId: String,
    ): Single<AmityJoinRequest> {
        return CommunityRemoteDataStore().getMyJoinRequest(communityId).flatMap { dto ->
            CommunityJoinRequestPersister().persist(dto)
                .andThen(
                    dto.joinRequests?.let { joinRequests ->
                        CommunityJoinRequestEntityMapper().map(joinRequests).firstOrNull()?.let {
                            Single.just(
                                CommunityJoinRequestModelMapper().map(it)
                            )
                        } ?: run {
                            Single.error(
                                AmityException.create(
                                    "No join requests found",
                                    null,
                                    AmityError.UNKNOWN
                                )
                            )
                        }
                    } ?: Single.error(
                        AmityException.create(
                            "corrupted payload",
                            null,
                            AmityError.UNKNOWN
                        )
                    )
                )
        }
    }

    fun joinRequest(communityId: String): Single<AmityJoinResult> {
        return CommunityRemoteDataStore().joinRequest(communityId).flatMap { dto ->
            CommunityJoinRequestPersister().persist(dto)
                .andThen(
                    when {
                        // If joinRequests exists and is not empty, return Pending with the request
                        dto.joinRequests?.isNotEmpty() == true -> {
                            val entity =
                                CommunityJoinRequestEntityMapper().map(dto.joinRequests)
                                    .first()
                            val request = CommunityJoinRequestModelMapper().map(entity)
                            Single.just(AmityJoinResult.Pending(request))
                        }
                        // If joinRequests is null or empty, it means the join was successful
                        else -> {
                            Single.just(AmityJoinResult.Success())
                        }
                    }
                )
        }.onErrorResumeNext { error ->
            Single.error(
                AmityException.create(
                    "Failed to join community: ${error.message}",
                    error,
                    AmityError.UNKNOWN
                )
            )
        }
    }

    fun approveJoinRequest(targetId: String, userId: String): Completable {
        return CommunityRemoteDataStore().approveJoinRequest(targetId, userId)
            .flatMapCompletable { response ->
                if (response.data.isSuccess) {
                    CommunityJoinRequestLocalDataStore().updateJoinRequest(
                        targetId = targetId,
                        userId = userId,
                        status = AmityJoinRequestStatus.APPROVED
                    )
                } else {
                    Completable.error(
                        AmityException.create(
                            "Failed to approve join request",
                            null,
                            AmityError.UNKNOWN
                        )
                    )
                }
            }
            .onErrorResumeNext { error ->
                Completable.error(
                    AmityException.create(
                        "Error approving join request: ${error.message}",
                        error,
                        AmityError.UNKNOWN
                    )
                )
            }
    }

    fun rejectJoinRequest(targetId: String, userId: String): Completable {
        return CommunityRemoteDataStore().rejectJoinRequest(targetId, userId)
            .flatMapCompletable { response ->
                if (response.data.isSuccess) {
                    CommunityJoinRequestLocalDataStore().updateJoinRequest(
                        targetId = targetId,
                        userId = userId,
                        status = AmityJoinRequestStatus.REJECTED
                    )
                } else {
                    Completable.error(
                        AmityException.create(
                            "Failed to reject join request",
                            null,
                            AmityError.UNKNOWN
                        )
                    )
                }
            }
            .onErrorResumeNext { error ->
                Completable.error(
                    AmityException.create(
                        "Error rejecting join request: ${error.message}",
                        error,
                        AmityError.UNKNOWN
                    )
                )
            }
    }

    fun cancelJoinRequest(targetId: String, userId: String): Completable {
        return CommunityRemoteDataStore().cancelJoinRequest(targetId)
            .flatMapCompletable { response ->
                if (response.data.isSuccess) {
                    CommunityJoinRequestLocalDataStore().deleteJoinRequest(targetId, userId)
                } else {
                    Completable.error(
                        AmityException.create(
                            "Failed to cancel join request",
                            null,
                            AmityError.UNKNOWN
                        )
                    )
                }
            }
            .onErrorResumeNext { error ->
                Completable.error(
                    AmityException.create(
                        "Error canceling join request: ${error.message}",
                        error,
                        AmityError.UNKNOWN
                    )
                )
            }
    }

    fun leaveCommunity(communityId: String): Completable {
        return CommunityRemoteDataStore().leaveCommunity(communityId)
            .flatMapCompletable { CommunityQueryPersister().persist(it) }
    }

    fun deleteCommunity(communityId: String): Completable {
        return CommunityRemoteDataStore().deleteCommunity(communityId)
            .flatMapCompletable {
                when (it.isSuccess) {
                    true -> CommunityLocalDataStore().deleteCommunity(communityId)
                    else -> Completable.complete()
                }
            }
    }

    fun getLatestCommunity(
        categoryId: String,
        filter: AmityCommunityFilter,
        isDeleted: Boolean?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int,
    ): Flowable<AmityCommunity> {
        return CommunityLocalDataStore().getLatestCommunity(
            categoryId,
            filter,
            isDeleted,
            dynamicQueryStreamKeyCreator,
            nonce
        ).map {
            CommunityModelMapper().map(it)
        }
    }

    fun searchCommunities(
        query: String,
        filter: AmityCommunityMembershipStatusFilter?,
        tags: AmityTags?,
        categoryIds: List<String>?,
        includeDiscoverablePrivateCommunity: Boolean,
    ): Flowable<PagingData<AmityCommunity>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true,
                initialLoadSize = (getDefaultPageSize() / 2),
                prefetchDistance = 0
            ),
            dynamicQueryStreamMediator = CommunitySemanticSearchMediator(
                query = query,
                tags = tags,
                filter = filter,
                categoryIds = categoryIds,
                includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity
            ),
            pagingSourceFactory = {
                CommunityLocalDataStore().getSearchCommunitiesPagingSource(
                    query = query,
                    filter = filter,
                    tags = tags,
                    categoryIds = categoryIds,
                    //includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity
                )
            },
            modelMapper = CommunityModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getPostCount(targetId: String, feedType: AmityFeedType): Flowable<Int> {
        return CommunityLocalDataStore().getPostCount(targetId, feedType)
    }

    fun fetchCommunityByIds(communityIds: List<String>): Completable {
        return if (communityIds.isEmpty()) {
            Completable.complete()
        } else {
            CommunityRemoteDataStore().getCommunityByIds(communityIds)
                .flatMapCompletable {
                    CommunityQueryPersister().persist(it)
                }
        }
    }

    fun getCommunityByIds(communityIds: List<String>): Flowable<List<AmityCommunity>> {
        return CommunityRemoteDataStore().getCommunityByIds(communityIds)
            .flatMap {
                CommunityQueryPersister().persist(it)
                    .andThen(
                        Single.just(it)
                            .map { it.communities?.map { it.communityId!! } ?: emptyList() })
            }
            .flatMapPublisher {
                CommunityLocalDataStore().getCommunities(it)
                    .map { entities ->
                        entities.map {
                            CommunityModelMapper().map(it)
                        }
                    }
            }
    }

}